﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinProgram.h>
#include <ShdrFile.h>

#if defined( near ) && defined( far )	// windows.h で両方 define される
#undef near
#undef far
#include <nn/g3d/g3d_ResMaterial.h>	// nn:util:math で near, far を使用している
#define near
#define far
#else
#include <nn/g3d/g3d_ResMaterial.h>
#endif

#include <util/UtilBit.h>


namespace nn { namespace g3dTool {

void BinProgram::Build(std::shared_ptr<Context> pCtx, const ShaderProgram* program)
{
    m_pProgram = program;
    pCtx->blocks.push_back(this);
}

void BinProgram::CalculateSize()
{
    if (m_pProgram == nullptr)
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
    }

    // ResShaderProgramData のサイズは ResShaderData に含まれるためにここでのサイズ計算には含めません。
    m_Chunk[SAMPLER_TABLE].size = nw::g3d::tool::util::Align(sizeof(int32_t) * m_pProgram->m_SamplerTable.size() * ShaderStage_StageCount);
    m_Chunk[UNIFORM_BLOCK_TABLE].size = nw::g3d::tool::util::Align(sizeof(int32_t) * m_pProgram->m_UniformBlockTable.size() * ShaderStage_StageCount);
    m_Chunk[SHADER_STORAGE_BLOCK_TABLE].size = nw::g3d::tool::util::Align(sizeof(int32_t) * m_pProgram->m_ShaderStorageBlockTable.size() * ShaderStage_StageCount);

    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, NUM_CHUNK, 8));
}

void BinProgram::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResShaderProgramData& program =
        *GetPtr<nn::g3d::ResShaderProgramData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    nn::gfx::ResShaderFileData* pShaderFileData = const_cast<BinShader*>( m_pShadingModel )->GetResShaderFileDataPtr( pCtx );
        //!< TODO: ここは BinShader の Convert で ResShaderFile へのオフセットが更新される前にコールされると
        //!<		パディング分アドレスがずれるので注意。

    nn::gfx::ResShaderContainerData* pResShaderContainerData =
        reinterpret_cast<nn::gfx::ResShaderContainerData*>( pShaderFileData->fileHeader.GetFirstBlock() );
    nn::gfx::ResShaderVariationData* pResShaderVariationData =
        pResShaderContainerData->pShaderVariationArray.ToPtr( pShaderFileData ) + m_pProgram->m_ResShaderVariationIdx;
    pCtx->LinkPtr( &program.pShader, pResShaderVariationData );

    pCtx->LinkPtr( &program.pShadingModel, m_pShadingModel->GetPtr( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) ) );
    program.flag = 0;
    program.samplerCount = static_cast<u8>(m_pProgram->m_SamplerTable.size());
    program.uniformBlockCount = static_cast<u8>(m_pProgram->m_UniformBlockTable.size());
    program.shaderStorageBlockCount = static_cast<u8>(m_pProgram->m_ShaderStorageBlockTable.size());

    int32_t* samplerDst = GetPtr<int32_t>(pCtx, Context::MemBlockType_Main, m_Chunk[SAMPLER_TABLE].offset);
    pCtx->LinkPtr( &program.pSamplerTable, samplerDst );
    int samplerIndex = 0;
    for (auto sampler = m_pProgram->m_SamplerTable.cbegin(); sampler != m_pProgram->m_SamplerTable.cend(); ++sampler, ++samplerIndex)
    {
        for (int i = 0; i < ShaderStage_StageCount; ++i)
        {
            *samplerDst = sampler->index[i];
            ++samplerDst;
        }
    }

    // uniform
    int32_t* uniformBlockDst = GetPtr<int32_t>(pCtx, Context::MemBlockType_Main, m_Chunk[UNIFORM_BLOCK_TABLE].offset);
    pCtx->LinkPtr( &program.pUniformBlockTable, uniformBlockDst);
    for (auto block = m_pProgram->m_UniformBlockTable.cbegin(); block != m_pProgram->m_UniformBlockTable.cend(); ++block)
    {
        for (int i = 0; i < ShaderStage_StageCount; ++i)
        {
            *uniformBlockDst = block->index[i];
            ++uniformBlockDst;
        }
    }

    // ssbo
    int32_t* shaderStorageBlockDst = GetPtr<int32_t>(pCtx, Context::MemBlockType_Main, m_Chunk[SHADER_STORAGE_BLOCK_TABLE].offset);
    pCtx->LinkPtr( &program.pShaderStorageBlockTable, shaderStorageBlockDst );
    for (auto block = m_pProgram->m_ShaderStorageBlockTable.cbegin(); block != m_pProgram->m_ShaderStorageBlockTable.cend(); ++block)
    {
        for (int i = 0; i < ShaderStage_StageCount; ++i)
        {
            *shaderStorageBlockDst = block->index[i];
            ++shaderStorageBlockDst;
        }
    }

    program.attribActiveFlag = 0;
    int attribFlag = 0x01;
    for (auto attrib = m_pProgram->m_AttribTable.cbegin();
        attrib != m_pProgram->m_AttribTable.cend(); ++attrib)
    {
        if (m_pShadingModel->GetAttribLocation(
            static_cast<int>(std::distance(m_pProgram->m_AttribTable.cbegin(), attrib))) >= 0)
        {
            if (attrib->index[ShaderStage_Vertex] >= 0)
            {
                program.attribActiveFlag |= attribFlag;
            }
            attribFlag <<= 1;
        }
    }
}

void BinProgram::Adjust( std::shared_ptr<Context> pCtx )
{
}

} // namespace g3dTool
} // namespace nn
