﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <shdrdefs.h>
#include <ShdrExtractor.h>
#include <ShaderProgram.h>
#include <BinShader.h>
#include <g3dif/ShaderConfig.h>
#include <g3dif/ShaderDefinition.h>
#include <g3dif/ShaderVariation.h>
#include <ShaderCompilerManager.h>

namespace nn { namespace g3dTool {


class SymbolName
{
public:
    SymbolName()
        : m_Symbol()
    {}

    std::string m_Symbol[ShaderStage_StageCount];
};

class ShaderMacro
{
public:
    ShaderMacro()
        : m_Name()
        , m_Value()
    {}

    std::string m_Name;
    std::string m_Value;
};

class ShaderSource
{
public:
    ShaderSource()
        : m_Path()
        , m_Stream()
    {}

    std::string m_Path;
    std::shared_ptr<void> m_Stream;
};

// 1 つのシェーダバリエーションを構築するために必要なデータ
class ShaderVariationBuilder
{
public:
    ShaderVariationBuilder(int keyLength)
        : m_ChoiceArray()
        , m_IsDefault(false)
    {
        m_Key.reset(malloc(keyLength * sizeof(uint32_t)));
    }

    // Expand してソースを作成するための代替文字列
    std::vector<SymbolName> m_ChoiceArray;
    // key の作成などに用いる choice の option 内での index
    std::vector<int> m_ChoiceIndexArray;

    std::shared_ptr<void> m_Key;
    bool m_IsDefault;
};

// シェーダをコンパイルするために必要なデータ
class ShaderBuilder
{
public:
    std::vector<ShaderVariationBuilder> m_ShaderVariationBuilderArray;
};

class ShadingModelBuilder
{
public:
    ShadingModelBuilder()
        : m_ShaderBuilderArray()
        , m_StreamoutArray()
        , m_ShaderSourceArray()
        , m_MacroArray()
        , m_AttribArray()
        , m_SamplerArray()
        , m_BlockArray()
        , m_OptionArray()
        , m_OptionDefaultIndex()
        , m_Name()
        , m_KeyLength()
        , m_OptionUniformInfoArray()
    {
        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
        {
            m_ShaderPath[stage] = std::string();
        }
    }

    std::vector<ShaderBuilder> m_ShaderBuilderArray;
    std::string m_ShaderPath[ShaderStage_StageCount];
    std::vector<std::string> m_StreamoutArray;

    // ソース
    std::vector<ShaderSource> m_ShaderSourceArray;

    // マクロ
    std::vector<ShaderMacro> m_MacroArray;

    // オプション変数の代替となる uniform に関する情報を保持するクラスです。
    typedef struct OptionUniformInfo
    {
        OptionUniformInfo() :
              m_pOptionUniformName()
            , m_pOptionUniformId( nullptr )
            , m_pElemUniformVar( nullptr )
            , m_pOptionUniformBlockName()
            , matrix()
        {
        };
        const std::string*	m_pOptionUniformBlockName[ShaderStage_StageCount];
        const std::string*	m_pOptionUniformName[ShaderStage_StageCount];				//!< TODO: uniform のシンボルは stage によって異なる可能性があるのか？
        const std::string*	m_pOptionUniformId;					//!< uniform id はステージに依りません。
        const nw::g3d::tool::g3dif::elem_uniform_var*	m_pElemUniformVar;
        nngfxToolShaderCompilerVariableType		m_Type;
        union
        {
            uint8_t				m_VectorComponents;
            struct
            {
                uint8_t column;
                uint8_t	row;
            } matrix;
        };
        std::vector<std::vector<uint32_t>>	m_Uint32Array;
    } OptionUniformInfo;
    std::vector<OptionUniformInfo>	m_OptionUniformInfoArray;		//!< choiceArray と同じサイズです。

    // 各種シンボル
    std::vector<SymbolName> m_AttribArray;
    std::vector<SymbolName> m_SamplerArray;
    std::vector<SymbolName> m_BlockArray;
    std::vector<SymbolName> m_SsboBlockArray;
    std::vector<SymbolName> m_OptionArray;

    // オプションのデフォルト値
    std::vector<int> m_OptionDefaultIndex;

    // シェーディングモデル名
    std::string m_Name;

    // Gfx shader compiler manager
    ShaderCompilerManager	m_ShaderCompilerManager;

    int m_KeyLength;
};

class ShaderArchiveBuilder
{
public:
    ShaderArchiveBuilder()
        : m_ShadingModelBuilderArray()
        , m_ForceIncludePathArray()
    {}

    std::vector<ShadingModelBuilder> m_ShadingModelBuilderArray;
    std::vector<std::string> m_ForceIncludePathArray;
};

class ShdrFile : public BinaryBlock
{
public:
    ShdrFile()
        : BinaryBlock()
        , m_Extractor()
        , m_ExtractedFiles()
        , m_FscArray()
        , m_FsdArray()
        , m_FsvArray()
        , m_BfshaArray()
        , m_DicShader()
        , m_ShaderArray()
        , m_DefinitionFromConfig(nullptr)
        , m_Name()
        , m_ArchiveBuilder()
        , m_IsForceVariation(false)
        , m_DumpFolder()
        , m_IsDumpAll(false)
        , m_IsShaderCodeTypeSource(false)
        , m_IsShaderCodeTypeBinary(false)
        , m_IsShaderCodeTypeIr(false)
        , m_IsBinaryAvailable(false)
        , m_ShaderReflectionCodeType( ShaderCompilerManager::ShaderCodeType_Binary )
        , m_SkipConvert(false)
        , m_GlslVersion()
        , m_Macros()
        , m_IsUsePreprocessorValiation( false )
        , m_ConvertMode( ConvertMode_None )
        , m_ShaderCacheOptionInfo()
        , m_VariationBeginRatio( 0.0f )
        , m_VariationEndRatio( 1.0f )
    {
    }

    ~ShdrFile()
    {
    }


    void Add(std::shared_ptr<const nw::g3d::tool::g3dif::elem_shader_config> fsc) { m_FscArray.push_back(fsc); }
    void Add(std::shared_ptr<const nw::g3d::tool::g3dif::elem_shader_definition> fsd) { m_FsdArray.push_back(fsd); }
    void Add(std::shared_ptr<const nw::g3d::tool::g3dif::elem_shader_variation> fsv) { m_FsvArray.push_back(fsv); }
    void Add(std::shared_ptr<const nn::g3d::ResShaderFile> bfsha) { m_BfshaArray.push_back(bfsha); }

    bool IsShaderCodeTypeSource( void ) const
    {
        return m_IsShaderCodeTypeSource;
    }
    bool IsShaderCodeTypeBinary( void ) const
    {
        return m_IsShaderCodeTypeBinary;
    }
    bool IsShaderCodeTypeIr( void ) const
    {
        return m_IsShaderCodeTypeIr;
    }
    bool IsBinaryAvailable( void ) const
    {
        m_IsBinaryAvailable;
    }
    ShaderCompilerManager::ShaderCodeType GetShaderReflectionCodeType( void ) const
    {
        return m_ShaderReflectionCodeType;
    }
    const std::vector<std::string>& GetGfxToolConverterOptions( void ) const
    {
        return m_GfxToolConvertOptions;
    }
    nw::g3d::tool::g3dif::elem_shader_definition* DefinitionFromConfig() const { return m_DefinitionFromConfig.get(); }

    void SetForceVariation(bool value) { m_IsForceVariation = value; }
    void SetDumpFolder(const std::string& path) { m_DumpFolder = path; }
    void SetDumpAll(bool value) { m_IsDumpAll = value; }

    void SetShaderCodeTypeSource(bool value) { m_IsShaderCodeTypeSource = value; }
    void SetShaderCodeTypeBinary(bool value) { m_IsShaderCodeTypeBinary = value; }
    void SetShaderCodeTypeIr(bool value) { m_IsShaderCodeTypeIr = value; }
    void SetBinaryAvailable(bool value) { m_IsBinaryAvailable = value; }
    void SetShaderReflectionCodeType(ShaderCompilerManager::ShaderCodeType value) { m_ShaderReflectionCodeType = value; }
    void SetSkipConvert( bool value ) { m_SkipConvert = value; }

    void SetGlslVersion(const std::string& glslVersion) { m_GlslVersion = glslVersion; }
    void SetMacros(const std::vector<std::string>& macros) { m_Macros = macros; }

    void AddGfxToolConverterOptions(const std::string& gfxToolConvertOptions)
    {
        m_GfxToolConvertOptions.push_back( gfxToolConvertOptions );
    }

    void SetUsePreprocessorVariation( bool value )
    {
        m_IsUsePreprocessorValiation = value;
    }

    bool IsUsePreprocessorVariation( void )
    {
        return m_IsUsePreprocessorValiation;
    }

    // Config から Definition を作成。
    void Make(bool unifiedAnnotation, bool autoExtract);
    // fsd, fsv からバリエーションコンパイル用の情報を作成します。
    void MakeVariation();
    void ConstructFsdFromBfsha();
    void CalculateShaderFileSize(std::shared_ptr<Context> pCtx);
    void Compile(std::shared_ptr<Context> pCtx);
    void Build(std::shared_ptr<Context> pCtx);
    virtual void CalculateSize();
    virtual void CalculateOffset( std::shared_ptr<Context> pCtx );
    virtual void Convert( std::shared_ptr<Context> pCtx );
    virtual void Adjust( std::shared_ptr<Context> pCtx );

    void SetName(const std::string& name)
    {
        m_Name = name;
    }

    const std::string& GetName() const
    {
        return m_Name;
    }

    void SetShaderCacheAccessFlag( ShaderCompilerManager::ShaderCacheAccessType shaderCacheAccessType )
    {
        m_ShaderCacheOptionInfo.shaderCacheAccessFlag |= shaderCacheAccessType;
    }

    void SetShaderCacheWriteDirectory( const std::string& shaderCacheWriteDirectory )
    {
        m_ShaderCacheOptionInfo.shaderCacheWriteDirectory = shaderCacheWriteDirectory;
    }

    void SetShaderCacheReadDirectory(const std::string& shaderCacheReadDirectory)
    {
        m_ShaderCacheOptionInfo.shaderCacheReadDirectoryArray.clear();
        m_ShaderCacheOptionInfo.shaderCacheReadDirectoryArray.push_back(shaderCacheReadDirectory);
    }

    void AddShaderCacheReadDirectory(const std::string& shaderCacheReadDirectory)
    {
        m_ShaderCacheOptionInfo.shaderCacheReadDirectoryArray.push_back(shaderCacheReadDirectory);
    }

    void SetVariationBeginRatio( float value )
    {
        m_VariationBeginRatio = value;
    }

    void SetVariationEndRatio( float value )
    {
        m_VariationEndRatio = value;
    }

    void SetConvertMode(ConvertMode mode)
    {
        m_ConvertMode = mode;
    }

    ConvertMode GetConvertMode() const
    {
        return m_ConvertMode;
    }

private:
    Extractor m_Extractor;
    FileDB m_ExtractedFiles;

    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_shader_config>> m_FscArray;
    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_shader_definition>> m_FsdArray;
    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_shader_variation>> m_FsvArray;
    std::vector<std::shared_ptr<const nn::g3d::ResShaderFile>> m_BfshaArray;

    BinDictionary m_DicShader;
    std::vector<BinShader> m_ShaderArray;

    std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_definition> m_DefinitionFromConfig;
    std::string m_Name;

    // シェーダバリエーションをコンパイルするための構造
    ShaderArchiveBuilder m_ArchiveBuilder;

    bool m_IsForceVariation;
    std::string m_DumpFolder;
    bool m_IsDumpAll;
    bool m_IsShaderCodeTypeSource;
    bool m_IsShaderCodeTypeBinary;
    bool m_IsShaderCodeTypeIr;
    bool m_IsBinaryAvailable;
    ShaderCompilerManager::ShaderCodeType m_ShaderReflectionCodeType;   // Reflection情報をどのコードタイプから取得するかを示します
    bool m_SkipConvert;

    std::string m_GlslVersion;
    std::vector<std::string> m_Macros;
    std::vector<std::string> m_GfxToolConvertOptions;

    bool m_IsUsePreprocessorValiation;

    ConvertMode m_ConvertMode;

    // シェーダーキャッシュのオプション情報
    ShaderCompilerManager::ShaderCacheOptionInfo m_ShaderCacheOptionInfo;

    float m_VariationBeginRatio;
    float m_VariationEndRatio;

    enum ChunkType
    {
        ChunkType_ShaderFileData,
        ChunkType_ShaderArchiveData,
        ChunkType_ShadingModelDataArray,	// 子の BinShader に渡します。
        ChunkType_Count
    };

    Chunk m_Chunk[ChunkType_Count];
};

} // namespace g3dTool
} // namespace nn
