﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <map>
#include <algorithm>
#include <string>
#include <vector>
#include <nn/gfxTool/gfxTool_Error.h>
#include <nn/gfxTool/gfxTool_Util.h>
#include <nn/gfxTool/gfxTool_ShaderCompilerApi.h>
#include <nn/gfxTool/gfxTool_ShaderCompilerApi-gx.h>
#include <nn/gfxTool/gfxTool_ShaderConverterApi.h>
#include <nn/gfxTool/gfxTool_ShaderUtils.h>
#include <nn/gfx/gfx_ResShader.h>
#include <Windows.h>
#include <ShaderConverterDll.h>

class WriteFileCallback
{
public:
    WriteFileCallback()
    {}

    virtual ~WriteFileCallback()
    {}

    void SetDirectory(const char* directory);

    static bool WriteFileFromDirectory(const void* pFileData, size_t fileDataSize, const char* pFilename, void* pParam);

    static bool WriteFile(const void* pFileData, size_t fileDataSize, const char* pFilename, void* pParam);

private:
    std::string  m_Directory;
};

class ReadFileCallback
{
public:
    ReadFileCallback()
    {
        m_CallbackParam.pFileArray = &m_FileArray;
    }
    virtual ~ReadFileCallback()
    {
    }

    void AddDirectory(const char* directory);

    static bool ReadFileFromDirectories(void** ppOutFileData, size_t* pOutFileDataSize, const char* pFileName, void* pParam);

    static bool ReadFile(void** ppOutFileData,	/* out: シェーダコードのストリームへのポインタ */
        size_t* pOutFileDataSize, /* out: シェーダコードの文字数、終端コードは含まない */
        const char* pFileName,	/* in: gfx の ShaderConverter から絶対パスのファイル名が入力される。*/
        void* pParam				/* in: 3d が gfxTool にセットした ReadFileCallbackParam へのポインタ */
    );

    typedef std::pair<std::string, std::vector<char>> File;
    typedef struct CallbackParam
    {
        std::vector<File>*	pFileArray;
    } CallbackParam;

    CallbackParam* GetCallbackParam(void)
    {
        return &m_CallbackParam;
    };

private:
    CallbackParam		m_CallbackParam;
    std::vector<File>	m_FileArray;
    std::vector< std::string > m_Directories;
};

namespace pugi
{
    class xml_node;
}

class ShaderCompilerManager
{
public:
    enum ShaderType		// g3dif で定義されている enum の順番に従っている。
    {
        ShaderType_Vertex,
        ShaderType_Geometry,
        ShaderType_Pixel,
        ShaderType_Compute,
        ShaderType_Hal,
        ShaderType_Domain,
        ShaderType_Count
    };

    enum ShaderCodeType
    {
        ShaderCodeType_Source,
        ShaderCodeType_Intermediate,
        ShaderCodeType_Binary,
        ShaderCodeType_Count
    };

    enum ShaderCacheAccessType
    {
        ShaderCacheAccessType_UnDefined = 0x00, // 初期状態
        ShaderCacheAccessType_Write = 0x01,     // --shader-cache-write-directory
        ShaderCacheAccessType_Read = 0x02,      // --shader-cache-read-directory
        ShaderCacheAccessType_Count = 0x04
    };

    typedef struct DataStream
    {
        std::shared_ptr<void> data;
        size_t size;
    } DataStream;

    typedef struct ShaderFileInfo
    {
        std::string*	pRelativeFilePathName;
        void*			pFileData;
        uint32_t		strLength;
    } ShaderFileInfo;

    typedef struct FileReadCallbackParam
    {
        std::vector<ShaderFileInfo>	shaderFileInfoArray;
        std::vector<DataStream>	includeFileDiscardedMessageStringDataStreamArray;
    } FileReadCallbackParam;

    typedef struct ShaderCacheOptionInfo
    {
        int shaderCacheAccessFlag;
        std::string shaderCacheWriteDirectory;
        std::vector<std::string> shaderCacheReadDirectoryArray;
    } ShaderCacheOptionInfo;

public:
    ShaderCompilerManager( void ) :
        m_GfxToolHandle( 0 ),
        m_pConvertArg( nullptr ),
        m_pVariationDefinitionArg( nullptr ),
        m_pConvertArgSourceBinary( nullptr ),
        m_IsSourceOnly( false ),
        m_ShaderCacheOptionInfo(),
        m_pMergeResShaderFileArray(),
        m_FileReadCbParam()
    {
        m_pVariationValueArgArray.resize( 0 );
        m_pMergeResShaderFileArray.resize( 0 );
    }

    virtual ~ShaderCompilerManager( void )
    {
    }

    static bool InitializeDll( void );
    static void FinalizeDll( void );

    bool Convert(size_t* pOutShaderFileSize, size_t* pOutShaderFileAlignment);

    nngfxToolResultCode Serialize( void* pBuff, size_t buffSize );

    //!< リロケーションされていない ResShaderFile を返します。
    nn::gfx::ResShaderFile* GetResShaderFile( void )
    {
        return m_pResShaderFile;
    }

    static bool SetLogStream(const nngfxToolSetLogStreamArg* pArg);

    void InitializeGfxToolConverterOptions( const std::vector<std::string>& options );

    void SetShaderSource( const std::string& shaderSource, ShaderType shaderType );

    nngfxToolResultCode InitializeConvertArg( void );
    nngfxToolResultCode FinalizeConvertArg( void );

    void AddVariationValueArg( nngfxToolShaderCompilerVariationValueArg* pVariatonValueArg )
    {
        m_pVariationValueArgArray.push_back( *pVariatonValueArg );
    }

    void SetVariationDefinition( nngfxToolShaderCompilerVariationDefinitionArg* pVariationDefinitionArg )
    {
        m_pVariationDefinitionArg = pVariationDefinitionArg;
    }

    void AddCallbackParam( const ShaderFileInfo& shaderFileInfo )
    {
        m_FileReadCbParam.shaderFileInfoArray.push_back( shaderFileInfo );
    }

    void SetShaderCacheOptionInfo(const ShaderCacheOptionInfo& shaderCacheOptionInfo)
    {
        m_ShaderCacheOptionInfo = shaderCacheOptionInfo;
    }

    void AddMergeResShaderFile(const void* shaderFile)
    {
        m_pMergeResShaderFileArray.push_back(shaderFile);
    }

private:
    void SetReflectionData(
        const nngfxToolShaderCompilerShaderProgramOutput& shaderCompilerShaderProgramOutput,
        int idxVariation, ShaderCodeType idxCodeType );

    void SetShaderInfo(
        const nngfxToolShaderCompilerShaderProgramOutput& shaderCompilerShaderProgramOutput,
        int idxVariation, ShaderCodeType idxCodeType);

private:
    static ShaderConverterDll	m_sShaderConverterDll;
    nn::gfx::ResShaderFile*		m_pResShaderFile;
    nngfxToolHandle				m_GfxToolHandle;
    nngfxToolHandle				m_GfxToolHandleSourceBinary;

    //!< コンバート設定を構築するためのオプション名です。
    std::vector<const char*> m_OptionNamePtrArray;
    std::vector<const char*> m_OptionNamePtrArraySourceBinary;
    std::vector<std::string> m_OptionNames;

    nngfxToolShaderConverterCreateHandleArg		m_CreateHandleArg;
    nngfxToolShaderConverterCreateHandleArg		m_CreateHandleArgSourceBinary;
        //!< CreateArg するためのハンドラです。
        //!< TODO: 複数持つ可能性は？

    //!< この二つは CreateConvertArg で必要。
    nngfxToolShaderConverterConvertArg*	m_pConvertArg;
    nngfxToolShaderConverterConvertArg*	m_pConvertArgSourceBinary;	// Source のみの時に Reflection を取得するための ConvertArg です。
        //!< GfxShaderConverter が中身を埋めます。
        //!< このマネージャクラスがポインタを保持しておきます。
        //!< dll の CreateArg() でアドレスが書き換えられます。

    nngfxToolShaderConverterCreateConvertArgArg	m_ConvertArgArg;
    nngfxToolShaderConverterCreateConvertArgArg	m_ConvertArgArgSourceBinary;
        //!< CreateConvertArg に必要です。

    //!< variation の数だけ存在する
    //!< 外部から登録して貰う。
    std::vector<nngfxToolShaderCompilerVariationValueArg>	m_pVariationValueArgArray;

    nngfxToolShaderCompilerVariationDefinitionArg*	m_pVariationDefinitionArg;

    ShaderCacheOptionInfo m_ShaderCacheOptionInfo;

    std::vector<const void*> m_pMergeResShaderFileArray;

    bool m_IsSourceOnly;

    FileReadCallbackParam	m_FileReadCbParam;
    WriteFileCallback	m_WriteFile;
    ReadFileCallback	m_ReadFile;

public:
    typedef struct AttribInfo
    {
        int index;		//!< reflection データ上での attribute の index です。
        //std::string attribName;	//!< attribute 名です。
        int type;		//!< attribute の型です。
        int arrayCount;
        int stageBit;
        int slot;
    } AttribInfo;
    typedef std::map<std::string, AttribInfo> AttribInfoSet;
    typedef std::vector<AttribInfoSet>		AttribInfoSetArray;	//!< 各 variation の attribute の情報です。
    AttribInfoSetArray	m_AttribInfoSetArray[ ShaderCodeType_Count ];

    const AttribInfoSet& GetAttribInfoSet( int variation, int shaderCodeType ) const
    {
        return m_AttribInfoSetArray[ shaderCodeType ][ variation ];
    }
    const AttribInfo* GetAttribInfo( std::string name, int variation, int shaderCodeType ) const
    {
        if( m_AttribInfoSetArray[ shaderCodeType ][ variation ].find( name ) != m_AttribInfoSetArray[ shaderCodeType ][ variation ].end() )
        {
            return &m_AttribInfoSetArray[ shaderCodeType ][ variation ].find( name )->second;
        }
        else
        {
            return nullptr;
        }
    }

    typedef struct UniformBlockInfo
    {
        int index;
        nngfxToolShaderCompilerShaderSlot shaderSlot;
        int activeVariableCount;
        int stageBit;	//!< どのステージが持つデータかを示すビットです。
        size_t size;
    } UniformBlockInfo;
    typedef UniformBlockInfo SsboBlockInfo;
    typedef std::map<std::string, UniformBlockInfo>	UniformBlockInfoSet;		//!< variation が持つ ub の情報
    typedef std::vector<UniformBlockInfoSet>		UniformBlockInfoSetArray;		//!< size は variation の数
    typedef UniformBlockInfoSet					SsboBlockInfoSet;
    typedef UniformBlockInfoSetArray			SsboBlockInfoSetArray;
    UniformBlockInfoSetArray	m_UniformBlockInfoSetArray[ ShaderCodeType_Count ];
    SsboBlockInfoSetArray		m_SsboBlockInfoSetArray[ShaderCodeType_Count];

    typedef struct UniformInfo
    {
        int blockIndex;
        int type;	//!< uniform の型情報です。
        std::string blockName;	//!< 所属する ub の名前です。
        int offset;
        int arrayCount;
        int stageBit;
    } UniformInfo;
    typedef UniformInfo SsboInfo;
    typedef std::map<std::string, UniformInfo> UniformInfoSet;
    typedef std::vector<UniformInfoSet> UniformInfoSetArray;
    typedef UniformInfoSet		SsboInfoSet;
    typedef UniformInfoSetArray SsboInfoSetArray;
    UniformInfoSetArray		m_UniformInfoSetArray[ ShaderCodeType_Count ];
    SsboInfoSetArray		m_SsboInfoSetArray[ShaderCodeType_Count];

    typedef struct SamplerInfo
    {
        int index;	//!< sampler の index です。
        std::string name;	//!< sampler の名前です。
        int type;	//!< sampler type です。
        nngfxToolShaderCompilerShaderSlot shaderSlot;	//!< sampler のスロットです。
        int stageBit;
    } SamplerInfo;
    typedef std::map<std::string, SamplerInfo> SamplerInfoSet;
    typedef std::vector<SamplerInfoSet> SamplerInfoSetArray;
    SamplerInfoSetArray		m_SamplerInfoSetArray[ ShaderCodeType_Count ];

    typedef struct ScratchMemoryInfo
    {
        uint32_t requiredScratchMemorySizePerWarp;
        uint32_t recommendedScratchMemorySizePerWarp;
    } ScratchMemoryInfo;

    typedef struct ShaderInfo
    {
        const char* pSource;
        void* pStats;
        ScratchMemoryInfo scratchMemoryStats;
    } ShaderInfo;

    typedef struct PipelineInfo
    {
        ShaderInfo vertexShaderInfo;
        ShaderInfo geometryShaderInfo;
        ShaderInfo pixelShaderInfo;
        ShaderInfo computeShaderInfo;
        ShaderInfo hullShaderInfo;
        ShaderInfo domainShaderInfo;
    } PipelineInfo;

    std::vector<PipelineInfo>	m_PipelineInfoArray[ ShaderCodeType_Count ];

    const int GetUniformBlockSize( std::string name, int variation, int shaderCodeType ) const
    {
        if( m_UniformBlockInfoSetArray[ shaderCodeType ][ variation ].find( name ) !=
            m_UniformBlockInfoSetArray[ shaderCodeType ][ variation ].end() )
        {
            return static_cast<int>( m_UniformBlockInfoSetArray[ shaderCodeType ][ variation ].find( name )->second.size );
        }
        return -1;
    }

    const int GetSsboBlockSize( std::string name, int variation, int shaderCodeType ) const
    {
        if( m_SsboBlockInfoSetArray[shaderCodeType][variation].find( name ) != m_SsboBlockInfoSetArray[shaderCodeType][variation].end() )
        {
            return static_cast<int>( m_SsboBlockInfoSetArray[ shaderCodeType ][ variation ].find( name )->second.size );
        }
        return -1;
    }

    const UniformBlockInfoSet& GetUniformBlockInfoSet( int variation, int shaderCodeType ) const
    {
        return m_UniformBlockInfoSetArray[ shaderCodeType ][ variation ];
    }

    const SsboBlockInfoSet& GetSsboBlockInfoSet( int variation, int shaderCodeType ) const
    {
        return m_SsboBlockInfoSetArray[shaderCodeType][variation];
    }

    const UniformInfoSet& GetUniformInfoSet( int variation, int shaderCodeType ) const
    {
        return m_UniformInfoSetArray[ shaderCodeType ][ variation ];
    }

    const SsboInfoSet& GetSsboInfoSet( int variation, int shaderCodeType ) const
    {
        return m_SsboInfoSetArray[shaderCodeType][variation];
    }

    const SamplerInfoSet& GetSamplerInfoSet( int variation, int shaderCodeType ) const
    {
        return m_SamplerInfoSetArray[ shaderCodeType ][ variation ];
    }

    const PipelineInfo& GetPipelineInfo( int variation, int ShaderCodeType ) const
    {
        return m_PipelineInfoArray[ ShaderCodeType ][ variation ];
    }

    void DumpStatistics(std::ostream* pOs, const ShaderInfo* pShader) const;
    void DumpXmlStatistics(pugi::xml_node* pRoot, const ShaderInfo* pShader) const;
};

class VariationValue
{
public:
    VariationValue( void ) :
        m_VariationValueArg(),
        m_VariationValueArray(),
        m_PreprocessorValueArray()
    {
    }

    virtual ~VariationValue( void )
    {
    }

public:
    typedef struct VariationConstantValueInfo
    {
        const void*	pData;
        uint32_t sizeIn32Bit;
    } VariationConstantValueInfo;

    nngfxToolShaderCompilerVariationValueArg* GetVariationValueArg( void )
    {
        return &m_VariationValueArg;
    }

    // nngfxToolShaderCompilerVariationValueArg を生成するためのヘルパー関数
    void CreateVariationValueArg( const std::vector<const std::string*>& preprocessorValueNameArray,
                                    std::vector<VariationConstantValueInfo>& variationConstantValueInfoArray,
                                    ShaderCompilerManager::ShaderType shaderType );

    nngfxToolShaderCompilerVariationValueArg	m_VariationValueArg;
    nngfxToolShaderCompilerVariationValue		m_VariationValueArray[ShaderCompilerManager::ShaderType_Count];
    std::vector<nngfxToolString>				m_PreprocessorValueArray[ ShaderCompilerManager::ShaderType_Count ];
    std::vector<nngfxToolShaderCompilerVariationConstantValue> m_UniformConstantValueArray[ShaderCompilerManager::ShaderType_Count];
};


//!< nngfxToolShaderCompilerVariationDefinitionArg を構築するヘルパークラスです。
class VariationDefinition
{
public:
    VariationDefinition( void ) :
          m_VariationDefinitionArg()
        , m_VariationDefinitionArray()
        , m_PreprocessorDefinitionArray()
        , m_UniformConsstantDefinitionArray()
    {
    }

    virtual ~VariationDefinition( void )
    {
    }

public:
    typedef struct UniformVariationDefinitionInfo
    {
        const std::string*	pUniformBufferName;
        const std::string*	pName;
        nngfxToolShaderCompilerVariableType type;
        uint32_t arrayLength;
        union
        {
            uint8_t vectorComponents;
            struct
            {
                uint8_t column;
                uint8_t row;
            } matrix;
        };
    } UniformVariationDefinitionInfo;

    nngfxToolShaderCompilerVariationDefinitionArg* GetVariationDefinitionArg( void )
    {
        return &m_VariationDefinitionArg;
    }

    void CreateVariationDefinitionArg( const std::vector<const std::string*>& preprocessorDefinitionNameArray, ShaderCompilerManager::ShaderType shaderType );
    void CreateVariationDefinitionArg( std::vector<const std::string*>& preprocessorDefinitionNameArray, std::vector<UniformVariationDefinitionInfo>& uniformConstantDefinitionArray, ShaderCompilerManager::ShaderType shaderType );

    nngfxToolShaderCompilerVariationDefinitionArg	m_VariationDefinitionArg;
    nngfxToolShaderCompilerVariationDefinition		m_VariationDefinitionArray[ ShaderCompilerManager::ShaderType_Count ];	//!< 各ステージの variation definition

    std::vector<nngfxToolShaderCompilerPreprocessorDefinitionDefinition> m_PreprocessorDefinitionArray[ShaderCompilerManager::ShaderType_Count];
    std::vector<nngfxToolShaderCompilerVariationConstantDefinition> m_UniformConsstantDefinitionArray[ShaderCompilerManager::ShaderType_Count];
};
