﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// RXMLElement LoadDocument
// RDataStream RBinDataStream
// RUserData
// RFileBuf

//=============================================================================
// include
//=============================================================================
#include "stdafx.h"
#include "Common.h"
#pragma comment(lib, "shlwapi.lib")

using namespace std;

//=============================================================================
// nwFtx ネームスペースを開始します。
//=============================================================================
namespace nwFtx {

//-----------------------------------------------------------------------------
//! @brief 整数値と文字列のポインタの構造体です。
//-----------------------------------------------------------------------------
struct RIntAndCharPtr
{
  public:
    int m_Value;
    const char* m_String;
};

//-----------------------------------------------------------------------------
//! @brief 2 のべき乗の数なら true を返します。
//-----------------------------------------------------------------------------
bool RIsPower2Number(const int number, const bool includeOne)
{
    if (includeOne && number == 1)
    {
        return true;
    }

    int cur = 1;
    do
    {
        cur *= 2;
        if (number == cur)
        {
            return true;
        }
    } while (cur < number);

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 指定した数以上で最小の 2 のべき乗の数を返します。
//-----------------------------------------------------------------------------
int RGetPower2Number(const int number, const bool includeOne)
{
    int ret = (includeOne) ? 1 : 2;
    while (ret < number)
    {
        ret <<= 1;
    }
    return ret;
}

//-----------------------------------------------------------------------------
//! @brief 指定した文字が Shift JIS の 2 バイト文字の 1 バイト目なら true を返します。
//-----------------------------------------------------------------------------
bool RIsShiftJisFirstByte(const unsigned char val)
{
    return ((0x81 <= val && val <= 0x9f) ||
            (0xe0 <= val && val <= 0xef));
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS の文字列を UTF-8 の文字列に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetUtf8FromShiftJis(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // Shift JIS -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    int wcharSize = MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, NULL, 0);
    if (wcharSize == 0)
    {
        return std::string();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, wcharBuf, wcharSize);

    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> UTF-8
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    int utf8Size = WideCharToMultiByte(CP_UTF8, 0, wcharBuf, -1, NULL, 0, NULL, NULL);
    char* utf8Buf = new char[utf8Size];
    WideCharToMultiByte(CP_UTF8, 0, wcharBuf, -1, utf8Buf, utf8Size, NULL, NULL);
    std::string dst(utf8Buf, strlen(utf8Buf));
    // 通常 strlen(utf8Buf) = utf8Size - 1

    //cerr << "[" << src.c_str() << "] [" << dst.c_str() << "] " << src.size() << ", " << wcharSize << ", " << utf8Size << ", " << dst.size() << endl;
    //for (int i = 0; i < utf8Size; ++i)
    //{
    //	cerr << hex << static_cast<int>(static_cast<u8>(utf8Buf[i])) << dec << endl;
    //}

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;
    delete[] utf8Buf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief UTF-8 の文字列を Shift-JIS の文字列に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetShiftJisFromUtf8(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // UTF-8 -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    int wcharSize = MultiByteToWideChar(CP_UTF8, 0, src.c_str(), -1, NULL, 0);
    if (wcharSize == 0)
    {
        return std::string();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_UTF8, 0, src.c_str(), -1, wcharBuf, wcharSize);

    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> Shift JIS
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    int sjisSize = WideCharToMultiByte(CP_ACP, 0, wcharBuf, -1, NULL, 0, NULL, NULL);
    char* sjisBuf = new char[sjisSize];
    WideCharToMultiByte(CP_ACP, 0, wcharBuf, -1, sjisBuf, sjisSize, NULL, NULL);
    std::string dst(sjisBuf, strlen(sjisBuf));
    // 通常 strlen(sjisBuf) = sjisSize - 1

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;
    delete[] sjisBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief get Unicode (UCS-2) from Shift JIS
//-----------------------------------------------------------------------------
std::wstring RGetUnicodeFromShiftJis(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // Shift JIS -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    const int wcharSize = MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, NULL, 0);
    if (wcharSize == 0)
    {
        return std::wstring();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, wcharBuf, wcharSize);
    std::wstring dst(wcharBuf, wcslen(wcharBuf));

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief get Shift JIS from Unicode (UCS-2)
//-----------------------------------------------------------------------------
std::string RGetShiftJisFromUnicode(const std::wstring& src)
{
    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> Shift JIS
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    const int sjisSize = WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, NULL, 0, NULL, NULL);
    if (sjisSize == 0)
    {
        return std::string();
    }

    char* sjisBuf = new char[sjisSize];
    WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, sjisBuf, sjisSize, NULL, NULL);
    std::string dst(sjisBuf, strlen(sjisBuf));

    //-----------------------------------------------------------------------------
    // free memory
    delete[] sjisBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief ユニコード C 配列を RStringArray に変換します。
//-----------------------------------------------------------------------------
int RGetStringArray(RStringArray& dsts, const wchar_t* srcs[])
{
    dsts.clear();
    int iStr = 0;
    while (srcs[iStr] != NULL)
    {
        dsts.push_back(RGetShiftJisFromUnicode(srcs[iStr]));
        ++iStr;
    }
    return static_cast<int>(dsts.size());
}

//-----------------------------------------------------------------------------
//! @brief 文字列を小文字に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetLowerCaseString(const std::string& src)
{
    std::string dst = src;
    int size = static_cast<int>(dst.size());
    for (int ic = 0; ic < size; ++ic)
    {
        char c = dst[ic];
        if ('A' <= c && c <= 'Z')
        {
            dst[ic] = c - 'A' + 'a';
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 文字列を大文字に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetUpperCaseString(const std::string& src)
{
    std::string dst = src;
    int size = static_cast<int>(dst.size());
    for (int ic = 0; ic < size; ++ic)
    {
        char c = dst[ic];
        if ('a' <= c && c <= 'z')
        {
            dst[ic] = c - 'a' + 'A';
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 大文字／小文字を無視して 2 つの文字列を比較し、同じなら true を返します。
//-----------------------------------------------------------------------------
bool RIsSameStringNoCase(const std::string& str1, const std::string& str2)
{
    return (RGetLowerCaseString(str1) == RGetLowerCaseString(str2));
}

//-----------------------------------------------------------------------------
//! @brief 文字列の前の不要な文字を削除します。
//-----------------------------------------------------------------------------
std::string RTrimLeftString(const std::string& src, const char* targets)
{
    const size_t iLeft = src.find_first_not_of(targets);
    return (iLeft != std::string::npos) ? src.substr(iLeft) : std::string();
}

//-----------------------------------------------------------------------------
//! @brief 文字列の後の不要な文字を削除します。
//-----------------------------------------------------------------------------
std::string RTrimRightString(const std::string& src, const char* targets)
{
    const size_t iRight = src.find_last_not_of(targets);
    return (iRight != std::string::npos) ? src.substr(0, iRight + 1) : std::string();
}

//-----------------------------------------------------------------------------
//! @brief 文字列の前後の不要な文字を削除します。
//-----------------------------------------------------------------------------
std::string RTrimString(const std::string& src, const char* targets)
{
    const size_t iLeft = src.find_first_not_of(targets);
    if (iLeft != std::string::npos)
    {
        const size_t iRight = src.find_last_not_of(targets);
        if (iRight != std::string::npos)
        {
            return src.substr(iLeft, iRight - iLeft + 1);
        }
    }
    return std::string();
}

//-----------------------------------------------------------------------------
//! @brief 文字列の前後の引用符を削除します。
//-----------------------------------------------------------------------------
std::string RDequoteString(const std::string& src, const char quote)
{
    std::string dst(src);
    const size_t iquotL = dst.find(quote);
    if (iquotL != std::string::npos)
    {
        dst = dst.substr(iquotL + 1);
        const size_t iquotR = dst.rfind(quote);
        if (iquotR != std::string::npos)
        {
            dst = dst.substr(0, iquotR);
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 文字列をトークンに分解します。
//-----------------------------------------------------------------------------
int RTokenizeString(RStringArray& tokens, const std::string& input, const char* delims)
{
    tokens.clear();
    size_t iEnd = 0;
    for (;;)
    {
        const size_t iStart = input.find_first_not_of(delims, iEnd);
        if (iStart == std::string::npos)
        {
            break;
        }
        iEnd = input.find_first_of(delims, iStart);
        if (iEnd == std::string::npos)
        {
            tokens.push_back(input.substr(iStart));
            break;
        }
        tokens.push_back(input.substr(iStart, iEnd - iStart));
    }
    return static_cast<int>(tokens.size());
}

//-----------------------------------------------------------------------------
//! @brief int 型の数値を文字列に変換します。
//-----------------------------------------------------------------------------
std::string RGetNumberString(const int num, const char* format)
{
    char buf[64];
    const char* fmt = (format != NULL) ? format : "%d";
    _snprintf_s(buf, sizeof(buf), _TRUNCATE, fmt, num);
    return std::string(buf);
}

//-----------------------------------------------------------------------------
//! @brief u32 型の数値を文字列に変換します。
//-----------------------------------------------------------------------------
std::string RGetNumberString(const u32 num, const char* format)
{
    char buf[64];
    const char* fmt = (format != NULL) ? format : "%u";
    _snprintf_s(buf, sizeof(buf), _TRUNCATE, fmt, num);
    return std::string(buf);
}

//-----------------------------------------------------------------------------
//! @brief ポインタを文字列に変換します。
//-----------------------------------------------------------------------------
std::string RGetPointerString(const void* ptr, const char* format)
{
    char buf[64];
    const char* fmt = (format != NULL) ? format : "%p";
    _snprintf_s(buf, sizeof(buf), _TRUNCATE, fmt, ptr);
    return std::string(buf);
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS 文字列を XML 用にエンコードします。
//-----------------------------------------------------------------------------
static const std::string XML_AMP_STR  = "&amp;";
static const std::string XML_LT_STR   = "&lt;";
static const std::string XML_GT_STR   = "&gt;";
static const std::string XML_QUOT_STR = "&quot;";

std::string REncodeXmlString(const std::string& src)
{
    std::string dst;
    for (size_t ic = 0; ic < src.size(); ++ic)
    {
        const char c = src[ic];
        if (RIsShiftJisFirstByte(static_cast<unsigned char>(c)))
        {
            dst += c;
            if (++ic < src.size())
            {
                dst += src[ic];
            }
        }
        else if (c == '&')
        {
            dst += XML_AMP_STR;
        }
        else if (c == '<')
        {
            dst += XML_LT_STR;
        }
        else if (c == '>')
        {
            dst += XML_GT_STR;
        }
        else if (c == '\"')
        {
            dst += XML_QUOT_STR;
        }
        else
        {
            dst += c;
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief XML 用にエンコードされた Shift-JIS 文字列をデコードします。
//-----------------------------------------------------------------------------
std::string RDecodeXmlString(const std::string& src)
{
    std::string dst;
    for (size_t ic = 0; ic < src.size(); ++ic)
    {
        const char c = src[ic];
        if (c == '&')
        {
            if (src.find(XML_AMP_STR, ic) == ic)
            {
                dst += '&';
                ic += XML_AMP_STR.size() - 1;
            }
            else if (src.find(XML_LT_STR, ic) == ic)
            {
                dst += '<';
                ic += XML_LT_STR.size() - 1;
            }
            else if (src.find(XML_GT_STR, ic) == ic)
            {
                dst += '>';
                ic += XML_GT_STR.size() - 1;
            }
            else if (src.find(XML_QUOT_STR, ic) == ic)
            {
                dst += '\"';
                ic += XML_QUOT_STR.size() - 1;
            }
            else
            {
                dst += c;
            }
        }
        else
        {
            dst += c;
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief Windows 形式のファイルパスを返します。"/" を "\" に変換します。
//!        Shift-JIS の std::string に対応しています。
//!
//! @param[in] path UNIX 形式のファイルパスです。
//!
//! @return Windows 形式のファイルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetWindowsFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    char* buf = new char[size + 1];

    const char* src = path.c_str();
    char* dst = buf;

    while (*src)
    {
        char c = *src++;
        if (c == '/')
        {
            c = '\\';
        }
        *dst++ = c;
    }
    *dst = '\0';

    std::string ret = buf;
    delete[] buf;

    return ret;
}

//-----------------------------------------------------------------------------
//! @brief UNIX 形式のファイルパスを返します。"\" を "/" に変換します。
//!        Shift-JIS の std::string に対応しています。
//!
//! @param[in] path Windows 形式のファイルパスです。
//!
//! @return UNIX 形式のファイルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetUnixFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    char* buf = new char[size + 1];

    const char* src = path.c_str();
    char* dst = buf;

    while (*src)
    {
        char c = *src++;
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            if (*src)
            {
                *dst++ = c;
                c = *src++;
            }
        }
        else if (c == '\\')
        {
            c = '/';
        }
        *dst++ = c;
    }
    *dst = '\0';

    std::string ret = buf;
    delete[] buf;

    return ret;
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS のファイルパス文字列からファイル名を抽出して返します。
//-----------------------------------------------------------------------------
std::string RGetFileNameFromFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
    }
    if (iSlash >= 0)
    {
        return std::string(str + iSlash + 1);
    }
    else
    {
        return path;
    }
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS のファイルパス文字列からフォルダのパス部分を抽出して返します。
//!        返り値の最後に / は付きません。
//-----------------------------------------------------------------------------
std::string RGetFolderFromFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
    }
    if (iSlash >= 0)
    {
        return std::string(str, iSlash); // without last slash
        //return std::string(str, iSlash + 1); // with last slash
    }
    else
    {
        return std::string(".");
    }
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS のファイルパス文字列から拡張子を抽出して返します。
//-----------------------------------------------------------------------------
std::string RGetExtensionFromFilePath(const std::string& path, const bool lower)
{
    const int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    int iDot = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
        else if (c == '.')
        {
            iDot = ic;
        }
    }

    if (iDot >= 0 && (iSlash == -1 || iDot > iSlash))
    {
        const std::string ext = std::string(str + iDot + 1);
        return (lower) ? RGetLowerCaseString(ext) : ext;
    }
    else
    {
        return std::string(); // empty
    }
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS のファイルパス文字列から拡張子を除いた部分を抽出して返します。
//-----------------------------------------------------------------------------
std::string RGetNoExtensionFilePath(const std::string& path)
{
    int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    int iDot = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
        else if (c == '.')
        {
            iDot = ic;
        }
    }
    if (iDot >= 0 && (iSlash == -1 || iDot > iSlash))
    {
        return std::string(str, iDot);
    }
    else
    {
        return path;
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイルパスがフルパスなら true を返します。
//-----------------------------------------------------------------------------
bool RIsFullFilePath(const std::string& path)
{
    if (path.size() >= 2)
    {
        if (path[1] == ':'                       ||
            (path[0] == '\\' && path[1] == '\\') ||
            (path[0] == '/'  && path[1] == '/' ))
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief ファイルパスをフルパスに変換します。
//!
//! @param[in] path 元のファイルパスです。
//! @param[in] isUnix フォルダの区切りを / にするなら true を指定します。
//!
//! @return フルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetFullFilePath(const std::string& path, const bool isUnix)
{
    std::string dst;
    TCHAR fullBuf[MAX_PATH];
    #ifdef UNICODE
    if (::PathSearchAndQualify(RGetUnicodeFromShiftJis(path).c_str(), fullBuf, MAX_PATH))
    {
        dst = RGetShiftJisFromUnicode(fullBuf);
    }
    else
    {
        dst = path;
    }
    #else
    if (::PathSearchAndQualify(path.c_str(), fullBuf, MAX_PATH))
    {
        dst = fullBuf;
    }
    else
    {
        dst = path;
    }
    #endif

    // ドライブ文字を大文字にします。
    if (dst.size() >= 2 && dst[1] == ':')
    {
        const char drive = dst[0];
        if ('a' <= drive && drive <= 'z')
        {
            dst[0] = drive - 'a' + 'A';
        }
    }

    return (isUnix) ? RGetUnixFilePath(dst) : dst;
}

//-----------------------------------------------------------------------------
//! @brief ファイルが存在するなら true を返します。
//-----------------------------------------------------------------------------
bool RFileExists(const std::string& path)
{
    WIN32_FIND_DATA ffd;
    #ifdef UNICODE
    HANDLE  hFindFile = FindFirstFile(RGetUnicodeFromShiftJis(path).c_str(), &ffd);
    #else
    HANDLE  hFindFile = FindFirstFile(path.c_str(), &ffd);
    #endif
    if (hFindFile == INVALID_HANDLE_VALUE)
    {
        return false;
    }
    FindClose(hFindFile);

    if (!(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) // check is file
    {
        return true;
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief フォルダが存在するなら true を返します。
//!        path の最後に '\' または '/' が付いていても付いていなくても構いません。
//-----------------------------------------------------------------------------
bool RFolderExists(const std::string& path)
{
    std::string unixPath = RGetUnixFilePath(path);
    int len = static_cast<int>(unixPath.length());
    if (len == 0)
    {
        return false;
    }

    while (len > 0)
    {
        if (unixPath[len - 1] != '/')
        {
            break;
        }
        --len;
    }
    string noSlashPath = string(unixPath.c_str(), len);
    //if (len != strlen(path))
    //	cerr << "no slash: " << path.c_str() << " -> " << noSlashPath.c_str() << endl;

    WIN32_FIND_DATA ffd;
    #ifdef UNICODE
    HANDLE  hFindFile = FindFirstFile(RGetUnicodeFromShiftJis(noSlashPath).c_str(), &ffd);
    #else
    HANDLE  hFindFile = FindFirstFile(noSlashPath.c_str(), &ffd);
    #endif
    if (hFindFile == INVALID_HANDLE_VALUE)
    {
        return false;
    }
    FindClose(hFindFile);

    if (ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) // check is folder
    {
        return true;
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイル用出力ストリームのフォーマットを初期化します。
//-----------------------------------------------------------------------------
void RInitOutStreamFormat(std::ostream& os)
{
    //-----------------------------------------------------------------------------
    // set format
    os.setf(ios_base::dec, ios_base::basefield);
    os.fill(' ');
    os.setf(ios_base::left, ios_base::adjustfield);
    os.setf(ios_base::uppercase); // 浮動小数の e を大文字に
    os.precision(R_FLOAT_OUT_PRECISION);
    //os.setf(ios_base::showpoint);
    //os.setf(ios_base::fixed, ios_base::floatfield);
    //os.setf(ios_base::scientific, ios_base::floatfield);
    //os.setf(ios_base::fmtflags(0), ios_base::floatfield); // default
}

//-----------------------------------------------------------------------------
//! @brief UTF-8 の BOM を出力します。
//-----------------------------------------------------------------------------
void ROutUtf8Bom(std::ostream& os)
{
    static const unsigned char utf8BOM[] = { 0xef, 0xbb, 0xbf };
    os.write(reinterpret_cast<const char*>(utf8BOM), 3);
}

//-----------------------------------------------------------------------------
//! @brief UTF-16 の BOM を出力します。
//-----------------------------------------------------------------------------
void ROutUtf16Bom(std::ostream& os)
{
    static const unsigned char utf16BOM[] = { 0xff, 0xfe };
    os.write(reinterpret_cast<const char*>(utf16BOM), 2);
}

//-----------------------------------------------------------------------------
//! @brief Shift JIS 文字列を UTF-8 文字列に変換して出力します。
//-----------------------------------------------------------------------------
void ROutUtf8FromShiftJis(std::ostream& os, const std::string& sjisStr)
{
    // UTF-8 の複数バイトコードは 0x80 から 0xfd までの範囲で表現されるので、
    // os はテキストモードでも改行の問題はない
    std::string utf8Str = RGetUtf8FromShiftJis(sjisStr);
    if (utf8Str.size() > 0)
    {
        os.write(utf8Str.c_str(), static_cast<int>(utf8Str.size()));
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定した数のタブ文字を出力します。
//-----------------------------------------------------------------------------
const char* RTab(const int count)
{
    const int TABS_MAX = 16;
    static const char* tabs = "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t"; // TABS_MAX 個

    if (count == 0)
    {
        return "";
    }
    else
    {
        return tabs + (TABS_MAX - count);
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの bool 型用文字列を出力します。
//-----------------------------------------------------------------------------
const char* RBoolStr(const bool value)
{
    return (value) ? "true" : "false";
}

//-----------------------------------------------------------------------------
//! @brief パディングを出力します。
//-----------------------------------------------------------------------------
void ROutPadding(std::ostream& os, const int size, const char value)
{
    if (size > 0)
    {
        std::string padding;
        for (int ibyte = 0; ibyte < size; ++ibyte)
        {
            padding += value;
        }
        os << padding;
    }
}

//-----------------------------------------------------------------------------
//! @brief 環境変数の値を取得します。
//-----------------------------------------------------------------------------
std::string RGetEnvVar(const char* name)
{
    std::string ret;
    char* pValue;
    size_t length;
    if (_dupenv_s(&pValue, &length, name) == 0)
    {
        // 環境変数が見つからない場合は、pValue が NULL に
        // length が 0、戻り値が 0（成功）となります。
        if (pValue != NULL)
        {
            ret = pValue;
            free(pValue);
        }
    }
    return ret;
}

//-----------------------------------------------------------------------------
//! @brief PC のユーザー名を返します。
//-----------------------------------------------------------------------------
std::string RGetUserName(void)
{
    const int userNameSize = 256;
    TCHAR userName[userNameSize];
    DWORD userNameSizeRet = userNameSize;
    GetUserName(userName, &userNameSizeRet);
    #ifdef UNICODE
    return RGetShiftJisFromUnicode(userName);
    #else
    return std::string(userName);
    #endif
}

//-----------------------------------------------------------------------------
//! @brief PC のコンピュータ名を返します。
//-----------------------------------------------------------------------------
std::string RGetComputerName(void)
{
    const int computerNameSize = MAX_COMPUTERNAME_LENGTH + 1;
    TCHAR computerName[computerNameSize];
    DWORD computerNameSizeRet = computerNameSize;
    GetComputerName(computerName, &computerNameSizeRet);
    #ifdef UNICODE
    return RGetShiftJisFromUnicode(computerName);
    #else
    return std::string(computerName);
    #endif
}

//-----------------------------------------------------------------------------
//! @brief SYSTEMTIME から time_t を取得して返します。
//-----------------------------------------------------------------------------
time_t RGetTimeT(const SYSTEMTIME& st)
{
    FILETIME ft;
    ::SystemTimeToFileTime(&st, &ft);

    LONGLONG ll;
    ll = (static_cast<LONGLONG>(ft.dwHighDateTime) << 32) + ft.dwLowDateTime;
    return static_cast<time_t>((ll - 116444736000000000) / 10000000);
}

//-----------------------------------------------------------------------------
//! @brief ISO 8601 に準拠した書式で現在の日時を返します。
//-----------------------------------------------------------------------------
std::string RGetDateTimeString(void)
{
    SYSTEMTIME time;
    GetLocalTime(&time);

    TIME_ZONE_INFORMATION tzi;
    GetTimeZoneInformation(&tzi);
    const int diffMin = -tzi.Bias; // UTC からの差（分単位）にするため符号反転します。

    char dtStr[64];
    sprintf_s(dtStr, "%d-%02d-%02dT%02d:%02d:%02d%c%02d:%02d",
        time.wYear, time.wMonth, time.wDay,
        time.wHour, time.wMinute, time.wSecond,
        (diffMin >= 0) ? '+' : '-', RAbs(diffMin) / 60, diffMin % 60);
    return std::string(dtStr);
}

//-----------------------------------------------------------------------------
//! @brief 指定されたモジュールを含む実行ファイルのフルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetModuleFileName(HMODULE hModule)
{
    TCHAR processPathBuf[MAX_PATH];
    if (GetModuleFileName(hModule, processPathBuf, MAX_PATH))
    {
        #ifdef UNICODE
        return RGetShiftJisFromUnicode(processPathBuf);
        #else
        return std::string(processPathBuf);
        #endif
    }
    else
    {
        return std::string();
    }
}

//-----------------------------------------------------------------------------
//! @brief ユーザーが使用できるアドレス空間の空き容量を返します。
//-----------------------------------------------------------------------------
size_t RGetAvailableVirtualMemorySize(void)
{
    MEMORYSTATUS ms;
    ms.dwLength = sizeof(MEMORYSTATUS);
    GlobalMemoryStatus(&ms);
    return ms.dwAvailVirtual;
    //return ms.dwAvailPhys;
    //return ms.dwAvailPageFile;
}

//-----------------------------------------------------------------------------
//! @brief 引数のトークンの値を取得します。
//!        name=value または name="value" から value を抽出します。
//!
//! @param[in] token 引数のトークンです。
//!
//! @return トークンの値を返します。
//-----------------------------------------------------------------------------
std::string RGetArgTokenValue(const std::string& token)
{
    std::string val;
    const size_t iEq = token.find('=');
    if (iEq != std::string::npos && iEq < token.size() - 1)
    {
        val = token.c_str() + iEq + 1;
        if (val.size() >= 2 && val[0] == '\"' && val[val.size() - 1] == '\"')
        {
            val = RDequoteString(val);
        }
    }
    return val;
}

//-----------------------------------------------------------------------------
//! @brief テキストファイルの 1 行から引数のトークンを取得します。
//-----------------------------------------------------------------------------
static void GetArgTokensFromLine(RStringArray& tokens, const std::string& lineBuf)
{
    bool inQuat = false;
    std::string token;
    const char* str = lineBuf.c_str();
    const size_t lineBytes = lineBuf.size();
    for (size_t ibyte = 0; ibyte < lineBytes; ++ibyte)
    {
        const char c = str[ibyte];
        if (RIsSpace(c) && !inQuat)
        {
            if (!token.empty())
            {
                tokens.push_back(token);
                token = "";
            }
        }
        else
        {
            if (c == '\"')
            {
                inQuat = !inQuat;
            }
            token += c;
        }
    }

    if (!token.empty())
    {
        tokens.push_back(token);
    }
}

//-----------------------------------------------------------------------------
//! @brief テキストファイルから引数のトークンを取得します。
//!
//! @param[out] pMessage エラーメッセージを格納します。NULL なら格納しません。
//! @param[in,out] tokens 取得した引数のトークンを追加する配列です。
//! @param[in] filePath テキストファイルのパスです。
//!
//! @return 処理成功なら true を返します。
//-----------------------------------------------------------------------------
bool RGetArgTokensFromFile(
    std::string* pMessage,
    RStringArray& tokens,
    const std::string& filePath
)
{
    //-----------------------------------------------------------------------------
    // open file
    ifstream ifs(filePath.c_str(), ios_base::in);
    if (!ifs)
    {
        if (pMessage != NULL)
        {
            *pMessage = "Can't open the file: " + filePath;
        }
        return false;
    }

    //-----------------------------------------------------------------------------
    // read
    std::string lineBuf;
    while (!ifs.eof())
    {
        getline(ifs, lineBuf);
        if (lineBuf.empty())
        {
            continue;
        }
        lineBuf = RTrimString(lineBuf);
        if (lineBuf.find('#') == 0)
        {
            continue;
        }
        GetArgTokensFromLine(tokens, lineBuf);
    }

    //-----------------------------------------------------------------------------
    // close file
    ifs.close();

    return true;
}

//-----------------------------------------------------------------------------
//! @brief XML 要素のエラーを表示します。
//-----------------------------------------------------------------------------
void RXMLElement::DisplayError(const std::string& message) const
{
    if (m_ErrorFunc != NULL)
    {
        m_ErrorFunc(message.c_str(), m_pErrorUserData);
    }
    else
    {
        cerr << message << endl;
    }
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の属性を解析してノードに記録します。
//-----------------------------------------------------------------------------
int RXMLElement::LoadAttributes(const char* src, const char end)
{
    int pos = 0;
    for (;;)
    {
        pos += static_cast<int>(strspn(&src[pos], " \t\r\n"));
        if (src[pos] == '>' || src[pos] == end)
        {
            break;
        }
        int namelen = static_cast<int>(strcspn(&src[pos], "="));
        std::string name(&src[pos], namelen);
        pos += namelen;
        std::string value;
        if (src[pos] == '=')
        {
            ++pos;
            if (src[pos] == '\"' || src[pos] == '\'')
            {
                char end2[2] = { src[pos], };
                ++pos;
                int valuelen = static_cast<int>(strcspn(&src[pos], end2));
                value = std::string(&src[pos], valuelen);
                pos += valuelen;
                pos += (src[pos] != '\0');
            }
            else
            {
                char end2[] = " \t\r\n";
                int valuelen = static_cast<int>(strcspn(&src[pos], end2));
                value = std::string(&src[pos], valuelen);
                pos += valuelen;
            }
        }
        attributes[name] = value;
    }
    return pos;
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の代入演算子です。
//-----------------------------------------------------------------------------
RXMLElement& RXMLElement::operator=(const RXMLElement& r)
{
    name = r.name;
    text = r.text;
    attributes.clear();
    for (AttrMap::const_iterator it = r.attributes.begin(); it != r.attributes.end(); ++it)
    {
        attributes[it->first] = it->second;
    }
    nodes.clear();
    nodes.resize(r.nodes.size());
    for (size_t i = 0; i < r.nodes.size(); ++i)
    {
        nodes[i] = r.nodes[i];
    }

    m_ErrorFunc = r.m_ErrorFunc;
    m_pErrorUserData = r.m_pErrorUserData;

    return *this;
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の指定のパスに該当する要素を指定個数分検索します。
//-----------------------------------------------------------------------------
void RXMLElement::FindElements(
    std::vector<const RXMLElement*>& list,
    const std::string& path,
    const bool showError,
    const int maxCount
) const
{
    // パス名を階層に区切ります。
    RStringArray entries;
    std::string subPath(path);
    for (size_t ic = 0; ic < subPath.size(); ++ic)
    {
        if (subPath[ic] == '\\')
        {
            subPath[ic] = '/';
        }
    }
    while (!subPath.empty())
    {
        int span = static_cast<int>(strcspn(subPath.c_str(), "/"));
        entries.push_back(subPath.substr(0, span));
        span += (subPath[span] == '/');
        subPath = subPath.substr(span);
    }
    const int entryCount = static_cast<int>(entries.size());

    // 階層の上から順に探索します（一意性が無いので総当たり）。
    int count = maxCount;
    std::vector<const RXMLElement*> stack(entryCount);
    std::vector<int> index(entryCount);
    int level = 0;
    index[0] = 0;
    stack[0] = this;
    const RXMLElement* node = this;
    int pos = 0;
    for (;;)
    {
        bool found = false;
        if (level >= entryCount)
        {
            // 全階層マッチしたら現在位置を記録します。
            list.push_back(node);
            if (--count <= 0)
            {
                // 指定の個数が見つかればここで終了です。
                break;
            }
        }
        else
        {
            // そうでなければ現在の階層を探索します。
            for (; pos < static_cast<int>(node->nodes.size()); ++pos)
            {
                if (node->nodes[pos].name == entries[level])
                {
                    found = true;
                    break;
                }
            }
        }
        if (found)
        {
            // マッチするノードがあればさらに下層へ。
            // 失敗時のために現在位置を退避します。
            index[level] = pos + 1;
            stack[level] = node;
            ++level;
            node = &node->nodes[pos];
            pos = 0;
        }
        else if (--level >= 0)
        {
            // 該当ノードが無ければスタックを戻ります。
            node = stack[level];
            pos = index[level];
        }
        else // 全要素の探索が終了した。
        {
            break;
        }
    }

    if (showError && list.size() == 0)
    {
        DisplayError("Element is not found: <" + path + ">");
    }
}

//-----------------------------------------------------------------------------
//! @brief XML 文書を解析してツリーを構築します（ASCII テキスト限定）。
//-----------------------------------------------------------------------------
void RXMLElement::LoadDocument(RXMLElement* decl, const std::string& xmlDoc)
{
    RXMLElement dummy;
    decl = (decl != NULL) ? decl : &dummy;
    const char* src = xmlDoc.c_str();
    int len = static_cast<int>(xmlDoc.size());
    if (static_cast<u8>(src[0]) == 0xef)
    {
        // BOM を読み捨てます。
        src += 3;
        len -= 3;
    }
    std::vector<RXMLElement*> stack;
    RXMLElement* node = this;
    stack.push_back(node);
    for (int pos = 0; pos < len; )
    {
        if (src[pos] == '\0')
        {
            break;
        }
        const int span = static_cast<int>(strcspn(&src[pos], "<"));
        node->text += std::string(&src[pos], span);
        pos += span;
        if (src[pos] == '<')
        {
            ++pos;
            if (src[pos] == '?') // 宣言要素。
            {
                ++pos;
                int namelen = static_cast<int>(strcspn(&src[pos], " \t\r\n?"));
                std::string name(&src[pos], namelen);
                pos += namelen;
                pos += decl->LoadAttributes(&src[pos], '?');
                pos += (src[pos] == '?');
            }
            else // 開始・終了タグ。
            {
                bool endtag = (src[pos] == '/');
                pos += endtag;
                int namelen = static_cast<int>(strcspn(&src[pos], " \t\r\n/>"));
                std::string name(&src[pos], namelen);
                pos += namelen;
                if (!endtag) // 開始タグ。
                {
                    node = node->CreateElement();
                    node->name = name;
                    stack.push_back(node);
                    pos += node->LoadAttributes(&src[pos], '/');
                    endtag = (src[pos] == '/');
                    pos += endtag;
                }
                if (endtag) // 終了タグ。
                {
                    for (int i = static_cast<int>(stack.size()) - 1; i >= 0; --i)
                    {
                        if (stack[i]->name == name)
                        {
                            if (i > 0 && stack[i] == node)
                            {
                                node = stack[i - 1];
                            }
                            stack.erase(stack.end() - 1);
                            break;
                        }
                    }
                }
                pos += static_cast<int>(strcspn(&src[pos], ">"));
            }
            pos += (src[pos] == '>');
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief XML 文書を取得します。
//-----------------------------------------------------------------------------
std::string RXMLElement::GetDocument(const int tabCount, const char* lf) const
{
    const int& tc = tabCount;
    std::string	doc;
    const std::string txt = RTrimString(text);
    if (name.empty())
    {
        doc += std::string("<?xml version=\"1.0\" encoding=\"utf-8\"?>") + lf;
        for (size_t iChild = 0; iChild < nodes.size(); ++iChild)
        {
            doc += nodes[iChild].GetDocument(tc, lf);
        }
    }
    else
    {
        doc += std::string(RTab(tc)) + "<" + name;
        for (AttrMap::const_iterator it = attributes.begin(); it != attributes.end(); ++it)
        {
            doc += " " + it->first + "=\"" + it->second + "\"";
        }
        if (txt.empty() && nodes.size() == 0)
        {
            doc += std::string(" />") + lf;
        }
        else
        {
            doc += std::string(">") + lf;
            if (!txt.empty())
            {
                doc += RTab(tc + 1) + txt + lf;
            }
            for (size_t iChild = 0; iChild < nodes.size(); ++iChild)
            {
                doc += nodes[iChild].GetDocument(tc + 1, lf);
            }
            doc += std::string(RTab(tc)) + "</" + name + ">" + lf;
        }
    }
    return doc;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルバージョンを整数（major * 100 + minor）で返します。
//-----------------------------------------------------------------------------
int RGetNw4f3difVerAsInt(const std::string& value)
{
    int ver = 0;
    RStringArray tokens;
    if (RTokenizeString(tokens, value, ".") >= 2)
    {
        ver = atoi(tokens[0].c_str()) * 100 + atoi(tokens[1].c_str());
    }
    return ver;
}

//-----------------------------------------------------------------------------
// @brief データ列のバイト値用のコンストラクタです。
//-----------------------------------------------------------------------------
RDataStream::RDataStream(const u8* array, const int bytes, const int column)
: m_Type(BYTE),
  m_Column(column)
{
    m_ByteValues.resize(bytes);
    for (int ibyte = 0; ibyte < bytes; ++ibyte)
    {
        m_ByteValues[ibyte] = array[ibyte];
    }
}

//-----------------------------------------------------------------------------
// @brief データ列にバイト値を追加します。
//-----------------------------------------------------------------------------
void RDataStream::Append(const u8* array, const int bytes)
{
    const int prevSize = static_cast<int>(m_ByteValues.size());
    m_ByteValues.resize(prevSize + bytes);
    for (int ibyte = 0; ibyte < bytes; ++ibyte)
    {
        m_ByteValues[prevSize + ibyte] = array[ibyte];
    }
}

//-----------------------------------------------------------------------------
//! @brief 整数型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount インデントに必要なタブの数です。
//! @param[in] array データ値配列です。
//! @param[in] column 列数です。
//-----------------------------------------------------------------------------
void ROutDataValues(
    std::ostream& os,
    const int tabCount,
    const RIntArray& array,
    const int column
)
{
    const int valSize = static_cast<int>(array.size());
    if (valSize == 0)
    {
        return;
    }

    const int& tc = tabCount;
    int ic = 0;
    for (int ival = 0; ival < valSize; ++ival)
    {
        if (ic == 0)
        {
            os << RTab(tc);
        }
        os << array[ival];
        if (ival < valSize - 1)
        {
            if (ic == column - 1)
            {
                os << R_ENDL;
            }
            else
            {
                os << " ";
            }
        }
        if (++ic >= column)
        {
            ic = 0;
        }
    }
    os << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 浮動小数点数型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount インデントに必要なタブの数です。
//! @param[in] array データ値配列です。
//! @param[in] column 列数です。
//-----------------------------------------------------------------------------
void ROutDataValues(
    std::ostream& os,
    const int tabCount,
    const RFloatArray& array,
    const int column
)
{
    const int valSize = static_cast<int>(array.size());
    if (valSize == 0)
    {
        return;
    }

    const int& tc = tabCount;
    int ic = 0;
    for (int ival = 0; ival < valSize; ++ival)
    {
        if (ic == 0)
        {
            os << RTab(tc);
        }
        os << array[ival];
        if (ival < valSize - 1)
        {
            if (ic == column - 1)
            {
                os << R_ENDL;
            }
            else
            {
                os << " ";
            }
        }
        if (++ic >= column)
        {
            ic = 0;
        }
    }
    os << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief バイト型のデータ値配列を出力します。
//!
//! @param[in,out] os 出力ストリームです。
//! @param[in] tabCount インデントに必要なタブの数です。
//! @param[in] array データ値配列です。
//! @param[in] column 列数です。
//-----------------------------------------------------------------------------
void ROutDataValues(
    std::ostream& os,
    const int tabCount,
    const RUCharArray& array,
    const int column
)
{
    //-----------------------------------------------------------------------------
    // check size
    const int valSize = static_cast<int>(array.size());
    if (valSize == 0)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // out
    const int& tc = tabCount;
    int ic = 0;
    for (int ival = 0; ival < valSize; ++ival)
    {
        if (ic == 0)
        {
            os << RTab(tc);
        }
        const u32 val = static_cast<u32>(array[ival]);
        const u32 vh = val >> 4;
        const u32 vl = val & 0x0f;
        os << static_cast<char>((vh >= 10) ? 'A' + vh - 10 : '0' + vh)
           << static_cast<char>((vl >= 10) ? 'A' + vl - 10 : '0' + vl);
        if (ic == column - 1 || ival == valSize - 1)
        {
            os << R_ENDL;
        }
        else if (column == 12)
        {
            os << (((ic % 3) == 2) ? "  " : " ");
        }
        else
        {
            os << (((ic & 3) == 3) ? "  " : " ");
        }
        if (++ic >= column)
        {
            ic = 0;
        }
    }
}

//-----------------------------------------------------------------------------
// @brief データ列を出力します。
//-----------------------------------------------------------------------------
void RDataStream::Out(std::ostream& os, const int tabCount, const int index) const
{
    static const char* const typeStrs[] = { "float", "int", "byte", "string", "wstring" };

    const int& tc = tabCount;
    const int count = GetCount();
    if (count > 0)
    {
        const bool isString = (m_Type == STRING || m_Type == WSTRING);
        os << RTab(tc) << "<stream stream_index=\"" << index
           << "\" type=\"" << typeStrs[m_Type]
           << "\" count=\"" << count
           << "\" column=\"" << m_Column
           << "\">";
        if (!isString) os << R_ENDL;

        if (m_Type == FLOAT)
        {
            ROutDataValues(os, tc + 1, m_FloatValues, m_Column);
        }
        else if (m_Type == INT)
        {
            ROutDataValues(os, tc + 1, m_IntValues, m_Column);
        }
        else if (m_Type == BYTE)
        {
            ROutDataValues(os, tc + 1, m_ByteValues, m_Column);
        }
        else // STRING, WSTRING
        {
            os << RGetUtf8FromShiftJis(REncodeXmlString(m_StringValue));
        }

        if (!isString) os << RTab(tc);
        os << "</stream>" << R_ENDL;
    }
}

//-----------------------------------------------------------------------------
// @brief データ列配列を出力します。
//-----------------------------------------------------------------------------
void ROutDataStreams(std::ostream& os, const RDataStreamArray& array)
{
    ROutArrayElement(os, 0, array, "stream_array");
}

//-----------------------------------------------------------------------------
//! @brief データ列配をバイナリ形式で出力します。
//-----------------------------------------------------------------------------
void RDataStream::OutBinary(std::ostream& os) const
{
    static const char* const STREAM_IDENTIFIER = "stream  ";

    //-----------------------------------------------------------------------------
    // ヘッダを出力します。
    os.write(STREAM_IDENTIFIER, strlen(STREAM_IDENTIFIER));

    int binType = 0;
    switch (m_Type)
    {
        case FLOAT   : binType = 0; break;
        case INT     : binType = 1; break;
        case BYTE    : binType = 2; break;
        case STRING  : binType = 3; break;
        case WSTRING : binType = 4; break;
    }
    os.write(reinterpret_cast<const char*>(&binType), sizeof(binType));

    const int count = GetCount();
    os.write(reinterpret_cast<const char*>(&count), sizeof(count));

    os.write(reinterpret_cast<const char*>(&m_Column), sizeof(m_Column));

    const int dataSize = GetSize();
    os.write(reinterpret_cast<const char*>(&dataSize), sizeof(dataSize));

    int padding = 0;
    os.write(reinterpret_cast<const char*>(&padding), sizeof(padding));
    os.write(reinterpret_cast<const char*>(&padding), sizeof(padding));

    //-----------------------------------------------------------------------------
    // データを出力します。
    u8* dataBuf = new u8[dataSize];
    u8* dst = dataBuf;
    if (m_Type == FLOAT)
    {
        for (int iData = 0; iData < count; ++iData, dst += sizeof(float))
        {
            memcpy(dst, &m_FloatValues[iData], sizeof(float));
        }
    }
    else if (m_Type == INT)
    {
        for (int iData = 0; iData < count; ++iData, dst += sizeof(int))
        {
            memcpy(dst, &m_IntValues[iData], sizeof(int));
        }
    }
    else if (m_Type == BYTE)
    {
        for (int iData = 0; iData < count; ++iData)
        {
            dst[iData] = m_ByteValues[iData];
        }
    }
    else if (m_Type == STRING)
    {
        memcpy(dst, m_StringValue.c_str(), m_StringValue.size());
        dst[m_StringValue.size()] = '\0'; // 終端文字を出力します。
    }
    else if (m_Type == WSTRING)
    {
        const std::wstring wstr = RGetUnicodeFromShiftJis(m_StringValue);
        memcpy(dst, wstr.c_str(), wstr.size() * sizeof(wchar_t));
        *(reinterpret_cast<wchar_t*>(dst) + wstr.size()) = 0; // 終端文字を出力します。
    }
    os.write(reinterpret_cast<const char*>(dataBuf), GetSize());
    delete[] dataBuf;

    //-----------------------------------------------------------------------------
    // パディングを出力します。
    ROutPadding(os, GetBinarySize() - (HEADER_SIZE + GetSize()));
}

//-----------------------------------------------------------------------------
//! @brief データ列配列をバイナリ形式で出力します。
//-----------------------------------------------------------------------------
void ROutBinaryDataStreams(
    std::ostream& os,
    const RDataStreamArray& dataStreams,
    const bool outZeroPadding
)
{
    // 参考
    // Tool\G3dTool\g3diflib\build\src\Io\Format\IfBinaryFormatter.cs
    // Tool\G3dTool\g3diflib\build\src\Data\streamBinaryType.cs の Write

    static const char* const STREAM_ARRAY_IDENTIFIER = "g3dstrma";

    //-----------------------------------------------------------------------------
    // データ列がなければ何も出力しません。
    const int streamCount = static_cast<int>(dataStreams.size());
    if (streamCount == 0)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // バイナリ形式の後半部の前のゼロパディングを出力します。
    if (outZeroPadding)
    {
        const int xmlSize = static_cast<int>(os.tellp());
        const int binOfs = RAlignValue(xmlSize + 1, R_BINARY_ALIGNMENT);
        ROutPadding(os, binOfs - xmlSize);
    }

    //-----------------------------------------------------------------------------
    // ストリーム配列チャンクを出力します。
    os.write(STREAM_ARRAY_IDENTIFIER, strlen(STREAM_ARRAY_IDENTIFIER));
    os.write(reinterpret_cast<const char*>(&streamCount), sizeof(streamCount));
    const int headerSize = 8 + 4 + (4 + 4) * streamCount;
    const int streamAddrTop = RAlignValue(headerSize, R_BINARY_ALIGNMENT);

    int streamAddr = streamAddrTop;
    for (int istream = 0; istream < streamCount; ++istream)
    {
        const int streamSize = dataStreams[istream].GetBinarySize();
        os.write(reinterpret_cast<const char*>(&streamAddr), sizeof(streamAddr));
        os.write(reinterpret_cast<const char*>(&streamSize), sizeof(streamSize));
        streamAddr += streamSize;
    }
    ROutPadding(os, streamAddrTop - headerSize);

    //-----------------------------------------------------------------------------
    // 各ストリームチャンクを出力します。
    for (int istream = 0; istream < streamCount; ++istream)
    {
        dataStreams[istream].OutBinary(os);
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの先頭からバイナリデータまでのオフセットを取得します。
//-----------------------------------------------------------------------------
int RGetBinaryOffset(const void* fileBuf, const int fileSize)
{
    int binOfs = fileSize;
    const u8* buf = reinterpret_cast<const u8*>(fileBuf);
    for (int iBuf = 0; iBuf < fileSize; ++iBuf)
    {
        if (buf[iBuf] == 0x00)
        {
            binOfs = RAlignValue(iBuf + 1, R_BINARY_ALIGNMENT);
            break;
        }
    }
    return binOfs;
}

//-----------------------------------------------------------------------------
//! @brief バイナリデータ列へのポインタを取得します（static 関数）。
//-----------------------------------------------------------------------------
const RBinDataStream* RBinDataStream::Get(const void* binTop, const int streamIdx)
{
    const RBinDataStream* streamTop = NULL;
    const u8* src = reinterpret_cast<const u8*>(binTop);
    src += 8; // "g3dstrma"
    const int streamCount = *reinterpret_cast<const int*>(src);
    src += sizeof(int);
    if (streamIdx < streamCount)
    {
        src += (4 + 4) * streamIdx; // address, size
        const int streamAddr = *reinterpret_cast<const int*>(src);
        src = reinterpret_cast<const u8*>(binTop) + streamAddr;
        streamTop = reinterpret_cast<const RBinDataStream*>(src);
    }
    return streamTop;
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータのコンストラクタです。
//-----------------------------------------------------------------------------
RUserData::RUserData(
    const RXMLElement* userDataElem,
    const RXMLElement* streamsElem,
    const void* binTop
)
{
    m_Name = RGetShiftJisFromUtf8(userDataElem->GetAttribute("name"));

    const RXMLElement* valueElem = &(userDataElem->nodes[0]);
    if (valueElem->name == "user_float" || valueElem->name == "user_int")
    {
        //-----------------------------------------------------------------------------
        // 数値型ユーザーデータを解析します。
        m_Type = (valueElem->name == "user_float") ? FLOAT : INT;
        m_NumberValueCount = atoi(valueElem->GetAttribute("count").c_str());
        if (m_NumberValueCount > 0)
        {
            m_NumberValuesText = valueElem->text;
        }
    }
    else if (valueElem->name == "user_string" || valueElem->name == "user_wstring")
    {
        //-----------------------------------------------------------------------------
        // 文字列型ユーザーデータを解析します。
        m_Type = (valueElem->name == "user_string") ? STRING : WSTRING;
        for (size_t iString = 0; iString < valueElem->nodes.size(); ++iString)
        {
            const RXMLElement* stringElem = &(valueElem->nodes[iString]);
            m_StringValues.push_back(
                RDecodeXmlString(RGetShiftJisFromUtf8(stringElem->text)));
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // ストリーム型ユーザーデータを解析します。
        m_Type = STREAM;
        const int iStream = atoi(valueElem->GetAttribute("stream_index").c_str());
        if (iStream == -1) // データなし
        {

        }
        else if (streamsElem != NULL)
        {
            const RXMLElement* streamElem = &(streamsElem->nodes[iStream]);
            const int dataCount = atoi(streamElem->GetAttribute("count").c_str());
            m_StreamValues.resize(dataCount);
            int iData = 0;
            std::istringstream iss(streamElem->text.c_str());
            while (iss && iData < dataCount)
            {
                std::string value;
                iss >> value;
                if (!value.empty())
                {
                    m_StreamValues[iData++] = static_cast<u8>(strtol(value.c_str(), NULL, 16));
                }
            }
        }
        else
        {
            const RBinDataStream* binStream = RBinDataStream::Get(binTop, iStream);
            m_StreamValues.resize(binStream->m_Count);
            const unsigned char* pSrc = reinterpret_cast<const unsigned char*>(binStream->m_Data);
            for (int iData = 0; iData < binStream->m_Count; ++iData)
            {
                m_StreamValues[iData] = *pSrc++;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータを出力します。
//-----------------------------------------------------------------------------
void RUserData::Out(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const int tabCount,
    const int index
) const
{
    static const char* const valueElemNames[] =
    {
        "user_float",
        "user_int",
        "user_string",
        "user_wstring",
        "user_stream",
    };

    const int& tc = tabCount;
    os << RTab(tc) << "<user_data index=\"" << index
       << "\" name=\"" << RGetUtf8FromShiftJis(m_Name)
       << "\">" << R_ENDL;

    const char* valueElemName = valueElemNames[m_Type];
    os << RTab(tc + 1) << "<" << valueElemName;
    if (m_Type != STREAM)
    {
        int dataCount = 0;
        switch (m_Type)
        {
            case FLOAT: dataCount = m_NumberValueCount; break;
            case INT  : dataCount = m_NumberValueCount; break;
            default   : dataCount = static_cast<int>(m_StringValues.size()); break;
        }
        os << " count=\"" << dataCount << "\"";
        if (dataCount == 0)
        {
            os << " />" << R_ENDL;
        }
        else
        {
            os << ">";
            if (m_Type == FLOAT || m_Type == INT)
            {
                // 現在は中間ファイルの文字列をそのまま取得して出力します。
                // 値の配列から出力する場合は次のコードと同様に処理します。
                //   Tool\G3dTool\nw4f_3dif\src\Format\G3dDataFormatter.cs
                //   FormatUserIntData、FormatUserFloatData
                // (4 タブの間隔を空けつつ一行あたり 4 つの値)
                // (符号が＋の場合は数値の前にスペースを挿入)
                os << m_NumberValuesText;
            }
            else // STRING, WSTRING
            {
                os << R_ENDL;
                const char* strElemName = (m_Type == STRING) ? "string" : "wstring";
                for (int iData = 0; iData < dataCount; ++iData)
                {
                    os << RTab(tc + 2) << "<" << strElemName << ">"
                       << RGetUtf8FromShiftJis(REncodeXmlString(m_StringValues[iData]))
                       << "</" << strElemName << ">" << R_ENDL;
                }
                os << RTab(tc + 1);
            }
            os << "</" << valueElemName << ">" << R_ENDL;
        }
    }
    else // STREAM
    {
        const int dataCount = static_cast<int>(m_StreamValues.size());
        const int iStream = (dataCount > 0) ? static_cast<int>(dataStreams.size()) : -1;
        os << " size=\"" << dataCount
           << "\" stream_index=\"" << iStream
           << "\" />" << R_ENDL;
        if (dataCount > 0)
        {
            dataStreams.push_back(RDataStream(&m_StreamValues[0], dataCount, 16));
        }
    }

    os << RTab(tc) << "</user_data>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief C 言語のソース形式でバイト型のデータ値配列を出力します。
//-----------------------------------------------------------------------------
void ROutCSourceData(
    std::ostream& os,
    const int tabCount,
    const RUCharArray& array,
    const int column
)
{
    //-----------------------------------------------------------------------------
    // check size
    const int valSize = static_cast<int>(array.size());
    if (valSize == 0)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // set os style
    const char fillBak = os.fill('0');
    const long flagBak = os.setf(ios_base::hex, ios_base::basefield);
    os.setf(ios_base::uppercase);
    os.setf(ios_base::right, ios_base::adjustfield);

    //-----------------------------------------------------------------------------
    // out
    const int& tc = tabCount;
    int ic = 0;
    for (int ival = 0; ival < valSize; ++ival)
    {
        if (ic == 0)
        {
            os << RTab(tc);
        }
        os << "0x" << setw(2) << static_cast<unsigned int>(array[ival]) << ",";
        if (ival < valSize - 1)
        {
            if (ic == column - 1)
            {
                os << R_ENDL;
            }
            else
            {
                //os << (((ic & 3) == 3) ? "  " : " ");
            }
        }
        if (++ic >= column)
        {
            ic = 0;
        }
    }
    os << R_ENDL;

    //-----------------------------------------------------------------------------
    // restore os style
    os.fill(fillBak);
    os.flags(flagBak);
}

//-----------------------------------------------------------------------------
//! @brief ファイルバッファにファイルをリードします。
//-----------------------------------------------------------------------------
bool RFileBuf::Read(const std::string& path)
{
    RFreeAndClearArray(m_FileBuf);

    m_FilePath = path;
    m_FileSize = 0;

    std::ifstream ifs(path.c_str(), ios_base::in | ios_base::binary);
    if (ifs)
    {
        //-----------------------------------------------------------------------------
        // read file
        ifs.seekg(0L, ios_base::end);
        m_FileSize = static_cast<int>(ifs.tellg());
        ifs.seekg(0L, ios_base::beg);
        if (m_FileSize > 0)
        {
            m_FileBuf = new u8[m_FileSize];
            ifs.read(reinterpret_cast<char*>(m_FileBuf), m_FileSize);
        }
        else
        {
            m_FileBuf = new u8[1];
        }

        //-----------------------------------------------------------------------------
        // close file
        ifs.close();
        return true;
    }
    else
    {
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列からサーフェスの次元を取得します。
//-----------------------------------------------------------------------------
GX2SurfaceDim RGetSurfaceDimensionFromString(bool* pIsArray, const std::string& str)
{
    if (pIsArray != NULL)
    {
        *pIsArray = (str.find("_array") != std::string::npos);
    }

    if (str == "Item1d"           ) return GX2_SURFACE_DIM_1D;				// 3DEditorで変更
    if (str == "Item2d"           ) return GX2_SURFACE_DIM_2D;				// 3DEditorで変更
    if (str == "Item3d"           ) return GX2_SURFACE_DIM_3D;				// 3DEditorで変更
    if (str == "cube"         ) return GX2_SURFACE_DIM_CUBE;
    if (str == "cube_array"   ) return GX2_SURFACE_DIM_CUBE;
    if (str == "Item1d_array"     ) return GX2_SURFACE_DIM_1D_ARRAY;		// 3DEditorで変更
    if (str == "Item2d_array"     ) return GX2_SURFACE_DIM_2D_ARRAY;		// 3DEditorで変更
    if (str == "Item2d_msaa"      ) return GX2_SURFACE_DIM_2D_MSAA;			// 3DEditorで変更
    if (str == "Item2d_msaa_array") return GX2_SURFACE_DIM_2D_MSAA_ARRAY;	// 3DEditorで変更
    return GX2_SURFACE_DIM_2D;
}

//-----------------------------------------------------------------------------
//! @brief サーフェスの次元から中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetSurfaceDimensionString(const GX2SurfaceDim dimension, const bool isArray)
{
    switch (dimension)
    {
        case GX2_SURFACE_DIM_1D:            return "1d";
        case GX2_SURFACE_DIM_2D:            return "2d";
        case GX2_SURFACE_DIM_3D:            return "3d";
        case GX2_SURFACE_DIM_CUBE:          return (!isArray) ? "cube" : "cube_array";
        case GX2_SURFACE_DIM_1D_ARRAY:      return "1d_array";
        case GX2_SURFACE_DIM_2D_ARRAY:      return "2d_array";
        case GX2_SURFACE_DIM_2D_MSAA:       return "2d_msaa";
        case GX2_SURFACE_DIM_2D_MSAA_ARRAY: return "2d_msaa_array";
        default:                            return "invalid";
    }
}

//-----------------------------------------------------------------------------
//! @brief フォーマットと文字列のペアのデータです。
//-----------------------------------------------------------------------------
namespace {

const RIntAndCharPtr s_FormatStrings[] =
{
    { GX2_SURFACE_FORMAT_TC_R8_UNORM          , "unorm_8"       },
    { GX2_SURFACE_FORMAT_TC_R8_SNORM          , "snorm_8"       },
    { GX2_SURFACE_FORMAT_TC_R8_UINT           , "uint_8"        },
    { GX2_SURFACE_FORMAT_TC_R8_SINT           , "sint_8"        },

    { GX2_SURFACE_FORMAT_T_R4_G4_UNORM        , "unorm_4_4"     }, // 自前で対応
    { GX2_SURFACE_FORMAT_TC_R8_G8_UNORM       , "unorm_8_8"     },
    { GX2_SURFACE_FORMAT_TC_R8_G8_SNORM       , "snorm_8_8"     },
    { GX2_SURFACE_FORMAT_TC_R8_G8_UINT        , "uint_8_8"      },
    { GX2_SURFACE_FORMAT_TC_R8_G8_SINT        , "sint_8_8"      },

    { GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM   , "unorm_5_6_5"   }, // 自前で対応

    { GX2_SURFACE_FORMAT_TC_R5_G5_B5_A1_UNORM , "unorm_5_5_5_1" }, // 自前で対応
    { GX2_SURFACE_FORMAT_TC_R4_G4_B4_A4_UNORM , "unorm_4_4_4_4" }, // 自前で対応

    { GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, "unorm_8_8_8_8" },
    { GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SNORM , "snorm_8_8_8_8" },
    { GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB , "srgb_8_8_8_8"  },
    { GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_UINT  , "uint_8_8_8_8"  },
    { GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SINT  , "sint_8_8_8_8"  },

    { GX2_SURFACE_FORMAT_TC_R16_FLOAT            , "float_16"          },
    { GX2_SURFACE_FORMAT_TCD_R32_FLOAT           , "float_32"          },
    { GX2_SURFACE_FORMAT_TC_R16_G16_FLOAT        , "float_16_16"       },
    { GX2_SURFACE_FORMAT_TC_R32_G32_FLOAT        , "float_32_32"       },
    { GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT    , "float_11_11_10"    },
    { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_FLOAT, "float_16_16_16_16" },
    { GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_FLOAT, "float_32_32_32_32" },

    { GX2_SURFACE_FORMAT_T_BC1_UNORM, "unorm_bc1" },
    { GX2_SURFACE_FORMAT_T_BC1_SRGB , "srgb_bc1"  },
    { GX2_SURFACE_FORMAT_T_BC2_UNORM, "unorm_bc2" },
    { GX2_SURFACE_FORMAT_T_BC2_SRGB , "srgb_bc2"  },
    { GX2_SURFACE_FORMAT_T_BC3_UNORM, "unorm_bc3" },
    { GX2_SURFACE_FORMAT_T_BC3_SRGB , "srgb_bc3"  },
    { GX2_SURFACE_FORMAT_T_BC4_UNORM, "unorm_bc4" },
    { GX2_SURFACE_FORMAT_T_BC4_SNORM, "snorm_bc4" },
    { GX2_SURFACE_FORMAT_T_BC5_UNORM, "unorm_bc5" },
    { GX2_SURFACE_FORMAT_T_BC5_SNORM, "snorm_bc5" },
};

} // namespace

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列からサーフェスのフォーマットを取得します。
//-----------------------------------------------------------------------------
GX2SurfaceFormat RGetSurfaceFormatFromString(const std::string& str)
{
    const int fmtCount = sizeof(s_FormatStrings) / sizeof(RIntAndCharPtr);
    for (int fmtIdx = 0; fmtIdx < fmtCount; ++fmtIdx)
    {
        const RIntAndCharPtr& fmtStr = s_FormatStrings[fmtIdx];
        if (str == fmtStr.m_String)
        {
            return static_cast<GX2SurfaceFormat>(fmtStr.m_Value);
        }
    }
    return GX2_SURFACE_FORMAT_INVALID;
}

//-----------------------------------------------------------------------------
//! @brief サーフェスのフォーマットから中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetSurfaceFormatString(const GX2SurfaceFormat format)
{
    const int fmtCount = sizeof(s_FormatStrings) / sizeof(RIntAndCharPtr);
    for (int fmtIdx = 0; fmtIdx < fmtCount; ++fmtIdx)
    {
        const RIntAndCharPtr& fmtStr = s_FormatStrings[fmtIdx];
        if (format == fmtStr.m_Value)
        {
            return fmtStr.m_String;
        }
    }
    return "invalid";
}

//-----------------------------------------------------------------------------
//! @brief サーフェスのフォーマットが BC1/BC2/BC3 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RIsBC123SurfaceFormat(const GX2SurfaceFormat format)
{
    switch (format)
    {
        case GX2_SURFACE_FORMAT_T_BC1_UNORM:
        case GX2_SURFACE_FORMAT_T_BC1_SRGB:
        case GX2_SURFACE_FORMAT_T_BC2_UNORM:
        case GX2_SURFACE_FORMAT_T_BC2_SRGB:
        case GX2_SURFACE_FORMAT_T_BC3_UNORM:
        case GX2_SURFACE_FORMAT_T_BC3_SRGB:
            return true;
        default:
            return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief サーフェスのフォーマットが BC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RIsBCSurfaceFormat(const GX2SurfaceFormat format)
{
    if (RIsBC123SurfaceFormat(format))
    {
        return true;
    }
    else
    {
        switch (format)
        {
            case GX2_SURFACE_FORMAT_T_BC4_UNORM:
            case GX2_SURFACE_FORMAT_T_BC4_SNORM:
            case GX2_SURFACE_FORMAT_T_BC5_UNORM:
            case GX2_SURFACE_FORMAT_T_BC5_SNORM:
                return true;
            default:
                return false;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief サーフェスのフォーマットが浮動小数点数型なら true を返します。
//-----------------------------------------------------------------------------
bool RIsFloatSurfaceFormat(const GX2SurfaceFormat format)
{
    switch (format)
    {
        case GX2_SURFACE_FORMAT_TC_R16_FLOAT:
        case GX2_SURFACE_FORMAT_TCD_R32_FLOAT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_FLOAT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_FLOAT:
        case GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_FLOAT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_FLOAT:
        case GX2_SURFACE_FORMAT_T_R32_FLOAT_X8_X24:
            return true;
        default:
            return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief サーフェスのフォーマットが整数型なら true を返します。
//-----------------------------------------------------------------------------
bool RIsIntSurfaceFormat(const GX2SurfaceFormat format)
{
    switch (format)
    {
        case GX2_SURFACE_FORMAT_TC_R8_UINT:
        case GX2_SURFACE_FORMAT_TC_R8_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_UINT:
        case GX2_SURFACE_FORMAT_TC_R16_SINT:
        case GX2_SURFACE_FORMAT_TC_R32_UINT:
        case GX2_SURFACE_FORMAT_TC_R32_SINT:
        case GX2_SURFACE_FORMAT_TC_R8_G8_UINT:
        case GX2_SURFACE_FORMAT_TC_R8_G8_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_UINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_SINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_UINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_SINT:
        case GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_UINT:
        case GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SINT:
        case GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_UINT:
        case GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_SINT:
        case GX2_SURFACE_FORMAT_TC_A2_B10_G10_R10_UINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_UINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_SINT:
        case GX2_SURFACE_FORMAT_T_X24_G8_UINT:
        case GX2_SURFACE_FORMAT_T_X32_G8_UINT_X24:
            return true;
        default:
            return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief サーフェスのフォーマットが符号ありなら true を返します。
//-----------------------------------------------------------------------------
bool RIsSignedSurfaceFormat(const GX2SurfaceFormat format)
{
    if (RIsFloatSurfaceFormat(format))
    {
        return true;
    }

    switch (format)
    {
        case GX2_SURFACE_FORMAT_TC_R8_SNORM:
        case GX2_SURFACE_FORMAT_TC_R8_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_SNORM:
        case GX2_SURFACE_FORMAT_TC_R16_SINT:
        case GX2_SURFACE_FORMAT_TC_R8_G8_SNORM:
        case GX2_SURFACE_FORMAT_TC_R8_G8_SINT:
        case GX2_SURFACE_FORMAT_TC_R32_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_SNORM:
        case GX2_SURFACE_FORMAT_TC_R16_G16_SINT:
        case GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_SNORM:
        case GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_SINT:
        case GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SNORM:
        case GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SNORM:
        case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_SINT:
        case GX2_SURFACE_FORMAT_T_BC4_SNORM:
        case GX2_SURFACE_FORMAT_T_BC5_SNORM:
            return true;
        default:
            return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief サーフェスのフォーマットが sRGB フェッチなら true を返します。
//-----------------------------------------------------------------------------
bool RIsSrgbFetchSurfaceFormat(const GX2SurfaceFormat format)
{
    switch (format)
    {
        case GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB:
        case GX2_SURFACE_FORMAT_T_BC1_SRGB:
        case GX2_SURFACE_FORMAT_T_BC2_SRGB:
        case GX2_SURFACE_FORMAT_T_BC3_SRGB:
            return true;
        default:
            return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列からタイルモードを取得します。
//-----------------------------------------------------------------------------
GX2TileMode RGetTileModeFromString(const std::string& str)
{
    if (str == "linear_special") return GX2_TILE_MODE_LINEAR_SPECIAL;
    if (str == "linear_aligned") return GX2_TILE_MODE_LINEAR_ALIGNED;
    if (str == "Item1d_tiled_thin1") return GX2_TILE_MODE_1D_TILED_THIN1;		// 3DEditorで変更
    if (str == "Item1d_tiled_thick") return GX2_TILE_MODE_1D_TILED_THICK;		// 3DEditorで変更
    if (str == "Item2d_tiled_thin1") return GX2_TILE_MODE_2D_TILED_THIN1;		// 3DEditorで変更
    if (str == "Item2d_tiled_thin2") return GX2_TILE_MODE_2D_TILED_THIN2;		// 3DEditorで変更
    if (str == "Item2d_tiled_thin4") return GX2_TILE_MODE_2D_TILED_THIN4;		// 3DEditorで変更
    if (str == "Item2d_tiled_thick") return GX2_TILE_MODE_2D_TILED_THICK;		// 3DEditorで変更
    if (str == "Item2b_tiled_thin1") return GX2_TILE_MODE_2B_TILED_THIN1;		// 3DEditorで変更
    if (str == "Item2b_tiled_thin2") return GX2_TILE_MODE_2B_TILED_THIN2;		// 3DEditorで変更
    if (str == "Item2b_tiled_thin4") return GX2_TILE_MODE_2B_TILED_THIN4;		// 3DEditorで変更
    if (str == "Item2b_tiled_thick") return GX2_TILE_MODE_2B_TILED_THICK;		// 3DEditorで変更
    if (str == "Item3d_tiled_thin1") return GX2_TILE_MODE_3D_TILED_THIN1;		// 3DEditorで変更
    if (str == "Item3d_tiled_thick") return GX2_TILE_MODE_3D_TILED_THICK;		// 3DEditorで変更
    if (str == "Item3b_tiled_thin1") return GX2_TILE_MODE_3B_TILED_THIN1;		// 3DEditorで変更
    if (str == "Item3b_tiled_thick") return GX2_TILE_MODE_3B_TILED_THICK;		// 3DEditorで変更
    return GX2_TILE_MODE_DEFAULT;
}

//-----------------------------------------------------------------------------
//! @brief タイルモードから中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetTileModeString(const GX2TileMode tileMode)
{
    switch (tileMode)
    {
        case GX2_TILE_MODE_DEFAULT       : return "linear_aligned";
        case GX2_TILE_MODE_LINEAR_SPECIAL: return "linear_special";
        case GX2_TILE_MODE_LINEAR_ALIGNED: return "linear_aligned";
        case GX2_TILE_MODE_1D_TILED_THIN1: return "Item1d_tiled_thin1";		// 3DEditorで変更
        case GX2_TILE_MODE_1D_TILED_THICK: return "Item1d_tiled_thick";		// 3DEditorで変更
        case GX2_TILE_MODE_2D_TILED_THIN1: return "Item2d_tiled_thin1";		// 3DEditorで変更
        case GX2_TILE_MODE_2D_TILED_THIN2: return "Item2d_tiled_thin2";		// 3DEditorで変更
        case GX2_TILE_MODE_2D_TILED_THIN4: return "Item2d_tiled_thin4";		// 3DEditorで変更
        case GX2_TILE_MODE_2D_TILED_THICK: return "Item2d_tiled_thick";		// 3DEditorで変更
        case GX2_TILE_MODE_2B_TILED_THIN1: return "Item2b_tiled_thin1";		// 3DEditorで変更
        case GX2_TILE_MODE_2B_TILED_THIN2: return "Item2b_tiled_thin2";		// 3DEditorで変更
        case GX2_TILE_MODE_2B_TILED_THIN4: return "Item2b_tiled_thin4";		// 3DEditorで変更
        case GX2_TILE_MODE_2B_TILED_THICK: return "Item2b_tiled_thick";		// 3DEditorで変更
        case GX2_TILE_MODE_3D_TILED_THIN1: return "Item3d_tiled_thin1";		// 3DEditorで変更
        case GX2_TILE_MODE_3D_TILED_THICK: return "Item3d_tiled_thick";		// 3DEditorで変更
        case GX2_TILE_MODE_3B_TILED_THIN1: return "Item3b_tiled_thin1";		// 3DEditorで変更
        case GX2_TILE_MODE_3B_TILED_THICK: return "Item3b_tiled_thick";		// 3DEditorで変更
        default:                           return "invalid";
    }
}

//-----------------------------------------------------------------------------
//! @brief 成分選択の成分をあらわす文字のデータです。
//-----------------------------------------------------------------------------
namespace {

const int COMP_SEL_COUNT = 6;
const char* const s_CompSelChars[COMP_SEL_COUNT] =
{
    "r", "g", "b", "a", "0", "1"
};

} // namespace

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列から成分選択を取得します。
//-----------------------------------------------------------------------------
GX2CompSel RGetCompSelFromString(const std::string& str)
{
    int comps[R_RGBA_BYTES] =
    {
        GX2_COMPONENT_X_R,
        GX2_COMPONENT_Y_G,
        GX2_COMPONENT_Z_B,
        GX2_COMPONENT_W_A,
    };
    int validCount = 0;
    std::istringstream iss(str);
    for (int iRgba = 0; iRgba < R_RGBA_BYTES; ++iRgba)
    {
        if (iss)
        {
            std::string c;
            iss >> c;
            for (int iComp = 0; iComp < COMP_SEL_COUNT; ++iComp)
            {
                if (c == s_CompSelChars[iComp])
                {
                    comps[iRgba] = iComp;
                    ++validCount;
                    break;
                }
            }
        }
    }

    // validCount != R_RGBA_BYTES なら GX2_COMP_SEL_NONE を返す？

    return GX2_GET_COMP_SEL(comps[0], comps[1], comps[2], comps[3]);
}

//-----------------------------------------------------------------------------
//! @brief 成分選択から中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetCompSelString(const GX2CompSel compSel)
{
    std::string value;
    for (int iRgba = 0; iRgba < R_RGBA_BYTES; ++iRgba)
    {
        const int iComp = (compSel >> (3 - iRgba) * 8) & 0xff;
        if (iRgba > 0) value += " ";
        value += ((iComp < COMP_SEL_COUNT) ? s_CompSelChars[iComp] : "0");
    }
    return value;
}

//-----------------------------------------------------------------------------
//! @brief サーフェスのフォーマットに対応したデフォルトの成分選択を取得します。
//-----------------------------------------------------------------------------
GX2CompSel RGetDefaultCompSel(const GX2SurfaceFormat format)
{
    switch (format)
    {
        case GX2_SURFACE_FORMAT_TC_R8_UNORM:
        case GX2_SURFACE_FORMAT_TC_R8_UINT:
        case GX2_SURFACE_FORMAT_TC_R8_SNORM:
        case GX2_SURFACE_FORMAT_TC_R8_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_UINT:
        case GX2_SURFACE_FORMAT_TC_R16_SNORM:
        case GX2_SURFACE_FORMAT_TC_R16_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_FLOAT:
        case GX2_SURFACE_FORMAT_TC_R32_UINT:
        case GX2_SURFACE_FORMAT_TC_R32_SINT:
        case GX2_SURFACE_FORMAT_TCD_R32_FLOAT:
        case GX2_SURFACE_FORMAT_T_BC4_UNORM:
        case GX2_SURFACE_FORMAT_T_BC4_SNORM:
            // rrr1
            return GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_C_1);

        case GX2_SURFACE_FORMAT_T_R4_G4_UNORM:
        case GX2_SURFACE_FORMAT_TC_R8_G8_UNORM:
        case GX2_SURFACE_FORMAT_TC_R16_G16_UNORM:
        case GX2_SURFACE_FORMAT_T_BC5_UNORM:
            // rrrg
            return GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_Y_G);

        case GX2_SURFACE_FORMAT_TC_R8_G8_SNORM:
        case GX2_SURFACE_FORMAT_TC_R8_G8_UINT:
        case GX2_SURFACE_FORMAT_TC_R8_G8_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_SNORM:
        case GX2_SURFACE_FORMAT_TC_R16_G16_UINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_FLOAT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_UINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_SINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_FLOAT:
        case GX2_SURFACE_FORMAT_T_BC5_SNORM:
            // rg01
            return GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_Y_G, GX2_COMPONENT_C_0, GX2_COMPONENT_C_1);

        case GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM:
        case GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT:
            // rgb1
            return GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_Y_G, GX2_COMPONENT_Z_B, GX2_COMPONENT_C_1);

        default:
            // rgba
            return GX2_COMP_SEL_XYZW;
    }
}

//-----------------------------------------------------------------------------
//! @brief サーフェスのフォーマットから成分数を取得します。
//-----------------------------------------------------------------------------
int RGetCompSize(const GX2SurfaceFormat format)
{
    switch (format)
    {
        case GX2_SURFACE_FORMAT_TC_R8_UNORM:
        case GX2_SURFACE_FORMAT_TC_R8_UINT:
        case GX2_SURFACE_FORMAT_TC_R8_SNORM:
        case GX2_SURFACE_FORMAT_TC_R8_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_UINT:
        case GX2_SURFACE_FORMAT_TC_R16_SNORM:
        case GX2_SURFACE_FORMAT_TC_R16_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_FLOAT:
        case GX2_SURFACE_FORMAT_TC_R32_UINT:
        case GX2_SURFACE_FORMAT_TC_R32_SINT:
        case GX2_SURFACE_FORMAT_TCD_R32_FLOAT:
        case GX2_SURFACE_FORMAT_T_BC4_UNORM:
        case GX2_SURFACE_FORMAT_T_BC4_SNORM:
            return 1;

        case GX2_SURFACE_FORMAT_T_R4_G4_UNORM:
        case GX2_SURFACE_FORMAT_TC_R8_G8_UNORM:
        case GX2_SURFACE_FORMAT_TC_R8_G8_UINT:
        case GX2_SURFACE_FORMAT_TC_R8_G8_SNORM:
        case GX2_SURFACE_FORMAT_TC_R8_G8_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_UNORM:
        case GX2_SURFACE_FORMAT_TC_R16_G16_UINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_SNORM:
        case GX2_SURFACE_FORMAT_TC_R16_G16_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_FLOAT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_UINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_SINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_FLOAT:
        case GX2_SURFACE_FORMAT_T_BC5_UNORM:
        case GX2_SURFACE_FORMAT_T_BC5_SNORM:
            return 2;

        case GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM:
        case GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT:
            return 3;

        default:
            return 4;
    }
}

//-----------------------------------------------------------------------------
//! @brief サーフェスのフォーマットの 1 ピクセルのビット数を返します。
//-----------------------------------------------------------------------------
int RGetBitPerPixel(const GX2SurfaceFormat format)
{
    switch (format)
    {
        case GX2_SURFACE_FORMAT_T_BC1_UNORM:
        case GX2_SURFACE_FORMAT_T_BC1_SRGB:
        case GX2_SURFACE_FORMAT_T_BC4_UNORM:
        case GX2_SURFACE_FORMAT_T_BC4_SNORM:
            return 4;

        case GX2_SURFACE_FORMAT_TC_R8_UNORM:
        case GX2_SURFACE_FORMAT_TC_R8_UINT:
        case GX2_SURFACE_FORMAT_TC_R8_SNORM:
        case GX2_SURFACE_FORMAT_TC_R8_SINT:
        case GX2_SURFACE_FORMAT_T_R4_G4_UNORM:
        case GX2_SURFACE_FORMAT_T_BC2_UNORM:
        case GX2_SURFACE_FORMAT_T_BC2_SRGB:
        case GX2_SURFACE_FORMAT_T_BC3_UNORM:
        case GX2_SURFACE_FORMAT_T_BC3_SRGB:
        case GX2_SURFACE_FORMAT_T_BC5_UNORM:
        case GX2_SURFACE_FORMAT_T_BC5_SNORM:
            return 8;

        case GX2_SURFACE_FORMAT_TCD_R16_UNORM:
        case GX2_SURFACE_FORMAT_TC_R16_UINT:
        case GX2_SURFACE_FORMAT_TC_R16_SNORM:
        case GX2_SURFACE_FORMAT_TC_R16_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_FLOAT:
        case GX2_SURFACE_FORMAT_TC_R8_G8_UNORM:
        case GX2_SURFACE_FORMAT_TC_R8_G8_UINT:
        case GX2_SURFACE_FORMAT_TC_R8_G8_SNORM:
        case GX2_SURFACE_FORMAT_TC_R8_G8_SINT:
        case GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM:
        case GX2_SURFACE_FORMAT_TC_R5_G5_B5_A1_UNORM:
        case GX2_SURFACE_FORMAT_TC_R4_G4_B4_A4_UNORM:
        case GX2_SURFACE_FORMAT_TC_A1_B5_G5_R5_UNORM:
            return 16;

        case GX2_SURFACE_FORMAT_TC_R32_UINT:
        case GX2_SURFACE_FORMAT_TC_R32_SINT:
        case GX2_SURFACE_FORMAT_TCD_R32_FLOAT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_UNORM:
        case GX2_SURFACE_FORMAT_TC_R16_G16_UINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_SNORM:
        case GX2_SURFACE_FORMAT_TC_R16_G16_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_FLOAT:
        case GX2_SURFACE_FORMAT_T_R24_UNORM_X8:
        case GX2_SURFACE_FORMAT_T_X24_G8_UINT:
        case GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT:
        case GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM:
        case GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_UINT:
        case GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_SNORM:
        case GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_SINT:
        case GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM:
        case GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_UINT:
        case GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SNORM:
        case GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SINT:
        case GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB:
        case GX2_SURFACE_FORMAT_TCS_A2_B10_G10_R10_UNORM:
        case GX2_SURFACE_FORMAT_TC_A2_B10_G10_R10_UINT:
            return 32;

        case GX2_SURFACE_FORMAT_T_R32_FLOAT_X8_X24:
        case GX2_SURFACE_FORMAT_T_X32_G8_UINT_X24:
        case GX2_SURFACE_FORMAT_TC_R32_G32_UINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_SINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_FLOAT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UNORM:
        case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SNORM:
        case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SINT:
        case GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_FLOAT:
            return 64;

        case GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_UINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_SINT:
        case GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_FLOAT:
            return 128;

        default:
            return 32;
    }
}

//=============================================================================
// nwFtx ネームスペースを終了します。
//=============================================================================
} // namespace nwFtx

