﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

namespace TextureCompositor
{
    public static class Utility
    {
        public static IEnumerable<string> ExpandFilePathPatterns(string pathPatternsText, string directorySearchFilePattern = "*")
        {
            List<string> result = new List<string>();
            if (string.IsNullOrEmpty(pathPatternsText))
            {
                return result;
            }

            string[] pathPatterns = pathPatternsText.Split(';');
            foreach (string pathPattern in pathPatterns)
            {
                var expandedPathPattern = Environment.ExpandEnvironmentVariables(pathPattern);
                string fullPathPattern = expandedPathPattern;
                if (!System.IO.Path.IsPathRooted(expandedPathPattern))
                {
                    fullPathPattern = System.IO.Path.Combine(System.Environment.CurrentDirectory, expandedPathPattern);
                }

                if (!fullPathPattern.Contains('*'))
                {
                    if (System.IO.File.Exists(fullPathPattern))
                    {
                        result.Add(fullPathPattern);
                        continue;
                    }
                    else if (System.IO.Directory.Exists(fullPathPattern))
                    {
                        result.AddRange(System.IO.Directory.EnumerateFiles(fullPathPattern, directorySearchFilePattern));
                        continue;
                    }
                    else
                    {
                        throw new Exception($"path was not found: {fullPathPattern}");
                    }
                }
                else
                {
                    var root = System.IO.Directory.GetDirectoryRoot(fullPathPattern);
                    var searchPattern = fullPathPattern.Substring(root.Length);

                    string fileNamePattern = System.IO.Path.GetFileName(fullPathPattern);
                    bool isFilePathPattern = fileNamePattern.Contains("*");
                    if (isFilePathPattern)
                    {
                        IEnumerable<string> files = System.IO.Directory.EnumerateFiles(root, searchPattern);
                        result.AddRange(files);
                    }
                    else
                    {
                        IEnumerable<string> folders = System.IO.Directory.EnumerateDirectories(root, searchPattern);
                        foreach (string folder in folders)
                        {
                            result.AddRange(System.IO.Directory.EnumerateFiles(folder, directorySearchFilePattern, System.IO.SearchOption.TopDirectoryOnly));
                        }
                    }
                }
            }

            return result.Distinct();
        }

        public static void ExecuteProcess(string cmd, string args)
        {
            StringBuilder stdout = new StringBuilder();
            StringBuilder stderror = new StringBuilder();
            try
            {
                int result = ExecuteProcessImpl(
                    cmd, args, false,
                    x => stdout.AppendLine(x),
                    x => stderror.AppendLine(x));
                Console.WriteLine(stdout.ToString());
                if (result != 0)
                {
                    throw new Exception($"エラーコード:0x{result.ToString("X8")}");
                }
            }
            catch (Exception e)
            {
                throw new Exception($"{cmd} {args} の実行に失敗しました。\n{e.Message}\n{stderror.ToString()}");
            }
        }

        public static void ExecuteProcess(string cmd, string args,
            Action<string> writeMessageAction,
            Action<string> errorMessageAction)
        {
            writeMessageAction($"{cmd} {args}");
            StringBuilder errorMessage = new StringBuilder();
            try
            {
                int result = ExecuteProcessImpl(
                    cmd, args, false,
                    writeMessageAction,
                    x => {
                        errorMessageAction(x);
                        errorMessage.AppendLine(x);
                    });
                if (result != 0)
                {
                    throw new Exception($"{cmd} {args} の実行に失敗しました。エラーコード:0x{result.ToString("X8")}");
                }
            }
            catch (Exception e)
            {
                throw new Exception($"{cmd} {args} の実行に失敗しました。\n{e.Message}\n{errorMessage.ToString()}");
            }
        }

        private static int ExecuteProcessImpl(
            string cmd, string args, bool useShellExecute,
            Action<string> messageAction,
            Action<string> errorMessageAction)
        {
            var info = new ProcessStartInfo(cmd, args) { UseShellExecute = useShellExecute };
            if (!info.UseShellExecute)
            {
                info.RedirectStandardError = true;
                info.RedirectStandardOutput = true;
                info.CreateNoWindow = true;
            }

            var process = new Process { StartInfo = info };
            if (messageAction != null)
            {
                process.OutputDataReceived += (s, e) =>
                {
                    if (e.Data != null)
                    {
                        messageAction(e.Data);
                    }
                };
            }

            if (errorMessageAction != null)
            {
                process.ErrorDataReceived += (s, e) =>
                {
                    if (e.Data != null)
                    {
                        errorMessageAction(e.Data);
                    }
                };
            }

            process.Start();

            if (!info.UseShellExecute)
            {
                process.BeginOutputReadLine();
                process.BeginErrorReadLine();
            }

            process.WaitForExit();

            return process.ExitCode;
        }

        public static string GetTempFilePath(string baseFileName)
        {
            return GetTempPath($"_{baseFileName}");
        }

        public static string GetTempFolderPath()
        {
            return GetTempPath(string.Empty);
        }

        private static string GetTempPath(string baseName)
        {
            string suffix = $"_{baseName}";
            if (string.IsNullOrEmpty(baseName))
            {
                suffix = string.Empty;
            }

            return System.IO.Path.Combine(
                System.IO.Path.GetTempPath(),
                $"Nintendo_{Assembly.GetExecutingAssembly().GetName().Name}_{Guid.NewGuid().ToString("N")}{suffix}");
        }

        public static IEnumerable<Assembly> EnumerateAssemblies()
        {
            var mainAsm = Assembly.GetExecutingAssembly();
            yield return mainAsm;
            foreach (var refAsmName in mainAsm.GetReferencedAssemblies())
            {
                yield return Assembly.Load(refAsmName);
            }
        }
    }
}
