﻿using Nintendo.ToolFoundation.CommandLine;
using nw.g3d.iflib;
using nw.g3d.nw4f_3dif;
using System;
using System.Collections.Generic;
using System.Linq;
using IronPython.Hosting;
using Microsoft.Scripting.Hosting;
using Nintendo.G3dTool.Entities;
using System.Diagnostics;
using System.Reflection;
using nw.g3d;
using System.Resources;
using nw.g3d.toollib;

namespace _3dIntermediateFileShaderVariationGenerator
{
    public class Context
    {
        private bool isSilent = false;

        public string OutputFsvPath { get; private set; } = string.Empty;

        public List<string> InputFmdPaths { get; } = new List<string>();

        public string InputFsdPath { get; private set; } = string.Empty;

        public string SkinningCountOptionId { get; private set; } = string.Empty;

        public string InputCommonFilterPath { get; private set; } = string.Empty;

        public string InputFilterPath { get; private set; } = string.Empty;

        public Context(string[] args)
        {
            // 文字リソースを初期化
            Strings.Initialize("_3dIntermediateFileShaderVariationGenerator.Resources.StringResource");
            G3dLocalization.SetCulture();

            var CommandLineResource = new ResourceManager(
                "_3dIntermediateFileShaderVariationGenerator.Resources.CommandLineHelp", Assembly.GetEntryAssembly() ?? Assembly.GetCallingAssembly());

            // 引数パース
            var rootCommand = ActionCommand.CreateRootCommand(true);
            rootCommand.GetBuilder().SetDescription(CommandLineResource.GetString("ApplicationDescription"));
            rootCommand.AddFlagOption('s', "silent", () => this.isSilent = true)
                .GetBuilder()
                .SetDescription(CommandLineResource.GetString("Silent"));

            rootCommand.AddValue(-1, path =>
            {
                string extension = System.IO.Path.GetExtension(path);
                if(extension.Contains("fmd"))
                {
                    // ファイル指定の場合
                    InputFmdPaths.Add(path);
                }
                else
                {
                    // ディレクトリ指定の場合
                    string[] files = System.IO.Directory.GetFiles(path, "*", System.IO.SearchOption.AllDirectories);
                    foreach(string file in files)
                    {
                        extension = System.IO.Path.GetExtension(file);
                        if (extension.Contains("fmd"))
                        {
                            InputFmdPaths.Add(file);
                        }
                    }
                }

                if (InputFmdPaths.Count() == 0)
                {
                    Strings.Throw("NotFoundFmd", path);
                }
            })
                .GetBuilder()
                .Require()
                .SetDescription(CommandLineResource.GetString("PositionalArgument"))
                .SetValueName("path");

            rootCommand.AddValueOption("shader-definition", path =>
            {
                string extension = System.IO.Path.GetExtension(path);
                if (extension.Contains("fsdb"))
                {
                    InputFsdPath = path;
                }
            })
                .GetBuilder()
                .Require()
                .SetDescription(CommandLineResource.GetString("ShaderDefinition"))
                .SetValueName("path");

            rootCommand.AddValueOption("skinning-count-option-id", name =>
            {
                SkinningCountOptionId = name;
            })
                .GetBuilder()
                .SetDescription(CommandLineResource.GetString("SkinningCountOptionId"))
                .SetValueName("name");

            rootCommand.AddValueOption("common-script", path =>
            {
                string extension = System.IO.Path.GetExtension(path);
                InputCommonFilterPath = path;
            })
                .GetBuilder()
                .SetDescription(CommandLineResource.GetString("CommonScript"))
                .SetValueName("path")
                .Hide();

            rootCommand.AddValueOption("script", path =>
            {
                string extension = System.IO.Path.GetExtension(path);
                InputFilterPath = path;
            })
                .GetBuilder()
                .SetDescription(CommandLineResource.GetString("Script"))
                .SetValueName("path")
                .Hide();

            rootCommand.AddValueOption('o', "output", path => this.OutputFsvPath = path)
                .GetBuilder()
                .SetDescription(CommandLineResource.GetString("Output"))
                .SetValueName("path");

            CommandLine.ParseArgs(args, rootCommand, new ParseSettings()
            {
                ErrorAction = message =>
                {
                    Console.WriteLine(CommandLine.GetHelpText(rootCommand));
                    throw new Exception(message);
                },
                HelpWriter = this.WriteMessage
            });

            if (string.IsNullOrEmpty(this.InputFsdPath))
            {
                Strings.Throw("NotFoundFsdb");
            }
        }

        public void WriteMessage(string message)
        {
            if (this.isSilent)
            {
                return;
            }

            Console.WriteLine(message);
        }

        public static void WriteErrorMessage(string message)
        {
            Console.Error.WriteLine($"{Strings.Get("Error")}: {message}");
        }
    }

    public static class Program
    {
        static Context context;

        // 集計対象のシェーダーアーカイブ名
        static string targetShaderArchiveName = string.Empty;

        // スキニングオプション id
        static string skinningCountOptionId = string.Empty;

        static ScriptEngine engine;
        static ScriptSource scriptCommon;
        static ScriptSource script;

        public static void Execute(string[] args)
        {
            var stopwatch = new System.Diagnostics.Stopwatch();
            stopwatch.Start();

            context = new Context(args);

            // Python を初期化&取得
            engine = Python.CreateEngine();
            if (context.InputCommonFilterPath != string.Empty)
            {
                scriptCommon = engine.CreateScriptSourceFromFile(context.InputCommonFilterPath);
            }
            else
            {
                scriptCommon = null;
            }

            if (context.InputFilterPath != string.Empty)
            {
                script = engine.CreateScriptSourceFromFile(context.InputFilterPath);
            }
            else
            {
                script = null;
            }

            // fsd ファイルを取得
            var shaderDefFile = IfReadUtility.ReadIntermediateFile(context.InputFsdPath, G3dToolUtility.GetXsdBasePath());
            ShaderDefinition shaderDef = shaderDefFile.GetRootEntity<ShaderDefinition>();

            // 検索対象の shaderArchiveName を取得
            targetShaderArchiveName = System.IO.Path.GetFileNameWithoutExtension(context.InputFsdPath);

            // fmd から 対象の shaderArchive を持つモデルを集計
            var inputModels = GatherModels();

            // オプションで指定したスキニングオプションが fsd 内に 1つでも存在するかを確認
            skinningCountOptionId = context.SkinningCountOptionId;
            if (!string.IsNullOrEmpty(skinningCountOptionId))
            {
                CheckSkinningCountOptionExist(shaderDef);
            }

            // fsv を作成
            IntermediateFile shaderVariationFile = CreateShaderVariationFile(inputModels, targetShaderArchiveName, shaderDef);
            string outputPath = context.OutputFsvPath;
            if (string.IsNullOrEmpty(outputPath))
            {
                // 出力パスが無いときは カレントディレクトリに 「fsdファイル名 + fsva」で出力する
                outputPath = System.IO.Path.Combine(System.Environment.CurrentDirectory, targetShaderArchiveName + ".fsva");
            }

            IfWriteUtility.WriteIntermediateFile(shaderVariationFile, outputPath, G3dToolUtility.GetXsdBasePath());

            stopwatch.Stop();
            DebugWriteLine($"　{stopwatch.Elapsed.Seconds}秒 {stopwatch.Elapsed.Milliseconds}ミリ秒 {inputModels.Count()}/{context.InputFmdPaths.Count}個");
        }

        public static IntermediateFile CreateShaderVariationFile(
            IEnumerable<Model> inputModels, string shaderArchiveName, ShaderDefinition shaderDefinition)
        {
            var shaderVariationFile = new IntermediateFile(IntermediateFileKind.ShaderVariation);
            var shaderVariation = shaderVariationFile.GetRootEntity<ShaderVariation>();
            shaderVariation.ShaderVariationInfo.ShaderArchive = shaderArchiveName;

            // 各シェーディングモデルのシェーダープログラム一覧を作成
            foreach (ShadingModel shadingModel in shaderDefinition.ShadingModels)
            {
                // シェーディングモデルがアサインされたモデルが存在するかを検索
                bool isTargetModelAvailable = inputModels.Any(model => model.Materials.Any(
                    mat => mat.ShaderAssign.ShaderArchive.Equals(shaderArchiveName)
                        && mat.ShaderAssign.ShadingModel.Equals(shadingModel.Name)));
                if (isTargetModelAvailable)
                {
                    var targetShader = new TargetShader()
                    {
                        ShadingModelName = shadingModel.Name,
                    };
                    AddShaderPrograms(targetShader, inputModels, shaderArchiveName, shadingModel);
                    shaderVariation.TargetShaders.Add(targetShader);
                }
            }

            return shaderVariationFile;
        }

        static void Main(string[] args)
        {
#if !DEBUG
            try
#endif
            {
                Execute(args);
            }
#if !DEBUG
            catch (Exception exception)
            {
                Context.WriteErrorMessage(exception.Message);
                var innerException = exception.InnerException;
                while (innerException != null)
                {
                    Context.WriteErrorMessage(innerException.Message);
                    innerException = innerException.InnerException;
                }

                Environment.ExitCode = 1;
            }
#endif
        }

        [Conditional("DEBUG")]
        private static void DebugWriteLine(string message)
        {
            Console.WriteLine(message);
        }

        private static IEnumerable<Model> GatherModels()
        {
            // 指定したシェーダーアーカイブがアサインされたモデル中間ファイルを検索する
            List<Model> inputModels = new List<Model>();
            foreach (string fmdpath in context.InputFmdPaths)
            {
                nw4f_3difType fileData = IfBinaryReadUtility.ReadTextPartOnly(fmdpath, null);
                var file = new IntermediateFile(fileData);
                var model = file.GetRootEntity<Model>();
                foreach (var material in model.Materials)
                {
                    if (!material.IsShaderAssigned)
                    {
                        continue;
                    }

                    var assign = material.ShaderAssign;
                    if (assign.ShaderArchive.Equals(targetShaderArchiveName))
                    {
                        inputModels.Add(model);
                        break;
                    }
                }
            }
            if (inputModels.Count == 0)
            {
                throw new Exception(Strings.Get("NotFoundModel"));
            }
            return inputModels;
        }

        private static void CheckSkinningCountOptionExist(ShaderDefinition shaderDefinition)
        {
            foreach (ShadingModel shadingModel in shaderDefinition.ShadingModels)
            {
                foreach (var optionParam in shadingModel.OptionVars)
                {
                    if (optionParam.Id.Equals(skinningCountOptionId))
                    {
                        // static オプションが指定されたらエラー
                        if (optionParam.Type == option_var_typeType.@static)
                        {
                            throw new Exception(Strings.Get("StaticSkinningCountOption"));
                        }
                        return;
                    }
                }
            }
            // fsd 内に指定したオプション id が存在しない場合はエラー
            throw new Exception(Strings.Get("NotFoundSkinningCountOption"));
        }

        private static IEnumerable<Option> CreateDefaultOptions(ShadingModel shadingModel)
        {
            List<Option> outOptions = new List<Option>();
            foreach (var optionParam in shadingModel.OptionVars)
            {
                if (optionParam.Type == option_var_typeType.@static)
                {
                    // static の場合は choice=default にしておく
                    var option = new Option()
                    {
                        Id = optionParam.Id,
                    };
                    option.Choice.SetValue(optionParam.Default);
                    outOptions.Add(option);
                }
                else
                {
                    // dynamic の場合は fsd の全バリエーションを指定しておく
                    var option = new Option()
                    {
                        Id = optionParam.Id,
                    };
                    option.Choice.DeepCopyFrom(optionParam.Choice);
                    outOptions.Add(option);
                }
            }

            // CommonScript で一括でデフォルトのオプションを再設定
            if (scriptCommon != null)
            {
                // ユーザーのフィルターで fsv の shader_program を書き換え
                ScriptScope scope = engine.CreateScope();
                scriptCommon.Execute(scope);

                option_arrayType outOptionArray = new option_arrayType()
                {
                    length = outOptions.Count(),
                    option = outOptions.Select(x => x.CreateSerializableData()).ToArray()
                };
                scope.SetVariable("outOptionArray", outOptionArray);
                scope.SetVariable("inShadingModel", shadingModel.CreateSerializableData());
                engine.Execute("Filter(outOptionArray, inShadingModel)", scope);
                return outOptionArray.Items.Select(x => new Option(x));
            }
            else
            {
                return outOptions;
            }
        }

        private static ShaderProgram CreateShaderProgram(IEnumerable<Option> defaultOptions, Model model, Shape shape, Material material)
        {
            ShaderProgram program = new ShaderProgram();
            List<Option> outOptions = new List<Option>();

            // デフォルト値で option_array を作成
            foreach (var defaultOption in defaultOptions)
            {
                var option = new Option()
                {
                    Id = defaultOption.Id,
                };
                option.Choice.DeepCopyFrom(defaultOption.Choice);
                outOptions.Add(option);
            }

            // モデルのシェーダーアサイン情報を choice に上書き反映
            foreach (var assignedOption in material.ShaderAssign.ShaderOptions)
            {
                foreach (Option outOption in outOptions)
                {
                    if (assignedOption.Id.Equals(outOption.Id))
                    {
                        outOption.Choice.SetValue(assignedOption.Value);
                    }
                }
            }

            // スキニング数を示すダイナミックオプションの choice を shape から取得する
            if(!string.IsNullOrEmpty(skinningCountOptionId))
            {
                foreach (Option outOption in outOptions)
                {
                    if (skinningCountOptionId.Equals(outOption.Id))
                    {
                        outOption.Choice.SetValue(shape.ShapeInfo.VertexSkinningCount.ToString());
                    }
                }
            }

            // script で choice を上書き反映
            if (script != null)
            {
                ScriptScope scope = engine.CreateScope();
                script.Execute(scope);

                option_arrayType outOptionArray = new option_arrayType()
                {
                    length = outOptions.Count(),
                    option = outOptions.Select(x => x.CreateSerializableData()).ToArray()
                };
                scope.SetVariable("outOptionArray", outOptionArray);
                scope.SetVariable("inModel", model.CreateSerializableData());
                scope.SetVariable("inShape", shape.CreateSerializableData());
                scope.SetVariable("inMaterial", material.CreateSerializableData());
                engine.Execute("Filter(outOptionArray, inModel, inShape, inMaterial)", scope);

                program.Options.Add(outOptionArray.Items.Select(x => new Option(x)));
            }
            else
            {
                program.Options.Add(outOptions);
            }

            return program;
        }

        private static void AddShaderPrograms(
            TargetShader target, IEnumerable<Model> inputModels, string shaderArchiveName, ShadingModel shadingModel)
        {
            if (shadingModel.OptionVars.Count > 0)
            {
                // fsd の option からデフォルトのオプションを作成。static は default値、dynamic は全ての範囲
                IEnumerable<Option> defaultOptions = CreateDefaultOptions(shadingModel);

                // 各モデルから検索対象の shadingmodel と一致するアサイン情報を探す
                foreach (var model in inputModels)
                {
                    // シェイプが無い場合は集計しない
                    if (model.Shapes.Count == 0)
                    {
                        continue;
                    }
                    // シェイプからマテリアルを引く
                    foreach (var shape in model.Shapes)
                    {
                        string matName = shape.ShapeInfo.MatName;
                        foreach (var material in model.Materials)
                        {
                            // マテリアルアサインが存在しない場合は集計しない
                            if (material.IsShaderAssigned == false || !material.Name.Equals(matName))
                            {
                                continue;
                            }
                            var assign = material.ShaderAssign;
                            if (assign.ShaderArchive.Equals(shaderArchiveName) && assign.ShadingModel.Equals(shadingModel.Name))
                            {
                                ShaderProgram newShaderProgram = CreateShaderProgram(defaultOptions, model, shape, material);

                                // 同じ内容の ShaderProgram がなければ追加
                                if (!target.ShaderPrograms.Any(x => x.Equals(newShaderProgram)))
                                {
                                    target.ShaderPrograms.Add(newShaderProgram);
                                }
                            }
                        }
                    }
                }
            }

            // オプションがない場合、空のシェーダープログラムを追加
            if(target.ShaderPrograms.Count == 0)
            {
                target.ShaderPrograms.Add(new ShaderProgram());
            }
        }
    }
}
