﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Xml;

namespace nw.g3d.nw4f_3dif
{
    /// <summary>
    /// ストリーム用のユーティリティです。
    /// </summary>
    public static class G3dStreamUtility
    {
        /// <summary>
        /// 中間ファイルがストリーム配列を持っているか確認します。
        /// </summary>
        /// <param name="nw4f_3dif">確認する中間ファイルです。</param>
        /// <returns>中間ファイルがストリーム配列を持っていれば true を返します。</returns>
        public static bool HasStreamArray(nw4f_3difType nw4f_3dif)
        {
            return ((nw4f_3dif.RootElement != null) &&
                (nw4f_3dif.RootElement.stream_array != null));
        }

        /// <summary>
        /// ストリーム配列からストリーム列に変換します。
        /// </summary>
        /// <param name="streams">結果を格納するストリーム列です。</param>
        /// <param name="stream_array">変換元のストリーム配列です。</param>
        public static void ToStreams(List<G3dStream> streams, stream_arrayType stream_array)
        {
            int count = stream_array.stream.Length;
            G3dStream[] streamsTemp = new G3dStream[count];
            G3dParallel.For(0, count, delegate(int i)
            {
                streamsTemp[i] = new G3dStream(stream_array.stream[i]);
            });
            streams.Clear();
            streams.AddRange(streamsTemp);
        }

        /// <summary>
        /// ストリーム列からストリーム配列に変換します。
        /// </summary>
        /// <param name="streams">変換元のストリーム列です。</param>
        /// <returns>変換結果のストリーム配列を返します。
        /// ストリーム配列の内容が無い場合は、null を返します。</returns>
        public static stream_arrayType ToStreamArray(List<G3dStream> streams)
        {
            if (streams.Count == 0) { return null; }

            stream_arrayType stream_array = new stream_arrayType();
            stream_array.length = streams.Count;
            stream_array.stream = new streamType[streams.Count];

            G3dParallel.For(0, streams.Count, delegate(int i)
            {
                streamType stream = streams[i].GetText(i);
                stream_array.stream[i] = stream;
            });
            return stream_array;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// XML 中間ファイルにストリーム列を設定します。
        /// </summary>
        /// <param name="document">設定先の XML 中間ファイルです。</param>
        /// <param name="streams">設定するストリーム列です。</param>
        public static void SetStreams(
            XmlDocument document, List<G3dStream> streams)
        {
            if (streams.Count == 0) { return; }

            XmlNode node = document.SelectSingleNode("/nw4f_3dif/*[last()]/*[last()]");
            XmlElement element = node as XmlElement;
            Nintendo.Foundation.Contracts.Assertion.Operation.True(element != null);

            if (element.Name == "user_tool_data")
            {
                element = element.PreviousSibling as XmlElement;
            }
            if (element.Name == "tool_data")
            {
                element = element.PreviousSibling as XmlElement;
            }
            if (element.Name == "comment")
            {
                element = element.PreviousSibling as XmlElement;
            }

            XmlElement stream_array = document.CreateElement("stream_array");
            stream_array.SetAttribute("length", streams.Count.ToString());
            for (int i = 0; i < streams.Count; i++)
            {
                stream_array.AppendChild(streams[i].GetElement(document, i));
            }

            element.ParentNode.InsertAfter(stream_array, element);
        }

        /// <summary>
        /// XML 中間ファイルからストリーム列を取得します。
        /// </summary>
        /// <param name="streams">結果を格納するストリーム列です。</param>
        /// <param name="document">取得元の XML 中間ファイルです。</param>
        public static void GetStreams(
            List<G3dStream> streams, XmlDocument document)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True((streams != null) && (document != null));
            XmlElement stream_array =
                document.SelectSingleNode("//stream_array") as XmlElement;
            if (stream_array == null) { return; }
            stream_array.ParentNode.RemoveChild(stream_array);

            foreach (XmlElement stream in stream_array.SelectNodes("stream"))
            {
                streams.Add(new G3dStream(stream));
            }
        }
    }
}
