﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.meshlib;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib.src.Optimize.Model.PlygonReductionOptimize
{
    public class Shape
    {
        public shapeType OrignalShapeType { get; set; }
        //(SIGLO-28482) 座標変換行列
        public Matrix44 AffineMatrx { get; set; }
        public IDictionary<SourceType, IList<Source>> SourcesDictionary { get; set; } = new Dictionary<SourceType, IList<Source>>();
        public IList<Mesh> Meshes { get; set; } = new List<Mesh>();

        internal static bool ImportSources(
            IDictionary<SourceType, IList<Source>> outSources,
            vertexType vertexType, IList<G3dStream> streams)
        {
            var vtxAttribIndex = 0;
            foreach (var vtxAttrib in vertexType.vtx_attrib_array.Enumerate())
            {
                var stream = streams[vertexType.vtx_attrib_array.vtx_attrib[vtxAttribIndex].stream_index];
                IList<Source> source;
                if (vtxAttrib.hint.StartsWith("position"))
                {
                    if (outSources.ContainsKey(SourceType.Position) == false)
                    {
                        outSources[SourceType.Position] = new List<Source>();
                    }
                    source = outSources[SourceType.Position];
                }
                else if (vtxAttrib.hint.StartsWith("normal"))
                {
                    if (outSources.ContainsKey(SourceType.Normal) == false)
                    {
                        outSources[SourceType.Normal] = new List<Source>();
                    }
                    source = outSources[SourceType.Normal];
                }
                else if (vtxAttrib.hint.StartsWith("color"))
                {
                    if (outSources.ContainsKey(SourceType.Color) == false)
                    {
                        outSources[SourceType.Color] = new List<Source>();
                    }
                    source = outSources[SourceType.Color];
                }
                else if (vtxAttrib.hint.StartsWith("uv"))
                {
                    if (outSources.ContainsKey(SourceType.Texture) == false)
                    {
                        outSources[SourceType.Texture] = new List<Source>();
                    }
                    source = outSources[SourceType.Texture];
                }
                else if (vtxAttrib.hint.StartsWith("binormal"))
                {
                    if (outSources.ContainsKey(SourceType.Binormal) == false)
                    {
                        outSources[SourceType.Binormal] = new List<Source>();
                    }
                    source = outSources[SourceType.Binormal];
                }
                else if (vtxAttrib.hint.StartsWith("tangent"))
                {
                    if (outSources.ContainsKey(SourceType.Tangent) == false)
                    {
                        outSources[SourceType.Tangent] = new List<Source>();
                    }
                    source = outSources[SourceType.Tangent];
                }
                else if (vtxAttrib.hint.StartsWith("blendweight"))
                {
                    if (outSources.ContainsKey(SourceType.BlendWeight) == false)
                    {
                        outSources[SourceType.BlendWeight] = new List<Source>();
                    }
                    source = outSources[SourceType.BlendWeight];
                }
                else
                {
                    switch (stream.type)
                    {
                        case stream_typeType.@float:
                            {
                                if (outSources.ContainsKey(SourceType.EtceteraDouble) == false)
                                {
                                    outSources[SourceType.EtceteraDouble] = new List<Source>();
                                }
                                source = outSources[SourceType.EtceteraDouble];
                            }
                            break;
                        case stream_typeType.@int:
                            {
                                if (outSources.ContainsKey(SourceType.EtceteraInt) == false)
                                {
                                    outSources[SourceType.EtceteraInt] = new List<Source>();
                                }
                                source = outSources[SourceType.EtceteraInt];
                            }
                            break;
                        case stream_typeType.@byte:
                            {
                                if (outSources.ContainsKey(SourceType.EtceteraInt) == false)
                                {
                                    outSources[SourceType.EtceteraInt] = new List<Source>();
                                }
                                source = outSources[SourceType.EtceteraInt];
                            }
                            break;
                        case stream_typeType.@string:
                            throw new ArgumentOutOfRangeException();
                        case stream_typeType.wstring:
                            throw new ArgumentOutOfRangeException();
                        default:
                            throw new ArgumentOutOfRangeException();
                    }
                }

                source.Add(new Source { OriginalStream = stream, OriginalVtxAttribType = vtxAttrib });

                ++vtxAttribIndex;
            }

            return true;
        }

        internal static bool ImportSubMeshes(IList<SubMesh> outSubMeshes, IEnumerable<submeshType> submeshTypes)
        {
            foreach (var submeshType in submeshTypes)
            {
                var subMesh = new SubMesh { IndexCount = submeshType.count, IndexOffset = submeshType.offset, OriginalSubmeshType = submeshType };

                lock (outSubMeshes)
                {
                    outSubMeshes.Add(subMesh);
                }
            }

            return true;
        }

        internal static bool ImportMeshes(IList<PlygonReductionOptimize.Mesh> outMeshes, IEnumerable<meshType> meshTypes,
            IList<G3dStream> streams)
        {
            foreach (var meshType in meshTypes)
            {
                var mesh = new PlygonReductionOptimize.Mesh
                {
                    IndexSource = new Source { OriginalStream = streams[meshType.stream_index] },
                    SourceOffset = 0,
                    IsOriginal = true,
                    OriginalMeshType = meshType,
                };

                if (ImportSubMeshes(mesh.SubMeshes, meshType.submesh_array.submesh) == false)
                {
                    throw ExcepHandle.CreateException("ImportSubMeshes Failed");
                }

                lock (outMeshes)
                {
                    outMeshes.Add(mesh);
                }
            }

            return true;
        }

        public static Shape ConvertFrom(shapeType sourceModel, vertexType sourceVertices, List<G3dStream> sourceStream)
        {
            var vertex = sourceVertices;
            var shape = new Shape { OrignalShapeType = sourceModel, AffineMatrx = Matrix44.Identity };
            {
                if (ImportSources(shape.SourcesDictionary, vertex, sourceStream) == false)
                {
                    throw ExcepHandle.CreateException("ImportOriginalStreams Failed");
                }
            }
            {
                if (ImportMeshes(shape.Meshes, sourceModel.mesh_array.mesh, sourceStream) == false)
                {
                    throw ExcepHandle.CreateException("ImportMeshes Failed");
                }
            }
            return shape;
        }
    }
}
