﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.nw4f_3dif;
using nw4f.meshlib;

namespace nw.g3d.iflib.src.Optimize.Model.PlygonReductionOptimize
{
    public class Model
    {
        public IList<Shape> Shapes { get; set; } = new List<Shape>();
        public Skeleton   Skeleton { get; set; }
        //(SIGLO-28482) FMDB内の骨情報
        public BoneInfo[] BoneInfos { get; set; }

        internal static BoneInfo[] CreateBoneInfo(modelType sourceModel)
        {
            //(SIGLO-28482)
            // BoneInfoを構築しておかないと、Boneから行列は取得できない仕組みになっている
            // IfModelBoneUniteCompressor のCompress 関数より抜粋
            boneType[] bones = sourceModel.skeleton.bone_array.bone;
            int boneCount = bones.Length;
            BoneInfo[] boneInfos = new BoneInfo[boneCount];
            // ボーン情報の初期化
            for (int i = 0; i < boneCount; i++)
            {
                boneInfos[i] = new BoneInfo(bones[i], sourceModel.skeleton.skeleton_info);
            }
            BoneInfo.Setup(boneInfos);

            return boneInfos;
        }

        public static  Model ConvertFrom(modelType sourceModel, List<G3dStream> sourceStream)
        {
            var model = new PlygonReductionOptimize.Model
            {
                Skeleton = new Skeleton
                {
                    SkeletonType = sourceModel.skeleton
                },
                //(SIGLO-28482) 簡略化前の座標変換の為に追加
                BoneInfos = CreateBoneInfo(sourceModel)
            };

            foreach (var shapeType in sourceModel.shape_array.shape)
            {
                var vertex = sourceModel.vertex_array.vertex[shapeType.shape_info.vertex_index];
                var shape = Shape.ConvertFrom(shapeType, vertex, sourceStream);
                // (SIGLO-28482) 2016/06/08 追加
                // 重複（クローンされたインスタンス）オブジェクトのリダクションを行うために
                // アフィン変換行列をかけて、重複のない位置への移動を行う必要がある。
                var boneinfo = model.BoneInfos.FirstOrDefault(o => o.Bone.name == shapeType.shape_info.bone_name);
                if (boneinfo != null)
                {
                    shape.AffineMatrx = ModelCompressUtility.GetBoneTransform(sourceModel, boneinfo);
                }

                // リストに追加されたオブジェクトは、簡略化を実行する
                // その直前にハッシュの計算を行う
                string hash = shapeType.shape_info.original_shape_hash;
                if (string.IsNullOrEmpty(hash))
                {
                    hash = IfOptimizePrimitiveHashCalculator.Calculate(
                        sourceModel, sourceStream, shapeType);
                    shapeType.shape_info.original_shape_hash = hash;
                }

                lock (model.Shapes)
                {
                    model.Shapes.Add(shape);
                }
            }
            return model;
        }
    }
}
