﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    public class WedgeSplitter
    {
        private Mesh mMesh = null;

        public WedgeSplitter(Mesh mesh)
        {
            mMesh = mesh;
        }

        private class SplitData
        {
            public int SplitCount { get; set; }
            public int VertexIndex { get; set; }
            public MeshSourceBase VertexSource { get; set; }
            public List<List<int>> FaceIndicesArray { get; set; }
        }

        /// <summary>
        /// 隣接ポリゴンリストを、ソート済みのプロパティでグループ分けされたリストに変換する
        /// グループインデックスとプロパティインデックスには関連性はない
        /// </summary>
        /// <param name="wv"></param>
        /// <param name="vtm"></param>
        /// <returns></returns>
        private ResizableList<List<Tuple<int, int>>> ConvertToSortedFaceGroupIndeces(WedgeVertex wv, VtxTriMesh vtm)
        {
            var faces = new ResizableList<List<Tuple<int, int>>>();
            faces.Resize(mMesh.Sources.Count - 1, null);
            for (int index = 1; index < mMesh.Sources.Count; index++)
            {
                var meshSourceBase = mMesh.Sources[index];
                faces[index - 1] = new List<Tuple<int, int>>();

                for (int i = 0; i < wv.Props[index - 1].Count; i++)
                {
                    var propId = wv.Props[index - 1][i];
                    var facelist = vtm.GetIncidentTrianglesForProperty(meshSourceBase.Type,
                        meshSourceBase.Name,
                        propId,
                        int.MaxValue,
                        wv.Vertex);

                    // Group　IDと、face ID でまとめる
                    foreach (var faceId in facelist)
                    {
                        faces[index - 1].Add(Tuple.Create(faceId, i));
                    }
                }
                faces[index - 1].Sort();
            }
            return faces;
        }

        /// <summary>
        ///  プロパティを見て同一グループに所属しているポリゴンに分割する。
        ///
        /// 同一グループ内の全ての面のプロパティは一致している状態を作る
        /// </summary>
        /// <param name="sourceList"></param>
        /// <returns></returns>
        private List<List<int>> GetFaceGroup(ResizableList<List<Tuple<int, int>>> sourceList)
        {
            var tmpList = new List<Tuple<int, int>>();
            var baseList = sourceList.First();
            var currentGroupId = 0;

            var isSameGroup = new bool[sourceList.Count];
            var prevGroupIds = new int[sourceList.Count];

            for (int i = 0; i < sourceList.Count; i++)
            {
                isSameGroup[i] = true;
                prevGroupIds[i] = 0;
            }

            var incicentFaceCount = baseList.Count;
            for (int i = 0; i < incicentFaceCount; i++)
            {
                var baseFaceId  = baseList[i].Item1;

                var isAllSameGroup = true;
                for (var j = 0; j < sourceList.Count; j++)
                {
                    var groupId = sourceList[j][i].Item2;
                    if (prevGroupIds[j] != groupId)
                    {
                        isAllSameGroup = false;
                    }
                    prevGroupIds[j] = groupId;
                }

                if (!isAllSameGroup)
                {
                    currentGroupId++;
                }
                tmpList.Add(Tuple.Create(baseFaceId, currentGroupId));
            }
            var result = new List<List<int>>();
            var maxGroupId = tmpList.Max(o => o.Item2) + 1;

            for (var i = 0; i < maxGroupId; i++)
            {
                var faces = tmpList.Where(o => o.Item2 == i).Select(oo => oo.Item1).ToList();
                result.Add(faces);
            }
            return result;
        }

        public void Run()
        {
            // ウェッジを作成します。
            var wedge = new MeshWedges(mMesh);
            wedge.Create();

            var vtxTriMesh = new VtxTriMesh();
            vtxTriMesh.Create(mMesh);

            var splitDatas = new List<SplitData>();
            foreach (var wv in wedge.Wedges)
            {
                // 未使用の頂点はスキップ
                if (wv == null)
                {
                    continue;
                }

                // 共通したプロパティインデックスを持足せることが可能な領域にTraingleFanを分解する
                var faces = ConvertToSortedFaceGroupIndeces(wv, vtxTriMesh);
                var faceGroupList = GetFaceGroup(faces);
                if (faceGroupList.Count == 1)
                {
                    continue;
                }

                // 頂点の分割情報をセットする
                var splitData = new SplitData
                {
                    VertexIndex = wv.Vertex,
                    VertexSource = mMesh.GetSource(SourceType.Position, null),
                    SplitCount = faceGroupList.Count,
                    FaceIndicesArray = faceGroupList
                };
                splitDatas.Add(splitData);
            }

            foreach (var splitData in splitDatas)
            {
                var source = splitData.VertexSource;
                var sourceValueIndex = splitData.VertexIndex;
                for (var splitIndex = 0; splitIndex < splitData.SplitCount; ++splitIndex)
                {
                    // add source
                    var originalSourceValueIndex = sourceValueIndex * source.Stride;
                    var originalSourceValues = new List<object>();
                    for (var strideIndex = 0; strideIndex < source.Stride; ++strideIndex)
                    {
                        originalSourceValues.Add(source.GetRowValue(originalSourceValueIndex + strideIndex));
                    }
                    var newSourceValueIndex = source.Count * source.Stride;
                    source.SetLength(source.Count + 1, false);
                    for (var strideIndex = 0; strideIndex < source.Stride; ++strideIndex)
                    {
                        source.SetRowValue(newSourceValueIndex + strideIndex, originalSourceValues[strideIndex]);
                    }

                    // replace index
                    var faceIndices = splitData.FaceIndicesArray[splitIndex];
                    foreach (var faceIndex in faceIndices)
                    {
                        var faceOffset = mMesh.VOffset[faceIndex];
                        var faceVertexCount = mMesh.VNums[faceIndex];
                        var sourceCount = mMesh.Sources.Count;
                        var sourceIndex = mMesh.Sources.IndexOf(source);

                        var replaced = false;
                        for (var faceVertexIndex = 0; faceVertexIndex < faceVertexCount; ++faceVertexIndex)
                        {
                            var index = mMesh.VBuffer[(faceOffset + faceVertexIndex) * sourceCount + sourceIndex];
                            if (index == sourceValueIndex)
                            {
                                mMesh.VBuffer[(faceOffset + faceVertexIndex) * sourceCount + sourceIndex] = newSourceValueIndex / source.Stride;
                                replaced = true;
                            }
                        }
                        if (replaced == false)
                        {
                            throw ExcepHandle.CreateException("Replacing index is not found");
                        }
                    }
                }
            }

            mMesh.RemoveUnusedVertex();

            Check(mMesh);
        }

        [Conditional("DEBUG")]
        private static void Check(Mesh mesh)
        {
#if false
            // ウェッジを作成します。
            var wedge = new MeshWedges(mesh);
            wedge.Create();

            foreach (var wv in wedge.Wedges)
            {
                // 未使用の頂点はスキップ
                if (wv == null) { continue; }

                // プロパティハードエッジ
                var propertyCount = wv.props.Select(prop => prop.Count).Concat(new[] { 0 }).Max();
                if (propertyCount > 1)
                {
                    throw new Exception("found property hard edge");
                }
            }
#endif
        }
    }
}
