﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw4f.tinymathlib;

namespace nw4f.meshlib
{
    /// <summary>
    /// 各頂点毎のWedge構造体
    /// </summary>
    public class WedgeVertex
    {
        public int Vertex { get; set; }
        public ResizableList<SourceKey> Keys { get; set; } = new ResizableList<SourceKey>();
        public ResizableList<ResizableList<int>> Props { get; set; } = new ResizableList<ResizableList<int>>();

        public int[] GetPropIndex(out int count, SourceType type, string name)
        {
            int n = Keys.Count;
            count = 0;
            if (type == SourceType.Position)
            {
                count = 1;
                return new int[] { Vertex };
            }
            for (int i = 0; i < n; i++)
            {
                if (Keys[i].type == type && Keys[i].name == name)
                {
                    count = Props[i].Count;
                    if (count > 0)
                    {
                        return Props[i].ToArray();
                    }
                    else
                    {
                        return null;
                    }
                }
            }
            return null;
        }

        /// <summary>
        /// 最大のプロパティ数を取得
        /// </summary>
        /// <returns></returns>
        public int GetMaxPropCount()
        {
            int n = Keys.Count;
            int result = 0;
            for (int i = 0; i < n; i++)
            {
                result = Math.Max(Props[i].Count, result);
            }
            return result;
        }

        /// <summary>
        /// インデックスのクリア
        /// </summary>
        public void ClearIndices()
        {
            for (int i = 0; i < Props.Count; i++)
            {
                Props[i].Clear();
            }
        }
    }

    /// <summary>
    /// Wedge構造
    /// </summary>
    public class MeshWedges
    {
        private ResizableList<WedgeVertex> mWList = new ResizableList<WedgeVertex>();
        private Mesh mMesh;

        public ResizableList<WedgeVertex> Wedges
        {
            get { return mWList; }
            set { mWList = value; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MeshWedges(Mesh mesh)
        {
            mMesh = mesh;
        }

        /// <summary>
        /// 必要なメモリ量を計算します
        /// </summary>
        /// <param name="mesh"></param>
        /// <returns></returns>
        public static int GetRequiredMemSize(Mesh mesh)
        {
            int faceN = mesh.FaceN;
            int sn = mesh.Sources.Count;
            int po = mesh.GetSourceOfset(SourceType.Position, null);

            int size = 0;
            for (int f = 0; f < faceN; f++)
            {
                int vn = mesh.mVnums[f];
                int vo = mesh.mVoffs[f];

                for (int v = 0; v < vn; v++)
                {
                    for (int s = 0, si = 0; s < sn; s++)
                    {
                        if (s == po)
                        {
                            continue;
                        }
                        size += mesh.Sources[s].Name.Length;
                        size += 2;
                        si++;
                    }
                }
            }
            return size;
        }

        /// <summary>
        /// 更新処理
        /// </summary>
        /// <param name="vid"></param>
        /// <param name="vtri"></param>
        public void Update(int keep, int kill, VtxTriMesh vtri)
        {
            int sn = mMesh.Sources.Count;
            int po = mMesh.GetSourceOfset(SourceType.Position, null);
            mWList[keep].ClearIndices();

            foreach (var f in vtri.VTL[keep].TriList)
            {
                int vn = mMesh.mVnums[f];
                int vo = mMesh.mVoffs[f];
                for (int v = 0; v < vn; v++)
                {
                    int pid = mMesh.mVbuffs[(vo + v) * sn];

                    if (pid == keep)
                    {
                        for (int s = 0, si = 0; s < sn; s++)
                        {
                            if (s == po) { continue; }

                            int vid = mMesh.mVbuffs[(vo + v) * sn + s];
                            if (!mWList[pid].Props[si].Contains(vid))
                            {
                                mWList[pid].Props[si].Add(vid);
                            }
                            si++;
                        }
                    }
                }
            }
            mWList[kill] = null;
        }

        /// <summary>
        /// Wedgeを作成
        /// </summary>
        public void Create()
        {
            try
            {
                int vtxN = mMesh.VertexN;
                int faceN = mMesh.FaceN;
                int sn = mMesh.Sources.Count;
                int po = mMesh.GetSourceOfset(SourceType.Position, null);

                //初期化
                mWList.Clear();
                mWList.Resize(vtxN, null);

                for (int f = 0; f < faceN; f++)
                {
                    int vn = mMesh.mVnums[f];
                    int vo = mMesh.mVoffs[f];

                    for (int v = 0; v < vn; v++)
                    {
                        int pid = mMesh.mVbuffs[(vo + v) * sn + po];
                        if (mWList[pid] == null)
                        {
                            mWList[pid] = new WedgeVertex();
                            mWList[pid].Vertex = pid;
                            mWList[pid].Props.Resize(sn - 1, null);

                            for (int s = 0, si = 0; s < sn; s++)
                            {
                                if (s == po)
                                {
                                    continue;
                                }
                                int vid = mMesh.mVbuffs[(vo + v) * sn + s];
                                mWList[pid].Keys.Add(new SourceKey(mMesh.Sources[s].Type, mMesh.Sources[s].Name));
                                mWList[pid].Props[si] = new ResizableList<int>();
                                si++;
                            }
                        }
                    }
                }

                //---------------------------------------------
                for (int f = 0; f < faceN; f++)
                {
                    int vn = mMesh.mVnums[f];
                    int vo = mMesh.mVoffs[f];
                    for (int v = 0; v < vn; v++)
                    {
                        int pid = mMesh.mVbuffs[(vo + v) * sn];
                        for (int s = 0, si = 0; s < sn; s++)
                        {
                            if (s == po) { continue; }

                            int vid = mMesh.mVbuffs[(vo + v) * sn + s];

                            if (!mWList[pid].Props[si].Contains(vid))
                            {
                                mWList[pid].Props[si].Add(vid);
                            }
                            si++;
                        }
                    }
                }

                //---------------------------------------------------
                CheckUnusedVertexRef();
            }
            catch
            {
                mWList.Clear();
                throw;
            }
        }

        [Conditional("DEBUG")]
        public void CheckUnusedVertexRef()
        {
            int vtxN = mMesh.VertexN;

            bool nullfind = false;
            for (int vi = 0; vi < vtxN; vi++)
            {
                if (mWList[vi] == null)
                {
                    nullfind = true;
                }
            }
            if (nullfind)
            {
                nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(@"Wedge find unreferenced verteces in mesh object.");
            }
        }
    }
}
