﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;

namespace nw4f.meshlib
{
    /// <summary>
    /// 境界線の種類
    /// </summary>
    public enum QemVertexType
    {
        Inner = 0,
        SubMesh = 0x4000, //サブメッシュ境界上の頂点：不動
        Exclude = 0x02, //除外メッシュ上の頂点：不動　　
        OpenEdge = 0x04, //開放端
        UVEdge = 0x08, //UV境界上の頂点
        BWeightEdge = 0x10, // ブレンドウェイト
        ColEdge = 0x20, //Opaque
        Other = 0x40,
        NmlEdge = 0x80, //Opaque
        FeatEdge = 0x100, //特徴的エッジ上の頂点
        Junction = 0x200, //交差点
        Nonmanifold = 0x400, //不正形状ないし、不動頂点
        Corner = 0x800,     // 角
        UvCorner = 0x1000,     // 角
        ShapeBdr = 0x2000,  // シェイプ間の境界上の頂点
        PropMask = (int)UVEdge | (int)ColEdge | (int)NmlEdge | (int)FeatEdge | (int)ShapeBdr | (int)BWeightEdge,
        TouchMask = (int)OpenEdge | (int)SubMesh | (int)Exclude | (int)Corner | (int)UVEdge | (int)FeatEdge | (int)UvCorner | (int)Junction | (int)Junction,
        ALL = 0xFFFF
    }

    /// <summary>
    /// 簡略化時に考慮するプロパティ、orを取って使う
    /// </summary>
    public enum QemUseProp
    {
        VERT = 0x01, /// 頂点有効
        NOR = 0x02, ///  法線有効
        TEX = 0x04, ///  UV有効
        COL = 0x08, ///  頂点色有効
        BWT = 0x10, /// ブレンドウェイト有効
        BIN = 0x20, /// 従法線有効
        TAN = 0x40, /// 接線有効
        ALL = 0xFF
    }

    /// <summary>
    /// 簡略化時の頂点縮退の方針
    /// </summary>
    public enum QemOptPos
    {
        /// エッジ上の中心
        Center = 0,
        /// エッジのどちらかの頂点、頂点位置は不動。（規定値
        ZeroOne = 1,
        /// エッジ上の最適な位置
        OptimizeEdge = 2,
        /// 計算で最適位置を求める
        CalcOptpos = 3,
        /// 計算で最適位置を求める
        OptimizeArea = 4
    }

    /// <summary>
    /// 簡略化時に優先すること
    /// </summary>
    public enum QemPolicy
    {
        AreaBased = 0, // 面積重視
        AngleBased = 1, // 形重視（正三角形に揃うようにする
        Default = 2, // 特に重み付けはしない（規定値
        AreaAngleBased = 3
    }

    /// <summary>
    /// エッジの種類
    /// </summary>
    internal enum QemEdgeType
    {
        Interior = 0x0000,       //　内部
        PropertyEdge = 0x0001,   //  プロパティのハードエッジ
        HardEdge = 0x0002,       //  ハードエッジそのもの
        TouchedHard = 0x0008,    //ハードエッジに対してTジャンクションなエッジ
        HardEdgeOT = 0x0010,     //ハードエッジに対してTジャンクションなエッジ
        LocalLimitOver = 0x0020, //リミット超え
        NonManifold = 0x8000,    //二多様体化する可能性のあるエッジ（絶対に動かしてはダメ）
        ALL = 0xFFFF
    }

    /// <summary>
    /// Qemを用いたリダクション
    /// </summary>
    public partial class QemSimp
    {
        public class QemSimpParam
        {
            public QemSimpParam()
            {
                PropWeights[(int)SourceType.Normal] = 0.01;
                PropWeights[(int)SourceType.BlendWeight] = 1.0; // ブレンドウェイトの初期値のみは1.0
                TargetRatePerMaterialShape = new Dictionary<string, float>();
            }

            public int DateDim { get; set; } = 0;
            public int PropUseflag { get; set; } = (int)QemUseProp.VERT | (int)QemUseProp.TEX;
            public float TargetRate { get; set; } = 1.0f;
            public Dictionary<string, float> TargetRatePerMaterialShape { get; set; }
            public double DistOffs { get; set; } = 0.0;
            public double Quality { get; set; } = 0.99;
            public double CheckRate { get; set; } = 0.0;
            public double LimitCost { get; set; }

            public double OldFeatureFaceAngle { get; set; } = 180.0;
            public double FeatureFaceAngle { get; set; } = 0.0;
            public double OldLockOpenEdgeAngle { get; set; } = 90.0;
            public double LockOpenEdgeAngle { get; set; } = 0.0;
            public QemPolicy Policy { get; set; } = QemPolicy.Default;
            public bool Smoothing { get; set; } = false;
            public bool CleanUp { get; set; } = true;
            public double OldLockUvHardEdgeAngle { get; set; } = 90.0;
            public double LockUvHardEdgeAngle { get; set; } = 0.0;
            public bool RemoveUnusedVtx { get; set; } = true;
            public bool RecalcTangentBasis { get; set; } = false;
            public bool CheckFlip { get; set; } = true;
            public bool LockSubMeshBoder { get; set; } = true;
            public QemOptPos OptPosPolicy { get; set; } = QemOptPos.OptimizeEdge;
            public double[] PropWeights { get; set; } = Enumerable.Repeat(0.05, Enum.GetNames(typeof(SourceType)).Length).ToArray();

            public double FeatureEdgeLockAngle { get; set; } = 180.0;    ///Opaque
            public double LockNormalHardEdge { get; set; } = 180.0;      ///Opaque
            public double LockColorHardEdge { get; set; } = 180.0;       ///Opaque
            public bool PBdrChkByIndex { get; set; } = false;
            public List<string> ExcludeMeshNames { get; set; } = new List<string>();  /// 除外メッシュリスト

            public QemSimpParam Clone()
            {
                return MemberwiseClone() as QemSimpParam;
            }

            /// <summary>
            /// 連結形状ごとの簡略化率設定を文字列にする
            /// </summary>
            /// <returns></returns>
            private string TargetRatePerConnectedShapeToString()
            {
                string result = string.Empty;

                foreach (var kvp in TargetRatePerMaterialShape)
                {
                    result += kvp.Key;
                    result += "-";
                    result += kvp.Value.ToString(CultureInfo.CurrentCulture);
                }
                return result;
            }

            public override string ToString()
            {
                return (DateDim + ":" + PropUseflag + ":" + TargetRate + ":" + TargetRatePerConnectedShapeToString() + ":" + DistOffs + ":" + Quality +
                       ":" + Policy + ":" + Smoothing + ":" + LockNormalHardEdge + ":" + LockUvHardEdgeAngle + ":" +
                       LockColorHardEdge + ":" + OptPosPolicy).Replace('.', '_').Replace(':', '.');
            }
        }

        public QemSimpParam Params { get; set; }
        public bool ErrorOccured { get; set; }

        /// <summary>
        /// 法線が有効かどうか？
        /// </summary>
        public bool IsNormalEnabled
        {
            get { return (PropFlag & (int)QemUseProp.NOR) != 0; }
        }

        /// <summary>
        /// UV が有効かどうか？
        /// </summary>
        public bool IsUVEnabled
        {
            get { return mPropuse[(int)SourceType.Texture]; }
        }

        /// <summary>
        /// ブレンドウェイトが有効かどうか？
        /// </summary>
        public bool IsBlendWeightEnabled
        {
            get { return ((PropFlag & (int)QemUseProp.BWT) != 0); }
        }

        /// <summary>
        /// プロパティ毎の重み
        /// </summary>
        public double[] PropWaits
        {
            get { return Params.PropWeights; }
        }

        /// <summary>
        /// UVに対する重み
        /// </summary>
        public double UvWeight
        {
            get { return Params.PropWeights[(int)SourceType.Texture]; }
            set
            {
                Params.PropWeights[(int)SourceType.Texture] = value > 0.0 ? Math.Sqrt(value) : 0.0;
            }
        }

        /// <summary>
        /// 法線に対する重み
        /// </summary>
        public double NormalWeight
        {
            get { return Params.PropWeights[(int)SourceType.Normal]; }
            set
            {
                Params.PropWeights[(int)SourceType.Normal] = value > 0.0 ? Math.Sqrt(value) : 0.0;
            }
        }

        /// <summary>
        /// カラーに対する重み
        /// </summary>
        public double ColorWeight
        {
            get { return Params.PropWeights[(int)SourceType.Color]; }
            set
            {
                Params.PropWeights[(int)SourceType.Color] = value > 0.0 ? Math.Sqrt(value) : 0.0;
            }
        }

        /// <summary>
        /// カラーに対する重み
        /// </summary>
        public double AnimBlendWeight
        {
            get { return Params.PropWeights[(int)SourceType.BlendWeight]; }
            set
            {
                Params.PropWeights[(int)SourceType.BlendWeight] = value > 0.0 ? Math.Sqrt(value) : 0.0;
            }
        }

        /// <summary>
        /// Quadric の重みを計算する必要があるかどうか？
        /// </summary>
        public bool IsRequiredComputingWeight
        {
            get { return (Policy != QemPolicy.Default || IsBlendWeightEnabled); }
        }

        /// <summary>
        /// サブメッシュの境界を保護します
        /// （規定値false）
        /// </summary>
        public bool LockSubMeshBoder
        {
            get { return Params.LockSubMeshBoder; }
            set { Params.LockSubMeshBoder = value; }
        }

        /// <summary>
        /// フェースの裏返りを防ぎます
        /// （規定値false）
        /// </summary>
        public bool CheckFlip
        {
            get { return Params.CheckFlip; }
            set { Params.CheckFlip = value; }
        }

        /// <summary>
        /// 簡略化の後に、接空間の基底を再計算するか？
        /// （規定値false）
        /// </summary>
        public bool RecalcTangentBasis
        {
            get { return Params.RecalcTangentBasis; }
            set { Params.RecalcTangentBasis = value; }
        }

        /// <summary>
        /// 簡略化の後に、未使用の頂点を頂点ストリームから消すかどうか？
        /// （規定値false）
        /// </summary>
        public bool RemoveUnUsedVertex
        {
            get { return Params.RemoveUnusedVtx; }
            set { Params.RemoveUnusedVtx = value; }
        }

        /// <summary>
        /// UVのハードエッジを保護します
        /// （規定値false）
        /// </summary>
        public double LockUvHardEdgeAngle
        {
            get { return Params.LockUvHardEdgeAngle; }
            set
            {
                Params.LockUvHardEdgeAngle = value;
                Params.LockUvHardEdgeAngle = Math.Max(0, Params.LockUvHardEdgeAngle);
                Params.LockUvHardEdgeAngle = Math.Min(180, Params.LockUvHardEdgeAngle);
            }
        }

        /// <summary>
        /// UVのハードエッジを保護します
        /// （規定値false）
        /// </summary>
        public double OldLockUvHardEdgeAngle
        {
            get { return Params.OldLockUvHardEdgeAngle; }
            set
            {
                Params.OldLockUvHardEdgeAngle = value;
                Params.OldLockUvHardEdgeAngle = Math.Max(0, Params.OldLockUvHardEdgeAngle);
                Params.OldLockUvHardEdgeAngle = Math.Min(180, Params.OldLockUvHardEdgeAngle);
            }
        }

        /// <summary>
        /// 頂点カラーのハードエッジを保護します
        /// （規定値false）
        /// </summary>
        public double LockColorHardEdge
        {
            get { return Params.LockColorHardEdge; }
            set { Params.LockColorHardEdge = value; }
        }

        /// <summary>
        /// 法線同士のハードエッジ
        /// </summary>
        public double LockNormalHardEdge
        {
            get { return Params.LockNormalHardEdge; }
            set
            {
                Params.LockNormalHardEdge = value;
            }
        }
        /// <summary>
        /// 頂点縮退を行う方法
        /// （規定値:ZeroOne
        /// </summary>
        public QemOptPos OptPosPolicy
        {
            get { return Params.OptPosPolicy; }
            set
            {
                Params.OptPosPolicy = value;
            }
        }

        /// <summary>
        /// 考慮するプロパティの種類
        /// （規定値:考慮するプロパティの種類
        /// </summary>
        public QemPolicy Policy
        {
            get { return Params.Policy; }
            set
            {
                if ((int)value < Enum.GetNames(typeof(QemPolicy)).Length)
                {
                    Params.Policy = value;
                }
            }
        }

        /// <summary>
        /// 高さオフセット、入れることで平面でのリダクションが良くなります。
        /// 凹凸のないオブジェクトのリダクション時に有効です。
        /// </summary>
        public double DistOffset
        {
            get { return Params.DistOffs; }
            set
            {
                if (value >= 0.0f)
                {
                    Params.DistOffs = value;
                }
                else
                {
                    throw ExcepHandle.CreateException("The value of dist offset is out of range. Legal range : [0.0,double_max)");
                }
            }
        }

        /// <summary>
        /// 特徴的な稜線になってもいい角度
        /// </summary>
        public double FeatureFaceAngle
        {
            get { return Params.FeatureFaceAngle; }
            set
            {
                Params.FeatureFaceAngle = value;
                Params.FeatureEdgeLockAngle = Math.Max(0, Params.FeatureEdgeLockAngle);
                Params.FeatureEdgeLockAngle = Math.Min(180, Params.FeatureEdgeLockAngle);
            }
        }

        /// <summary>
        /// 特徴的な稜線になってもいい角度
        /// </summary>
        public double OldFeatureFaceAngle
        {
            get { return Params.OldFeatureFaceAngle; }
            set
            {
                Params.OldFeatureFaceAngle = value;
                Params.OldFeatureFaceAngle = Math.Max(0, Params.OldFeatureFaceAngle);
                Params.OldFeatureFaceAngle = Math.Min(90, Params.OldFeatureFaceAngle);
            }
        }

        /// <summary>
        /// 連続したOpenEdgeの間の角度
        /// </summary>
        public double FeatureEdgeLockAngle
        {
            get { return Params.FeatureEdgeLockAngle; }
            set
            {
                Params.FeatureEdgeLockAngle = value;
                Params.FeatureEdgeLockAngle = Math.Max(0, Params.FeatureEdgeLockAngle);
                Params.FeatureEdgeLockAngle = Math.Min(180, Params.FeatureEdgeLockAngle);
            }
        }

        /// <summary>
        /// 一回の操作で変動可能な体積の割合を、大雑把に設定する
        /// </summary>
        public double Quality
        {
            get { return Params.Quality; }
            set
            {
                if (value <= 1.0f && value >= 0.0f)
                {
                    Params.Quality = value;
                }
                else
                {
                    throw ExcepHandle.CreateException("The value of quality is out of range. Legal range :[0.0,1.0]");
                }
            }
        }

        /// <summary>
        /// 一回の操作で変動可能な体積の割合を、大雑把に設定する
        /// </summary>
        public double MovePosCheckRate
        {
            get { return Params.CheckRate; }
            set
            {
                if (value <= 1.0f && value >= 0.0f)
                {
                    Params.CheckRate = value;
                }
                else
                {
                    throw ExcepHandle.CreateException("The value of check rate is out of range. Legal range : [0.0,1.0]");
                }
            }
        }

        /// <summary>
        /// 連続したOpenEdgeの間の角度
        /// </summary>
        public double OldLockOpenEdgeAngle
        {
            get { return Params.OldLockOpenEdgeAngle; }
            set
            {
                Params.OldLockOpenEdgeAngle = value;
                Params.OldLockOpenEdgeAngle = Math.Max(0, Params.OldLockOpenEdgeAngle);
                Params.OldLockOpenEdgeAngle = Math.Min(180, Params.OldLockOpenEdgeAngle);
            }
        }

        /// <summary>
        /// 連続したOpenEdgeの間の角度
        /// </summary>
        public double LockOpenEdgeAngle
        {
            get { return Params.LockOpenEdgeAngle; }
            set
            {
                Params.LockOpenEdgeAngle = value;
                Params.LockOpenEdgeAngle = Math.Max(0, Params.LockOpenEdgeAngle);
                Params.LockOpenEdgeAngle = Math.Min(180, Params.LockOpenEdgeAngle);
            }
        }

        /// <summary>
        /// 結果に対して、スムース化を行いたい場合
        /// </summary>
        public bool Smoothing
        {
            get { return Params.Smoothing; }
            set { Params.Smoothing = value; }
        }

        /// <summary>
        /// 結果に対して、スムース化を行いたい場合
        /// </summary>
        private bool CleanUp
        {
            get { return Params.CleanUp; }
            set { Params.CleanUp = value; }
        }

        /// <summary>
        /// データの次元数を取得
        /// </summary>
        public int DataDimension
        {
            get { return Params.DateDim; }
            set { Params.DateDim = value; }
        }

        /// <summary>
        /// どのプロパティを使ってリダクションするかを取得/設定
        /// </summary>
        public int PropFlag
        {
            get { return Params.PropUseflag; }
            set { Params.PropUseflag = (value & (int)QemUseProp.ALL); }
        }

        /// <summary>
        /// どのプロパティが使われているか？のフラグ
        /// </summary>
        private ResizableList<bool> PropUse
        {
            get { return mPropuse; }
        }

        /// <summary>
        /// ソースメッシュを取得
        /// </summary>
        public Mesh SourceMesh
        {
            get { return mMesh; }
        }

        /// <summary>
        /// 目標となる元のポリゴンに対する簡略化後のポリゴン数の割合。
        /// メッシュ全体に対しての設定値。
        /// </summary>
        public float TargetRate
        {
            get { return Params.TargetRate; }
            set
            {
                if (value <= 1.0f && value >= 0.0f)
                {
                    Params.TargetRate = value;
                }
                else
                {
                    throw ExcepHandle.CreateException("The value of parameter is out of range. Legal range : [0.0,1.0].");
                }
            }
        }

        /// <summary>
        /// 目標となる元のポリゴンに対する簡略化後のポリゴン数の割合。
        /// 連結形状ごとの設定値。
        /// </summary>
        public Dictionary<string, float> TargetRatePerMaterialShape
        {
            get { return Params.TargetRatePerMaterialShape;  }
            set { Params.TargetRatePerMaterialShape = value; }
        }

        /// <summary>
        /// 一回の操作で変動可能な体積の割合を、大雑把に設定する
        /// シェイプごとに相対的に設定
        /// </summary>
        internal double LimitCost
        {
            get { return Params.LimitCost; }
            set
            {
                Params.LimitCost = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="code"></param>
        /// <param name="type"></param>
        /// <returns></returns>
        internal bool IsEdgeType(int code, QemEdgeType type)
        {
            if ((code & (int)type) != 0) { return true; }
            return false;
        }

        /// <summary>
        /// プロパティの境界を距離によって判定する( UV, Color)
        /// 距離は、Mesh.minimalDistForPropで判定する
        /// </summary>
        public bool PropBorderCheckByIndex
        {
            get { return Params.PBdrChkByIndex; }
            set { Params.PBdrChkByIndex = value; }
        }

        /// <summary>
        /// 簡略化の除外対象になるメッシュの名前リスト
        /// </summary>
        public List<string> ExcludeMeshNames
        {
            get { return Params.ExcludeMeshNames; }
            set { Params.ExcludeMeshNames = value; }
        }
    }
}
