﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography;
using Vector3 = nw4f.tinymathlib.Vector3;

namespace nw4f.meshlib
{
    /// <summary>
    /// メッシュのバウンディングボックスの再構成を行う
    /// ソースメッシュは、頂点結合が完了してる前提です。
    /// </summary>
    public class ConnectedMeshDecomposer
    {
        public class FaceInfo
        {
            public int MeshIndex { get; set; } = -1;
            public int ShapeIndex { get; set; } = -1;
            public int FaceIndex { get; set; } = -1;
        }

        public Mesh SourceMesh { get; set; }
        public double TotalVolume { get; private set; }
        public double AverageVolume { get; private set; }
        public Dictionary<int, double> Volumes { get; private set; }

        public List<List<int>> ConnectedFacesList { get; private set; }

        public ConnectedMeshDecomposer(Mesh mesh)
        {
            SourceMesh = mesh;
        }

        /// <summary>
        /// 連結形状分割処理を実行する。
        /// </summary>
        public void Execute()
        {
            // DecomposeIntoConnectedFaces() からの
            // var activeShapeID = GetActiveShapeId(fgid,faceId) の様なイメージ
            DecomposeIntoConnectedFaces();
            ComputeActiveShapeVolume();
        }

        /// <summary>
        /// 連結のあるフェースに分割を行います。
        /// </summary>
        private void DecomposeIntoConnectedFaces()
        {
            // WingedEdge によるトポロジー走査処理
            MeshSourceBase source = SourceMesh.GetSource(SourceType.Position, null);
            var wingedEdges = new WingedEdgeMesh();
            wingedEdges.SetRepresentativeKey(source.Key.type, source.Key.name);
            wingedEdges.CreateAll(SourceMesh);
            wingedEdges.CreateFaceEdgeList();
            // エッジ間参照を作成する
            wingedEdges.BuildEdgeReference();

            var exploredList = new ResizableList<bool>();
            exploredList.Resize(SourceMesh.FaceN, false);
            ConnectedFacesList = new List<List<int>>();

            // 未踏の面が消えるまで処理を継続する
            while (exploredList.Any(o => o == false))
            {
                var queue = new Queue<int>();
                var unexploredIndex = exploredList.FindIndex(o => o == false);
                var newConnectedFaces = new List<int>();
                queue.Enqueue(unexploredIndex);

                while (queue.Count > 0)
                {
                    AddIntoConnectedGroup(queue, wingedEdges, exploredList, newConnectedFaces, queue.Dequeue());
                }
                ConnectedFacesList.Add(newConnectedFaces);
            }

            // メッシュ本体に連結面リスト情報を構築する
            SourceMesh.FaceConnectedShapeID.Resize(SourceMesh.FaceN, 0);
            SourceMesh.ConnectedShapeInfomationList.Resize(ConnectedFacesList.Count, null);
            foreach (var connectedFaces in ConnectedFacesList)
            {
                var index = ConnectedFacesList.IndexOf(connectedFaces);
                SourceMesh.ConnectedShapeInfomationList[index] = new Mesh.ConnectedShapeInfo
                {
                    Index = index,
                    MaterialShapeIndeces = new ResizableList<int>()
                };
                foreach (var face in connectedFaces)
                {
                    SourceMesh.FaceConnectedShapeID[face] = index;
                }
            }

            // MaterialShapeとConnectedShapeの間をつなぐ
            for (var i = 0; i < SourceMesh.FaceN; i++)
            {
                var csId = SourceMesh.FaceConnectedShapeID[i];
                var msId = SourceMesh.FaceMaterialShapeId[i];
                SourceMesh.ConnectedShapeInfomationList[csId].ShapeName = string.Empty;
                if (!SourceMesh.ConnectedShapeInfomationList[csId].MaterialShapeIndeces.Contains(msId))
                {
                    SourceMesh.ConnectedShapeInfomationList[csId].MaterialShapeIndeces.Add(msId);
                }
            }
        }

        /// <summary>
        /// 再帰関数。
        /// 自分自身を連結FaceListに加えて、隣接Faceに処理を引き継ぐ。
        /// 隣接面がなくなるか、全てがexploredになるまで処理を継続
        /// </summary>
        /// <param name="wem"></param>
        /// <param name="visited"></param>
        /// <param name="connectedFaces"></param>
        /// <param name="faceId"></param>
        private void AddIntoConnectedGroup(Queue<int>  queue, WingedEdgeMesh wem, ResizableList<bool> exploredList, List<int> connectedFaces, int faceId)
        {
            if (faceId == int.MaxValue || exploredList[faceId])
            {
                return;
            }

            var faceEdges = wem.RepFEL[faceId];
            exploredList[faceId] = true;
            connectedFaces.Add(faceId);

            foreach (var faceEdge in faceEdges.Edges)
            {
                int oppositFace = faceEdge.Face0 == faceId ? faceEdge.Face1 : faceEdge.Face0;
                queue.Enqueue(oppositFace);
            }
        }

        /// <summary>
        /// メッシュ内のポリゴン全体をバウンディングボックスに突っ込む
        /// </summary>
        private double ComputePartialMeshVolume(Vector3 p1, Vector3 p2, Vector3 p3)
        {
            double result = 0.0;
            result = -p3.x * p2.y * p1.z + p2.x * p3.y * p1.z + p3.x * p1.y * p2.z
                     - p1.x * p3.y * p2.z - p2.x * p1.y * p3.z + p1.x * p2.y * p3.z;
            result = result / 6.0;
            return result;
        }

        /// <summary>
        /// 有効な体積を計算する
        /// </summary>
        private void ComputeActiveShapeVolume()
        {
            // 現在はActive Shape = Shape
            var shapeN = SourceMesh.ConnectedShapeInfomationList.Count;
            Volumes = new Dictionary<int, double>();
            for (var i = 0; i < shapeN; i++)
            {
                Volumes.Add(SourceMesh.ConnectedShapeInfomationList[i].Index, 0.0);
            }
            var mesh = SourceMesh;
            var posIndices = new int[3];
            var positions = new Vector3[3] { new Vector3(), new Vector3(), new Vector3() };
            var faceN = mesh.FaceN;

            for (var i = 0; i < faceN; i++)
            {
                var csid = mesh.FaceConnectedShapeID[i];
                mesh.GetFaceVtxIndices(i, ref posIndices);
                for (var vi = 0; vi < 3; vi++)
                {
                    positions[vi].SetZero();
                    mesh.GetVertexPos(posIndices[vi], ref positions[vi]);
                }
                // ポリゴンから計算するtetraの体積を合算していく
                Volumes[csid] += ComputePartialMeshVolume(positions[0], positions[1], positions[2]);
            }
            //裏返った面が存在する場合など、近似値の体積が負の場合は反転
            for (int i = 0; i < shapeN; i++)
            {
                Volumes[i] = Math.Abs(Volumes[i]);
            }
            TotalVolume = Volumes.Values.Sum();
            AverageVolume = Volumes.Values.Average();
        }

        /// <summary>
        /// 指定のフェースが所属するグループの有効な体積を取得
        /// </summary>
        /// <param name="faceId"></param>
        /// <returns></returns>
        public double GetActiveVolume(int faceId)
        {
            return Volumes[ SourceMesh.FaceConnectedShapeID[ faceId]];
        }

        public double GetActiveVolumeRatio(int faceId)
        {
            return Volumes[SourceMesh.FaceConnectedShapeID[ faceId]] / AverageVolume;
        }
    }
}
