﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using nw.g3d.iflib.src.Optimize.Model.PlygonReductionOptimize;
using nw4f.meshlib;
using Microsoft.Scripting.Utils;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib.src.Optimize.Model
{
    public class IfModelPolygonReductionOptimizer : IfModelOptimizer
    {
        public class Options
        {
            // このクラスを変更した場合は、この数字を増やしてください
            private const int OptionRevision = 1;

            public Dictionary<string, float> TargetRatePerShapes { get; set; }
            public double DistOffset { get; set; }
            public bool Smoothing { get; set; }
            public int PropFlag { get; set; }
            public QemPolicy Policy { get; set; }
            public double Quality { get; set; }
            public QemOptPos OptPosPolicy { get; set; }
            public double LockOpenEdgeAngle { get; set; }
            public double OldLockOpenEdgeAngle { get; set; }
            public double LockUvHardEdgeAngle { get; set; }
            public double OldLockUvHardEdgeAngle { get; set; }
            public double FeatureFaceAngle { get; set; }
            public double OldFeatureFaceAngle { get; set; }
            public double UvWeight { get; set; }
            public double NormalWeight { get; set; }
            public double AnimBlendWeight { get; set; }
            public double ClosedVertexThreshold { get; set; }
            public List<string> ExcludeMeshNames { get; private set; }

            public bool IsMargeShape { get; set; }
            public SortedDictionary<int, float> TargetRates { get; set; }

            public Options()
            {
                var qemSimp = new QemSimp();
                TargetRatePerShapes = qemSimp.TargetRatePerMaterialShape;
                DistOffset = qemSimp.DistOffset;
                Smoothing = qemSimp.Smoothing;
                LockUvHardEdgeAngle = qemSimp.LockUvHardEdgeAngle;
                OldLockUvHardEdgeAngle = qemSimp.OldLockUvHardEdgeAngle;
                PropFlag = qemSimp.PropFlag;
                Policy = qemSimp.Policy;
                Quality = qemSimp.Quality;
                OptPosPolicy = qemSimp.OptPosPolicy;
                LockOpenEdgeAngle = qemSimp.LockOpenEdgeAngle;
                OldLockOpenEdgeAngle = qemSimp.OldLockOpenEdgeAngle;
                FeatureFaceAngle = qemSimp.FeatureFaceAngle;
                OldFeatureFaceAngle = qemSimp.OldFeatureFaceAngle;
                UvWeight = qemSimp.UvWeight;
                NormalWeight = qemSimp.NormalWeight;
                AnimBlendWeight = qemSimp.AnimBlendWeight;
                IsMargeShape = true;
                ClosedVertexThreshold = ShapeMerger.Percentage;
                ExcludeMeshNames = new List<string>();
            }

            public override string ToString()
            {
                var excludeList = ExcludeMeshNames.Any() ? ExcludeMeshNames.Aggregate((o, r) => r + "," + o) : string.Empty;
                // 旧・角度パラメータは不要になり次第削除
                var returnString =
                    OptionRevision + "," +
                    DistOffset + "," +
                    Smoothing + "," +
                    LockUvHardEdgeAngle + "," +
                    OldLockUvHardEdgeAngle + "," +
                    PropFlag + "," +
                    Policy + "," +
                    Quality + "," +
                    OptPosPolicy + "," +
                    LockOpenEdgeAngle + "," +
                    OldLockOpenEdgeAngle + "," +
                    FeatureFaceAngle + "," +
                    OldFeatureFaceAngle + "," +
                    UvWeight + "," +
                    NormalWeight + "," +
                    AnimBlendWeight + "," +
                    ClosedVertexThreshold + "," +
                    IsMargeShape + "," +
                    excludeList;

                foreach (var targetRate in TargetRates)
                {
                    returnString += "," + targetRate;
                }

                foreach (var targetRate in TargetRatePerShapes)
                {
                    returnString += "," + targetRate;
                }
                return returnString;
            }

            /// <summary>
            /// 簡略化にどのプロパティを使うかを設定する
            /// </summary>
            /// <param name="flagString"> 簡略化に使用するプロパティを格納した文字列</param>
            public void SetPropertyFlags(string[] flagStrings)
            {
                var tmpFlags = new List<string>();
                if (flagStrings.Length == 1)
                {
                    int tmpVal = 0;
                    if (int.TryParse(flagStrings[0], out tmpVal))
                    {
                        PropFlag = tmpVal;
                        return;
                    }
                    tmpFlags = flagStrings[0].Split(new[] { '|' }).ToList();
                }

                // パラメーターを単一文字列に分解
                foreach (var flag in flagStrings)
                {
                    if (flag.Contains('|'))
                    {
                        tmpFlags.AddRange(flag.Split(new[] { '|' }));
                    }
                    else
                    {
                        tmpFlags.Add(flag);
                    }
                }

                // 設定する場合は規定値を上書きする
                PropFlag = 0;
                foreach (var flag in tmpFlags)
                {
                    if (flag == "Position")
                    {
                        PropFlag |= (int)QemUseProp.VERT;
                    }
                    else if (flag == "Normal")
                    {
                        PropFlag |= (int)QemUseProp.NOR;
                    }
                    else if (flag == "UV")
                    {
                        PropFlag |= (int)QemUseProp.TEX;
                    }
                    else if (flag == "BlendWeight")
                    {
                        PropFlag |= (int)QemUseProp.BWT;
                    }
                    else if (flag == "All")
                    {
                        PropFlag |= (int)QemUseProp.ALL;
                    }
                    else
                    {
                        _3dIntermediateFilePlygonReduction.IfStrings.Throw("PolygonReduction_IllegalParam_Error",
                            "property-flag", flag);
                    }
                }
            }

            /// <summary>
            /// 簡略化後の頂点位置の設定を行う
            /// </summary>
            /// <param name="flag">簡略化を行う場合の最適化位置決定パラメーターを指定する文字列</param>
            public void SetOptPosPolicy(string flag)
            {
                int tmpVal;
                if (int.TryParse(flag, out tmpVal))
                {
                    OptPosPolicy = (QemOptPos)tmpVal;
                    return;
                }

                OptPosPolicy = QemOptPos.OptimizeEdge;
                if (flag == "OptimalOnSpace")
                {
                    OptPosPolicy = QemOptPos.CalcOptpos;
                }
                else if (flag == "OptimalOnEdge")
                {
                    OptPosPolicy = QemOptPos.OptimizeEdge;
                }
                else if (flag == "OptimalOnFace")
                {
                    OptPosPolicy = QemOptPos.OptimizeArea;
                }
                else if (flag == "SharedPosition")
                {
                    OptPosPolicy = QemOptPos.ZeroOne;
                }
                else
                {
                    _3dIntermediateFilePlygonReduction.IfStrings.Throw("PolygonReduction_IllegalParam_Error",
                        "optimize-position-policy", flag);
                }
            }

            /// <summary>
            /// 簡略化時に考慮する重み（面積、ポリゴンの形）を指定する
            /// </summary>
            /// <param name="flag">頂点位置決定する際に、重視すべき項目（面積、ポリゴンの形）を指定する文字列</param>
            public void SetPolicy(string flag)
            {
                int tmpVal;
                if (int.TryParse(flag, out tmpVal))
                {
                    Policy = (QemPolicy)tmpVal;
                    return;
                }
                Policy = QemPolicy.Default;
                if (flag == "AreaBased")
                {
                    Policy = QemPolicy.AreaBased;
                }
                else if (flag == "AngleBased")
                {
                    Policy = QemPolicy.AngleBased;
                }
                else if (flag == "AreaAngleBased")
                {
                    Policy = QemPolicy.AreaAngleBased;
                }
                else if (flag == "Default")
                {
                    Policy = QemPolicy.Default;
                }
                else
                {
                    _3dIntermediateFilePlygonReduction.IfStrings.Throw("PolygonReduction_IllegalParam_Error",
                        "policy", flag);
                }
            }

            /// <summary>
            /// 除外Shapeリストを設定します
            /// </summary>
            /// <param name="shapes"></param>
            public void SetExcludeShapeList(string[] shapeNames)
            {
                ExcludeMeshNames.Clear();

                var tmpList = new List<string>();
                // パラメーターを単一文字列に分解してリスト化する
                foreach (var shapeName in shapeNames)
                {
                    if (shapeName.Contains(','))
                    {
                        tmpList.AddRange(shapeName.Split(new[] { ',' }));
                    }
                    else
                    {
                        tmpList.Add(shapeName);
                    }
                }

                foreach (var excludeMeshName in tmpList)
                {
                    // ワイルドカードとクエスチョンを置換する
                    var regexPattern = System.Text.RegularExpressions.Regex.Replace(
                        excludeMeshName,
                        ".",
                        o =>
                        {
                            //? -> 任意の1文字を示す正規表現(.)
                            if (o.Value.Equals("?"))
                            {
                                return ".";
                            }
                            //*->0文字以上の任意の文字列を示す正規表現(.*)
                            else if (o.Value.Equals("*"))
                            {
                                return ".*";
                            }
                            else
                            {
                                return System.Text.RegularExpressions.Regex.Escape(o.Value);
                            }
                        });
                    // 除外リストに追加
                    ExcludeMeshNames.Add(regexPattern);
                }
            }

            /// <summary>
            /// マテリアルシェイプごとに簡略化率を設定する
            /// </summary>
            /// <param name="optionValues"></param>
            public void SetRateForMaterialShapes(string[] optionValues)
            {
                TargetRatePerShapes.Clear();

                var shapeRateStrList = new List<string>();
                // パラメーターを単一文字列に分解してリスト化する
                foreach (var optionValue in optionValues)
                {
                    if (optionValue.Contains(','))
                    {
                        shapeRateStrList.AddRange(optionValue.Split(new[] { ',' }));
                    }
                    else
                    {
                        shapeRateStrList.Add(optionValue);
                    }
                }

                var tmpShapeRates = new Dictionary<string, float>();
                foreach (var shapeRateStr in shapeRateStrList)
                {
                    if (!shapeRateStr.Contains(':'))
                    {
                        _3dIntermediateFilePlygonReduction.IfStrings.Throw("PolygonReduction_InvalidOptionValueFormat", shapeRateStr);
                    }

                    var nameAndRate = shapeRateStr.Split(new[] { ':' });
                    if (nameAndRate.Count() != 2)
                    {
                        _3dIntermediateFilePlygonReduction.IfStrings.Throw("PolygonReduction_InvalidOptionValueFormat", shapeRateStr);
                    }

                    string shapeName = nameAndRate[0].Trim();
                    string rateStr = nameAndRate[1].Trim();
                    double rate;
                    if (!double.TryParse(rateStr, out rate))
                    {
                        _3dIntermediateFilePlygonReduction.IfStrings.Throw("PolygonReduction_InvalidOptionValueFormat", shapeRateStr);
                    }

                    tmpShapeRates.Add(shapeName, (float)rate);
                }

                foreach (var kvp in tmpShapeRates)
                {
                    // ワイルドカードとクエスチョンを置換する
                    var regexPattern = System.Text.RegularExpressions.Regex.Replace(
                        kvp.Key,
                        ".",
                        o =>
                        {
                            //? -> 任意の1文字を示す正規表現(.)
                            if (o.Value.Equals("?"))
                            {
                                return ".";
                            }
                            //*->0文字以上の任意の文字列を示す正規表現(.*)
                            else if (o.Value.Equals("*"))
                            {
                                return ".*";
                            }
                            else
                            {
                                return System.Text.RegularExpressions.Regex.Escape(o.Value);
                            }
                        });
                    // 除外リストに追加
                    TargetRatePerShapes.Add(regexPattern, kvp.Value);
                }
            }
        }

        /// <summary>
        /// G3dParallel タスクコンテキストをログ整理のために拡張
        /// </summary>
        private class MessageWriter : G3dParallel.TaskContext
        {
            /// <summary>
            /// メッセージの種類
            /// </summary>
            private enum MessageType
            {
                Log,    // ログ
                Error,  // エラー
            }

            private readonly List<Tuple<MessageType, string>> message_ = new List<Tuple<MessageType, string>>();
            private readonly Action<string> output_;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="output">ログ出力先</param>
            public MessageWriter(Action<string> output)
                : base(null)
            {
                output_ = output;
            }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="parent">親コンテキスト</param>
            private MessageWriter(MessageWriter parent)
                : base(parent)
            {
                // ログ出力は親 (ルート) コンテキストが行うので、ここでは出力しない。
                output_ = null;
            }

            /// <summary>
            /// リソース解放
            /// </summary>
            public override void Dispose()
            {
                var parent = Parent as MessageWriter;
                if (parent != null)
                {
                    // 自身が溜め込んだメッセージを親に渡す。
                    lock (this)
                    {
                        parent.Write(message_);
                    }
                }
                else if (output_ != null)
                {
                    output_(Message);
                }
            }

            /// <summary>
            /// 新規タスク用の子コンテキストを作成する
            /// </summary>
            /// <returns></returns>
            public override G3dParallel.TaskContext CreateChildContext()
            {
                return new MessageWriter(this);
            }

            /// <summary>
            /// メッセージを文字列として取得する
            /// </summary>
            public string Message
            {
                get
                {
                    lock (this)
                    {
                        return message_.Select(x => x.Item2).Aggregate(string.Concat);
                    }
                }
            }

            /// <summary>
            /// ログを文字列として取得する
            /// </summary>
            public string Log
            {
                get
                {
                    lock (this)
                    {
                        return message_.Where(x => x.Item1 == MessageType.Log).Select(y => y.Item2).Aggregate(string.Concat);
                    }
                }
            }

            /// <summary>
            /// エラーを文字列として取得する
            /// </summary>
            public string Error
            {
                get
                {
                    lock (this)
                    {
                        return message_.Where(x => x.Item1 == MessageType.Error).Select(y => y.Item2).Aggregate(string.Concat);
                    }
                }
            }

            public void WriteLog(string msg)
            {
                Write(MessageType.Log, msg);
            }

            public void WriteLogLine(string msg)
            {
                WriteLog(msg + System.Environment.NewLine);
            }

            public void WriteError(string msg)
            {
                Write(MessageType.Error, msg);
            }

            public void WriteErrorLine(string msg)
            {
                WriteError(msg + System.Environment.NewLine);
            }

            private void Write(MessageType type, string msg)
            {
                lock (this)
                {
                    message_.Add(Tuple.Create(type, msg));
                }
            }

            private void Write(IEnumerable<Tuple<MessageType, string>> msg)
            {
                lock (this)
                {
                    message_.AddRange(msg.ToArray());
                }
            }
        }

        private static readonly StringBuilder LogBuilder = new StringBuilder();
        private readonly Options _options;
        private readonly SortedDictionary<int, FaceCount> _reduceFaceCounts = new SortedDictionary<int, FaceCount>();
        private readonly List<string> _resultMessages = new List<string>();

        private struct FaceCount
        {
            public int Before { get; set; }
            public int After { get; set; }
            public bool IsMarged { get; set; }
            public List<string> MargeShapeNames { get; set; }
        }

        private class AddStream
        {
            public shapeType ShapeType { get; set; }
            public meshType MeshType { get; set; }
            public G3dStream Stream { get; set; }
        }

        // コンストラクタ
        public IfModelPolygonReductionOptimizer(string argument, Options options) :
            base("IfModelPlygonRedyctionOptimizer_Log", argument)
        {
            _options = options;
            EnableProcessLog = true;
        }

        public override string Process
        {
            get { return "polygon_reduction"; }
        }

        public override string GetResult()
        {
            var resultStr = string.Empty;
            foreach (var reduceNumber in _reduceFaceCounts)
            {
                resultStr += reduceNumber.Key + "[" + reduceNumber.Value.After + "/" + reduceNumber.Value.Before + "]" + " ";
                if (reduceNumber.Value.IsMarged)
                {
                    var sortedShapeName = reduceNumber.Value.MargeShapeNames.OrderBy(s => s);
                    foreach (var margeShapeName in sortedShapeName)
                    {
                        resultStr += reduceNumber.Key + "[" + margeShapeName + " is skiped]" + " ";
                    }
                }
            }
            foreach (var resultMessage in _resultMessages)
            {
                resultStr += resultMessage + " ";
            }
            return resultStr;
        }

        // デバッグ時の利便性の為、ログメッセージをコンソールにも出力します。
        [Conditional("DEBUG")]
        private static void WriteToConsole(string message)
        {
            Console.Write(message);
        }

        public static void WriteLog(string message)
        {
            // 実行中のタスクに関連付けられたコンテキストを取得。
            var ctx = G3dParallel.TaskContextStack.Current as MessageWriter;
            if (ctx != null)
            {
                ctx.WriteLog(message);
                WriteToConsole(message);
            }
            else
            {
                lock (LogBuilder)
                {
                    LogBuilder.Append(message);
                    WriteToConsole(message);
                }
            }
        }

        public static void WriteLogLine(string message)
        {
            WriteLog(message + "\n");
        }

        private static bool ExportSources(IEnumerable<KeyValuePair<SourceType, IList<Source>>> sources)
        {
            foreach (var outSourceItem in sources)
            {
                foreach (var source in outSourceItem.Value)
                {
                    var originalStream = source.OriginalStream;
                    var addStream = source.AddStream;
                    var vtxAttribType = source.OriginalVtxAttribType;
                    // 簡略化の結果の量子化型は noneにするのが妥当 SIGLO-79065
                    vtxAttribType.quantize_type = vtx_attrib_quantize_typeType.none;
                    switch (originalStream.type)
                    {
                        case stream_typeType.@float:
                            foreach (var data in addStream.Datas)
                            {
                                originalStream.FloatData.Add((float)(double)data);
                            }
                            vtxAttribType.count = originalStream.FloatData.Count / originalStream.column;
                            break;
                        case stream_typeType.@int:
                            foreach (var data in addStream.Datas)
                            {
                                originalStream.IntData.Add((int)data);
                            }
                            vtxAttribType.count = originalStream.IntData.Count / originalStream.column;
                            break;
                        case stream_typeType.@byte:
                            foreach (var data in addStream.Datas)
                            {
                                originalStream.ByteData.Add((byte)(int)data);
                            }
                            vtxAttribType.count = originalStream.ByteData.Count / originalStream.column;
                            break;
                        case stream_typeType.@string:
                            throw new ArgumentOutOfRangeException();
                        case stream_typeType.wstring:
                            throw new ArgumentOutOfRangeException();
                        default:
                            throw new ArgumentOutOfRangeException();
                    }
                }
            }

            return true;
        }

        private static bool ExportSubMeshes(out ICollection<submeshType> outSubmeshTypes, IEnumerable<SubMesh> subMeshes)
        {
            outSubmeshTypes = new List<submeshType>();
            var submeshIndex = 0;
            foreach (var subMesh in subMeshes)
            {
                outSubmeshTypes.Add(new submeshType { index_hint = subMesh.OriginalSubmeshType.index_hint, offset = subMesh.IndexOffset, count = subMesh.IndexCount, submesh_index = submeshIndex });
                ++submeshIndex;
            }

            return true;
        }

        private static bool ExportMeshes(ICollection<meshType> outMeshTypes, ICollection<G3dStream> outStreams, IEnumerable<PlygonReductionOptimize.Mesh> meshes)
        {
            var lastStreamIndex = 0;
            PlygonReductionOptimize.Mesh lastMesh = null;
            foreach (var mesh in meshes)
            {
                var currentMesh = mesh;

                if (currentMesh.IsOriginal)
                {
                    lastMesh = currentMesh;

                    lastStreamIndex = currentMesh.OriginalMeshType.stream_index;
                    continue;
                }

                int streamIndex;
                if (currentMesh.IsMarged)
                {
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(lastMesh != null, "lastMesh != null");

                    currentMesh = lastMesh;

                    outStreams.Add(null);

                    streamIndex = lastStreamIndex;
                }
                else
                {
                    lastMesh = currentMesh;

                    var newStream = new G3dStream
                    {
                        type = currentMesh.IndexSource.OriginalStream.type,
                        column = currentMesh.IndexSource.OriginalStream.column,
                    };
                    foreach (var data in currentMesh.IndexSource.AddStream.Datas)
                    {
                        newStream.IntData.Add((int)data);
                    }

                    outStreams.Add(newStream);

                    streamIndex = -outStreams.Count;
                    lastStreamIndex = streamIndex;
                }

                int indicesCount;
                if (currentMesh.IsOriginal)
                {
                    indicesCount = currentMesh.IndexSource.OriginalStream.IntData.Count;
                }
                else
                {
                    indicesCount = currentMesh.IndexSource.AddStream.Datas.Count;
                }

                Nintendo.Foundation.Contracts.Assertion.Operation.True(currentMesh != null, "currentMesh != null");

                {
                    ICollection<submeshType> outSubmeshTypes;
                    if (ExportSubMeshes(out outSubmeshTypes, currentMesh.SubMeshes) == false)
                    {
                        throw new Exception("ExportSubMeshes Failed");
                    }

                    var newMeshType = new meshType
                    {
                        index = outMeshTypes.Count,
                        stream_index = streamIndex,
                        // TODO:取り敢えずこれしか対応してない
                        mode = mesh_modeType.triangles,
                        index_hint = currentMesh.OriginalMeshType.index_hint,
                        quantize_type = currentMesh.OriginalMeshType.quantize_type,
                        submesh_array = new submesh_arrayType { Items = outSubmeshTypes.ToArray() },
                        count = indicesCount,
                    };

                    outMeshTypes.Add(newMeshType);
                }
            }

            return true;
        }

        private static bool ExportLodOffsets(out lod_offsetType lodOffsetType, IEnumerable<PlygonReductionOptimize.Mesh> meshes)
        {
            lodOffsetType = new lod_offsetType();

            var offset = 0;
            var count = 0;
            foreach (var mesh in meshes)
            {
                if (mesh.IsOriginal)
                {
                    continue;
                }

                if (count > 0)
                {
                    lodOffsetType.Value += " ";
                }

                if (mesh.IsMarged == false)
                {
                    offset = mesh.SourceOffset;
                }

                lodOffsetType.Value += offset;
                ++count;
            }

            lodOffsetType.count = count;

            return true;
        }

        private static SortedDictionary<int, FaceCount> CreateLog(IEnumerable<PlygonReductionOptimize.Mesh> meshes, string shapeName)
        {
            var result = new SortedDictionary<int, FaceCount>();
            var baseFaceCount = 0;
            var level = 1;
            PlygonReductionOptimize.Mesh lashMesh = null;
            foreach (var mesh in meshes)
            {
                if (mesh.IsOriginal)
                {
                    if (mesh.OriginalMeshType.mode == mesh_modeType.triangles)
                    {
                        baseFaceCount = mesh.IndexSource.OriginalStream.IntData.Count / 3;
                    }
                    else if (mesh.OriginalMeshType.mode == mesh_modeType.triangle_fan)
                    {
                        baseFaceCount = mesh.IndexSource.OriginalStream.IntData.Count - 2;
                    }
                    else if (mesh.OriginalMeshType.mode == mesh_modeType.triangle_strip)
                    {
                        baseFaceCount = mesh.IndexSource.OriginalStream.IntData.Count - 2;
                    }
                    lashMesh = mesh;
                }
                else
                {
                    if (mesh.IsMarged)
                    {
                        Nintendo.Foundation.Contracts.Assertion.Operation.True(lashMesh != null, "lashMesh != null");

                        result.Add(level++, new FaceCount { Before = baseFaceCount, After = lashMesh.IsOriginal ? baseFaceCount : lashMesh.IndexSource.AddStream.Datas.Count / lashMesh.IndexSource.AddStream.Column, IsMarged = true, MargeShapeNames = new List<string> { shapeName } });
                    }
                    else
                    {
                        result.Add(level++, new FaceCount { Before = baseFaceCount, After = mesh.IndexSource.AddStream.Datas.Count / mesh.IndexSource.AddStream.Column });
                        lashMesh = mesh;
                    }
                }
            }
            return result;
        }

        private static bool IsValid(out List<string> notValidMessages, modelType modelType)
        {
            var isNotValid = false;
            var messages = new List<string>();

            G3dParallel.ForEach(modelType.shape_array.shape, delegate (shapeType shapeType)
            {
                var vertexType = modelType.vertex_array.vertex[shapeType.shape_info.vertex_index];
                var positionCount = 0;
                foreach (var vtxAttrib in vertexType.vtx_attrib_array.Enumerate())
                {
                    if (vtxAttrib.hint.StartsWith("position"))
                    {
                        ++positionCount;
                    }
                }
                if (positionCount > 1)
                {
                    isNotValid = true;
                    messages.Add("Skip because there is more than one position stream.");
                }
            });

            notValidMessages = messages;
            return isNotValid == false;
        }

        protected BoneInfo[] CreateBoneInfo()
        {
            //(SIGLO-28482)
            // BoneInfoを構築しておかないと、Boneから行列は取得できない仕組みになっている
            // IfModelBoneUniteCompressor のCompress 関数より抜粋
            boneType[] bones = Target.skeleton.bone_array.bone;
            int boneCount = bones.Length;
            BoneInfo[] boneInfos = new BoneInfo[boneCount];
            // ボーン情報の初期化
            for (int i = 0; i < boneCount; i++)
            {
                boneInfos[i] = new BoneInfo(bones[i], this.Target.skeleton.skeleton_info);
            }
            BoneInfo.Setup(boneInfos);

            return boneInfos;
        }

        protected override void Optimize()
        {
            // G3dParallel にコンテキストを登録する。
            using (var messageWriter = new MessageWriter((x) => { lock (LogBuilder) { LogBuilder.Append(x); } }))
            using (var stack = new G3dParallel.TaskContextStack(messageWriter))
            {
                OptimizeImpl();
            }
        }

        private void OptimizeImpl()
        {
            IfModelPolygonReductionOptimizer.WriteLogLine(@"IfModelPolygonReductionOptimizer:0.1.86.0");

            var optionsString = "pr:" + _options;
            Dictionary<shapeType, bool> skilShapes = null;
            {// スキップするシェイプのチェック
                var ifModelTrimLodModelOptimizer = new IfModelTrimLodModelOptimizer { Target = Target, Streams = Streams };

                // MargeShapeの場合、すべてのシェイプのオプションが同じ場合はスキップ
                if (_options.IsMargeShape)
                {
                    var isSkip = true;
                    foreach (var shapeType in Target.shape_array.shape)
                    {
                        var polygonReductionModes = string.IsNullOrEmpty(shapeType.shape_info.polygon_reduction_mode) ? new string[0] : shapeType.shape_info.polygon_reduction_mode.Split(';');
                        if (polygonReductionModes.Any(s => s == optionsString) == false)
                        {
                            isSkip = false;
                            break;
                        }
                    }
                    if (isSkip)
                    {
                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Skip_Proc"));
                        return;
                    }

                    {
                        // 以前のLODメッシュ削除
                        ifModelTrimLodModelOptimizer.Optimize(Target, Streams);
                    }
                }
                else
                {// MargeShapeじゃない場合は、オプションの違うShapeのみ以前のLODメッシュ削除
                    skilShapes = new Dictionary<shapeType, bool>();
                    var isRemove = false;
                    foreach (var shapeType in Target.shape_array.shape)
                    {
                        var polygonReductionModes = string.IsNullOrEmpty(shapeType.shape_info.polygon_reduction_mode) ? new string[0] : shapeType.shape_info.polygon_reduction_mode.Split(';');
                        if (polygonReductionModes.Any(s => s == optionsString) == false)
                        {
                            if (shapeType.mesh_array.length > 1)
                            {
                                ifModelTrimLodModelOptimizer.TrimLod(shapeType);
                                isRemove = true;
                            }
                            skilShapes.Add(shapeType, false);
                        }
                        else
                        {
                            skilShapes.Add(shapeType, true);
                        }
                    }
                    if (isRemove)
                    {
                        StreamUtility.SortStream(Target, Streams);
                    }
                    if (skilShapes.Any(pair => pair.Value == false) == false)
                    {
                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Skip_Proc"));
                        return;
                    }
                }
            }

            {// 値チェック
                List<string> notValidMessages;
                if (IsValid(out notValidMessages, Target) == false)
                {
                    foreach (var notValidMessage in notValidMessages)
                    {
                        nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(notValidMessage);
                        _resultMessages.Add(notValidMessage);
                    }
                    return;
                }
            }
#if true
            var model = new PlygonReductionOptimize.Model
            {
                Skeleton = new Skeleton
                {
                    SkeletonType = Target.skeleton
                },
                //(SIGLO-28482) 簡略化前の座標変換の為に追加
                BoneInfos = CreateBoneInfo()
            };

            {// データ入力

                foreach (var shapeType in Target.shape_array.shape)
                {
                    if (skilShapes != null)
                    {
                        bool isSkip;
                        if (skilShapes.TryGetValue(shapeType, out isSkip) == false)
                        {
                            throw ExcepHandle.CreateException("skilShapes tryget Failed");
                        }
                        if (isSkip)
                        {
                            nw.g3d.iflib.src.Optimize.Model.IfModelPolygonReductionOptimizer.WriteLogLine(_3dIntermediateFilePlygonReduction.IfStrings.Get("PolygonReduction_Skip_Shape", shapeType.index, shapeType.name));
                            return;
                        }
                    }
                    var polygonReductionModes = string.IsNullOrEmpty(shapeType.shape_info.polygon_reduction_mode) ? new string[0] : shapeType.shape_info.polygon_reduction_mode.Split(';');
                    var newList = new List<string>();
                    foreach (var polygonReductionMode in polygonReductionModes)
                    {
                        if (polygonReductionMode.StartsWith("pr:") == false)
                        {
                            newList.Add(polygonReductionMode);
                        }
                    }
                    newList.Add(optionsString);

                    shapeType.shape_info.polygon_reduction_mode = string.Empty;
                    foreach (var newItem in newList)
                    {
                        shapeType.shape_info.polygon_reduction_mode += newItem + ";";
                    }
                    shapeType.shape_info.polygon_reduction_mode = shapeType.shape_info.polygon_reduction_mode.Remove(shapeType.shape_info.polygon_reduction_mode.Length - 1);

                    var vertex = Target.vertex_array.vertex[shapeType.shape_info.vertex_index];
                    var shape = Shape.ConvertFrom(shapeType, vertex, Streams);

                    // (SIGLO-28482) 2016/06/08 追加　
                    // 重複（クローンされたインスタンス）オブジェクトのリダクションを行うために
                    // アフィン変換行列をかけて、重複のない位置への移動を行う必要がある。
                    var boneinfo = model.BoneInfos.FirstOrDefault(o => o.Bone.name == shapeType.shape_info.bone_name);
                    if (boneinfo != null)
                    {
                        shape.AffineMatrx = ModelCompressUtility.GetBoneTransform(Target, boneinfo);
                    }

                    // リストに追加されたオブジェクトは、簡略化を実行する
                    // その直前にハッシュの計算を行う
                    string hash = shapeType.shape_info.original_shape_hash;
                    if (string.IsNullOrEmpty(hash))
                    {
                        hash = IfOptimizePrimitiveHashCalculator.Calculate(
                            this.Target, this.Streams, shapeType);
                        shapeType.shape_info.original_shape_hash = hash;
                    }

                    // ポリゴンリダクション適用後のメッシュのインデックスストリームは最適化が適用されていない
                    // 状態になるので、プリミティブ最適化の情報を破棄する。
                    shapeType.shape_info.optimize_primitive_mode = string.Empty;

                    lock (model.Shapes)
                    {
                        model.Shapes.Add(shape);
                    }
                }
            }

            {// リダクション
                if (Reduction.Execute(model, _options) == false)
                {
                    throw ExcepHandle.CreateException("Reduction.Execute Failed");
                }
            }

            var addStreams = new SynchronizedCollection<AddStream>();
            {
                // データ出力（追加ストリーム以外）
                foreach (var shape in model.Shapes)
                {
                    var shapeType = shape.OrignalShapeType;
                    var vertex = Target.vertex_array.vertex[shapeType.shape_info.vertex_index];
                    {
                        if (shape.SourcesDictionary != null)
                        {
                            if (ExportSources(shape.SourcesDictionary) == false)
                            {
                                throw ExcepHandle.CreateException("ExportSources Failed");
                            }
                        }
                    }
                    {
                        IList<meshType> outMeshTypes = new List<meshType>();
                        IList<G3dStream> outStreams = new List<G3dStream>();

                        if (ExportMeshes(outMeshTypes, outStreams, shape.Meshes) == false)
                        {
                            throw ExcepHandle.CreateException("ExportMeshes Failed");
                        }

                        IList<meshType> newMeshTypes = new List<meshType>(shapeType.mesh_array.mesh);
                        var meshTypeIndex = 0;
                        foreach (var outMeshType in outMeshTypes)
                        {
                            outMeshType.index = newMeshTypes.Count;
                            newMeshTypes.Add(outMeshType);

                            if (outStreams[meshTypeIndex] != null)
                            {
                                addStreams.Add(new AddStream
                                {
                                    ShapeType = shapeType,
                                    MeshType = outMeshType,
                                    Stream = outStreams[meshTypeIndex]
                                });
                            }

                            ++meshTypeIndex;
                        }

                        shapeType.mesh_array.mesh = newMeshTypes.ToArray();
                    }
                    {
                        lod_offsetType lodOffsetType;
                        if (ExportLodOffsets(out lodOffsetType, shape.Meshes) == false)
                        {
                            throw ExcepHandle.CreateException("ExportLodOffsets Failed");
                        }
                        vertex.lod_offset = lodOffsetType;
                    }
                    {
                        var reduceFaceCounts = CreateLog(shape.Meshes, shape.OrignalShapeType.name);

                        foreach (var reduceFaceCount in reduceFaceCounts)
                        {
                            lock (_reduceFaceCounts)
                            {
                                if (_reduceFaceCounts.ContainsKey(reduceFaceCount.Key) == false)
                                {
                                    _reduceFaceCounts.Add(reduceFaceCount.Key, new FaceCount { MargeShapeNames = new List<string>() });
                                }
                                var faceCount = _reduceFaceCounts[reduceFaceCount.Key];
                                faceCount.Before += reduceFaceCount.Value.Before;
                                faceCount.After += reduceFaceCount.Value.After;
                                faceCount.IsMarged = faceCount.IsMarged | reduceFaceCount.Value.IsMarged;
                                if (reduceFaceCount.Value.MargeShapeNames != null)
                                {
                                    faceCount.MargeShapeNames.AddRange(reduceFaceCount.Value.MargeShapeNames);
                                }
                                _reduceFaceCounts[reduceFaceCount.Key] = faceCount;
                            }
                        }
                    }
                }
            }
#else
            var addStreams = new SynchronizedCollection<AddStream>();
            G3dParallel.ForEach(Target.shape_array.shape, delegate(shapeType shapeType)
            {
                var vertex = Target.vertex_array.vertex[shapeType.shape_info.vertex_index];
                var shape = new Shape();
                {
                    if (ImportSources(shape.SourcesDictionary, vertex, Streams) == false)
                    {
                        throw new Exception("ImportOriginalStreams Failed");
                    }
                }
                {
                    if (ImportMeshes(shape.Meshes, shapeType.mesh_array.mesh, Streams) == false)
                    {
                        throw new Exception("ImportMeshes Failed");
                    }
                }
                {
                    if (Reduction.Execute(shape, _options) == false)
                    {
                        throw new Exception("Reduction.Execute Failed");
                    }
                }
                {
                    if (shape.SourcesDictionary != null)
                    {
                        if (ExportSources(shape.SourcesDictionary) == false)
                        {
                            throw new Exception("ExportSources Failed");
                        }
                    }
                }
                {
                    IList<meshType> outMeshTypes = new List<meshType>();
                    IList<G3dStream> outStreams = new List<G3dStream>();

                    if (ExportMeshes(outMeshTypes, outStreams, shape.Meshes) == false)
                    {
                        throw new Exception("ExportMeshes Failed");
                    }

                    IList<meshType> newMeshTypes = new List<meshType>(shapeType.mesh_array.mesh);
                    var meshTypeIndex = 0;
                    foreach (var outMeshType in outMeshTypes)
                    {
                        outMeshType.index = newMeshTypes.Count;
                        newMeshTypes.Add(outMeshType);

                        addStreams.Add(new AddStream { ShapeType = shapeType, MeshType = outMeshType, Stream = outStreams[meshTypeIndex] });

                        ++meshTypeIndex;
                    }

                    shapeType.mesh_array.mesh = newMeshTypes.ToArray();
                }
                {
                    lod_offsetType lodOffsetType;
                    if (ExportLodOffsets(out lodOffsetType, shape.Meshes) == false)
                    {
                        throw new Exception("ExportLodOffsets Failed");
                    }
                    vertex.lod_offset = lodOffsetType;
                }
                {
                    var sourcesCount = shape.SourcesDictionary.Sum(pair => pair.Value.Count);
                    var reduceFaceCounts = CreateLog(shape.Meshes, sourcesCount);

                    foreach (var reduceFaceCount in reduceFaceCounts)
                    {
                        lock (_reduceFaceCounts)
                        {
                            if (_reduceFaceCounts.ContainsKey(reduceFaceCount.Key) == false)
                            {
                                _reduceFaceCounts.Add(reduceFaceCount.Key, new FaceCount());
                            }
                            var faceCount = _reduceFaceCounts[reduceFaceCount.Key];
                            faceCount.Before += reduceFaceCount.Value.Before;
                            faceCount.After += reduceFaceCount.Value.After;
                            _reduceFaceCounts[reduceFaceCount.Key] = faceCount;
                        }
                    }
                }
            });
#endif
            {// 追加ストリーム出力
                var addStreamsSorted = addStreams.ToSortedList((stream, addStream) =>
                {
                    if (stream.ShapeType.index > addStream.ShapeType.index)
                    {
                        return 1;
                    }
                    if (stream.ShapeType.index < addStream.ShapeType.index)
                    {
                        return -1;
                    }
                    if (stream.MeshType.index > addStream.MeshType.index)
                    {
                        return 1;
                    }
                    if (stream.MeshType.index < addStream.MeshType.index)
                    {
                        return -1;
                    }
                    return 0;
                });

                foreach (var addStream in addStreamsSorted)
                {
                    int streamIndex = addStream.MeshType.stream_index;
                    var meshes = addStream.ShapeType.mesh_array.mesh.Where(type => type.stream_index == streamIndex);
                    foreach (var meshType in meshes)
                    {
                        meshType.stream_index = Streams.Count;
                    }
                    Streams.Add(addStream.Stream);
                }

                StreamUtility.SortStream(Target, Streams);
            }
        }

        public override string ToString()
        {
            var builder = new StringBuilder();
            builder.AppendLine(base.ToString());

            lock (LogBuilder)
            {
                builder.Append(LogBuilder.ToString());
            }

            builder.AppendLine(GetResult());

            return builder.ToString();
        }
    }
}
