﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.IO;
using System;
using nw.g3d.iflib;
using nw.g3d.toollib;
using nw.g3d.toollib.Ext;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.optcvtr
{
    // テクスチャパターンアニメオプティマイザ
    internal class TexPatternAnimOptimizer : Optimizer
    {
        // コンストラクタ
        internal TexPatternAnimOptimizer(g3doptcvtrParams fileOption)
            : base(fileOption)
        {
            DisableCompressOption();

#if false
            // 中間ファイルの修正はデバッグ目的のみ使用
            ArgumentOption repair = fileOption.GetOption("--repair");
            if (repair != null)
            {
                RepairFile = true;
            }

            // 中間ファイルのチェックはデバッグ目的のみ使用
            ArgumentOption check = fileOption.GetOption("--check_stream");
            if (check != null)
            {
                CheckFile = true;
            }
#endif

            // マージ
            if (fileOption.Merge)
            {
                if (fileOption.MergeOptions == null)
                {
                    Strings.Throw("Error_OptionNotSpecified", fileOption.GetOptionName(() => fileOption.MergeOptions));
                }

                var mergeOptions = fileOption.MergeOptions;

                // マージファイルの設定
                if (!File.Exists(mergeOptions.MergeFile))
                {
                    Strings.Throw("Error_OptionFileNotFound", mergeOptions.MergeFile);
                }
                if (!G3dPath.IsTexPatternAnimPath(mergeOptions.MergeFile))
                {
                    Strings.Throw("Error_Merge_InvalidFileType", mergeOptions.MergeFile);
                }

                this.MergeFilePath = mergeOptions.MergeFile;
            }

            // 同一頂点の削除
            CheckUnsupportedOption(fileOption, () => fileOption.DeleteNearVertex);

            // サブメッシュ分割
            CheckUnsupportedOption(fileOption, () => fileOption.UniteSubmesh);
            CheckUnsupportedOption(fileOption, () => fileOption.DivideSubmesh);

            // PolygonReduction
            CheckUnsupportedOption(fileOption, () => fileOption.PolygonReduction);
            CheckUnsupportedOption(fileOption, () => fileOption.PolygonReductionOptions);

            // プリミティブ最適化
            CheckUnsupportedOption(fileOption, () => fileOption.OptimizePrimitive);

            // LOD モデル結合・削除
            CheckUnsupportedOption(fileOption, () => fileOption.UniteLodModel);
            CheckUnsupportedOption(fileOption, () => fileOption.TrimLodModel);

            // マテリアルアニメ統合
            if (fileOption.Combine)
            {
                CombineFile = true;
                if (string.IsNullOrEmpty(fileOption.Output) || !G3dPath.IsMaterialAnimPath(fileOption.Output))
                {
                    Strings.Throw("Error_CombineOutputMustbeMatAnim", fileOption.Output);
                }

                if (fileOption.CombineOptions != null)
                {
                    string errorPath;
                    if (!IfMaterialAnimCombineUtility.CheckCombinePaths(fileOption.CombineOptions.Paths, out errorPath))
                    {
                        Strings.Throw("Error_Combine_InvalidFilePath", errorPath);
                    }
                    CombineFilePaths = fileOption.CombineOptions.Paths;
                    CombineIgnoreFrameLoopUnmatch = fileOption.CombineOptions.IgnoreFrameLoopUnmatch;
                    CombineIgnoreAnimationConflict = fileOption.CombineOptions.IgnoreAnimationConflict;
                    this.CombineProcessLogArgument = fileOption.CombineOptions.CreateProcessLogArgument();
                }
            }

            // 量子化分析
            if (fileOption.QuantizationAnalysis)
            {
                this.Optimizers.Add(new IfTexPatternAnimQuantizationAnalysisOptimizer());
            }
        }

        //=====================================================================
        // 最適化
        internal override void Optimize()
        {
            // 読み込み
            Read();
            tex_pattern_animType tex_pattern_anim =
                this.nw4f_3dif.Item as tex_pattern_animType;

            // データの正当性チェック
            Check(tex_pattern_anim);

            // 修正
            Repair(tex_pattern_anim);

            // マージ
            Merge(tex_pattern_anim);

            // マテリアルアニメ統合
            Combine(tex_pattern_anim);

            // テクスチャパターンアニメ最適化
            foreach (IfTexPatternAnimOptimizer optimizer in this.Optimizers)
            {
                optimizer.Optimize(tex_pattern_anim, this.Streams);

                // 最適化ログの追加
                if (optimizer.EnableProcessLog)
                {
                    IfProcessLogUtility.Add(this.nw4f_3dif,
                        optimizer.Process, optimizer.Argument, optimizer.GetResult());
                }
            }

            IfProcessLogUtility.ToDistinctLatest(this.nw4f_3dif);

            // 書き出し
            Write();
        }

        //---------------------------------------------------------------------
        // 確認
        private void Check(tex_pattern_animType texPatternAnim)
        {
            if (!CheckFile) { return; }

            // ストリームインデックスの正当性を確認する
            string checkLog = string.Empty;
            bool checkResult =
                StreamChecker.CheckStreamIndex(out checkLog, texPatternAnim, this.Streams);
            // チェックが失敗したらログを例外として投げて表示させる
            if (!checkResult)
            {
                throw new Exception(checkLog);
            }
            else
            {
                Console.WriteLine("ok.");
            }
        }

        //---------------------------------------------------------------------
        // 修正
        private void Repair(tex_pattern_animType texPatternAnim)
        {
            if (!RepairFile) { return; }

            StreamUtility.SortStream(texPatternAnim, this.Streams);
        }

        //---------------------------------------------------------------------
        // マージ
        private void Merge(tex_pattern_animType newTexPatternAnim)
        {
            if (this.MergeFilePath == null) { return; }

            this.MergeStopwatch.Start();
            // マージファイルの読み込み
            List<G3dStream> oldStreams = new List<G3dStream>();
            nw4f_3difType oldIf = IfReadUtility.Read(
                oldStreams, this.MergeFilePath, this.XsdBasePath);
            tex_pattern_animType oldTexPatternAnim = oldIf.Item as tex_pattern_animType;
            IfTexPatternAnimMerger.Merge(
                newTexPatternAnim,
                this.Streams,
                oldTexPatternAnim,
                oldStreams);

            // 最適化ログの追加
            IfProcessLogUtility.Add(this.nw4f_3dif, "merge",
                Path.GetFullPath(this.MergeFilePath).Replace('\\', '/'), string.Empty); /// TODO: Result 対応

            this.MergeStopwatch.Stop();
        }

        //---------------------------------------------------------------------
        // マテリアルアニメ統合
        private void Combine(tex_pattern_animType tex_pattern_anim)
        {
            if (!CombineFile) { return; }
            this.CombineStopwatch.Start();

            // マテリアルアニメに変換
            var materialAnim = IfMaterialAnimCombineUtility.ConvertToMaterialAnim(tex_pattern_anim, this.Streams);
            this.nw4f_3dif.Item = materialAnim;

            var combinedFiles = new List<string>();
            if (this.CombineFilePaths != null)
            {
                combinedFiles.AddRange(this.CombineFilePaths);
                var combineSettings = new CombineSettings()
                {
                    XsdBasePath = this.XsdBasePath,
                    IgnoreFrameLoopUnmatch = this.CombineIgnoreFrameLoopUnmatch,
                    IgnoreAnimationConflict = this.CombineIgnoreAnimationConflict
                };

                if (!IfMaterialAnimCombineUtility.Combine(materialAnim, this.Streams, this.CombineFilePaths, combineSettings))
                {
                    throw new Exception(combineSettings.MessageLog);
                }
            }

            // 統合ログの追加
            IfProcessLogUtility.Add(this.nw4f_3dif, "combine", this.CombineProcessLogArgument, string.Empty);
            this.CombineStopwatch.Stop();
        }

        //---------------------------------------------------------------------
        // ログの取得
        internal override string GetLog()
        {
            StringBuilder builder = new StringBuilder();
            builder.AppendLine(base.GetLog());
            builder.AppendLine(GetReadLog());

            // マージ
            if (this.MergeFilePath != null)
            {
                builder.AppendLine(Strings.Get("MergeLog",
                    this.MergeStopwatch.ElapsedMilliseconds,
                    this.MergeFilePath));
            }

            // 統合
            if (this.CombineFilePaths != null)
            {
                builder.AppendLine(Strings.Get("CombineLog",
                    this.CombineStopwatch.ElapsedMilliseconds,
                    string.Join(",", this.CombineFilePaths.Select(s => Path.GetFullPath(s).Replace('\\', '/')))));
            }

            // テクスチャパターンアニメ最適化
            foreach (IfTexPatternAnimOptimizer optimizer in this.Optimizers)
            {
                builder.AppendLine("    " + optimizer.ToString());
            }

            builder.Append(GetWriteLog());
            return builder.ToString();
        }

        //=====================================================================
        private readonly string MergeFilePath;
        private readonly bool RepairFile = false;
        private readonly bool CheckFile = false;
        private readonly bool CombineFile = false;
        private readonly string[] CombineFilePaths;
        private readonly bool CombineIgnoreAnimationConflict = false;
        private readonly bool CombineIgnoreFrameLoopUnmatch = false;
        private readonly string CombineProcessLogArgument = string.Empty;
        protected readonly Stopwatch CombineStopwatch = new Stopwatch();
        protected readonly Stopwatch MergeStopwatch = new Stopwatch();
        private List<IfTexPatternAnimOptimizer> Optimizers =
            new List<IfTexPatternAnimOptimizer>();
    }
}
