﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.IO;
using System;
using nw.g3d.iflib;
using nw.g3d.toollib;
using nw.g3d.toollib.Ext;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.optcvtr
{
    // シーンアニメオプティマイザ
    internal class SceneAnimOptimizer : Optimizer
    {
        // コンストラクタ
        internal SceneAnimOptimizer(g3doptcvtrParams fileOption)
            : base(fileOption)
        {
            DisableCompressOption();

#if false
            // 中間ファイルの修正はデバッグ目的のみ使用
            ArgumentOption repair = fileOption.GetOption("--repair");
            if (repair != null)
            {
                RepairFile = true;
            }

            // 中間ファイルのチェックはデバッグ目的のみ使用
            ArgumentOption check = fileOption.GetOption("--check_stream");
            if (check != null)
            {
                CheckFile = true;
            }
#endif

            // マージ
            if (fileOption.Merge)
            {
                if (fileOption.MergeOptions == null)
                {
                    Strings.Throw("Error_OptionNotSpecified", fileOption.GetOptionName(() => fileOption.MergeOptions));
                }

                var mergeOptions = fileOption.MergeOptions;

                // マージファイルの設定
                if (!File.Exists(mergeOptions.MergeFile))
                {
                    Strings.Throw("Error_OptionFileNotFound", mergeOptions.MergeFile);
                }
                if (!G3dPath.IsSceneAnimPath(mergeOptions.MergeFile))
                {
                    Strings.Throw("Error_Merge_InvalidFileType", mergeOptions.MergeFile);
                }
                this.MergeFilePath = mergeOptions.MergeFile;
            }

            // 同一頂点の削除
            CheckUnsupportedOption(fileOption, () => fileOption.DeleteNearVertex);

            // サブメッシュ分割
            CheckUnsupportedOption(fileOption, () => fileOption.UniteSubmesh);
            CheckUnsupportedOption(fileOption, () => fileOption.DivideSubmesh);

            // PolygonReduction
            CheckUnsupportedOption(fileOption, () => fileOption.PolygonReduction);
            CheckUnsupportedOption(fileOption, () => fileOption.PolygonReductionOptions);

            // プリミティブ最適化
            CheckUnsupportedOption(fileOption, () => fileOption.OptimizePrimitive);

            // LOD モデル結合・削除
            CheckUnsupportedOption(fileOption, () => fileOption.UniteLodModel);
            CheckUnsupportedOption(fileOption, () => fileOption.TrimLodModel);

            // マテリアルアニメ統合
            CheckUnsupportedOption(fileOption, () => fileOption.Combine);
            CheckUnsupportedOption(fileOption, () => fileOption.CombineOptions);

            // 量子化分析
            if (fileOption.QuantizationAnalysis)
            {
                this.Optimizers.Add(new IfSceneAnimQuantizationAnalysisOptimizer());
            }
        }

        //=====================================================================
        // 最適化
        internal override void Optimize()
        {
            // 読み込み
            Read();
            scene_animType scene_anim =
                this.nw4f_3dif.Item as scene_animType;

            // データの正当性チェック
            Check(scene_anim);

            // 修正
            Repair(scene_anim);

            // マージ
            Merge(scene_anim);

            // シーンアニメ最適化
            foreach (IfSceneAnimOptimizer optimizer in this.Optimizers)
            {
                optimizer.Optimize(scene_anim, this.Streams);

                // 最適化ログの追加
                if (optimizer.EnableProcessLog)
                {
                    IfProcessLogUtility.Add(this.nw4f_3dif,
                        optimizer.Process, optimizer.Argument, optimizer.GetResult());
                }
            }

            IfProcessLogUtility.ToDistinctLatest(this.nw4f_3dif);

            // 書き出し
            Write();
        }

        //---------------------------------------------------------------------
        // 確認
        private void Check(scene_animType sceneAnim)
        {
            if (!CheckFile) { return; }

            // ストリームインデックスの正当性を確認する
            string checkLog = string.Empty;
            bool checkResult =
                StreamChecker.CheckStreamIndex(out checkLog, sceneAnim, this.Streams);
            // チェックが失敗したらログを例外として投げて表示させる
            if (!checkResult)
            {
                throw new Exception(checkLog);
            }
            else
            {
                Console.WriteLine("ok.");
            }
        }

        //---------------------------------------------------------------------
        // 修正
        private void Repair(scene_animType sceneAnim)
        {
            if (!RepairFile) { return; }

            StreamUtility.SortStream(sceneAnim, this.Streams);
        }

        //---------------------------------------------------------------------
        // マージ
        private void Merge(scene_animType newSceneAnim)
        {
            if (this.MergeFilePath == null) { return; }

            this.MergeStopwatch.Start();
            // マージファイルの読み込み
            List<G3dStream> oldStreams = new List<G3dStream>();
            nw4f_3difType oldIf = IfReadUtility.Read(
                oldStreams, this.MergeFilePath, this.XsdBasePath);
            scene_animType oldSceneAnim = oldIf.Item as scene_animType;

            IfSceneAnimMergeContext context = new IfSceneAnimMergeContext(
                newSceneAnim, this.Streams, oldSceneAnim, oldStreams);
            context.Setup();
            IfSceneAnimMerger.Merge(context);

            // 最適化ログの追加
            IfProcessLogUtility.Add(this.nw4f_3dif, "merge",
                Path.GetFullPath(this.MergeFilePath).Replace('\\', '/'), string.Empty);

            this.MergeStopwatch.Stop();
        }

        //---------------------------------------------------------------------
        // ログの取得
        internal override string GetLog()
        {
            StringBuilder builder = new StringBuilder();
            builder.AppendLine(base.GetLog());
            builder.AppendLine(GetReadLog());

            // マージ
            if (this.MergeFilePath != null)
            {
                builder.AppendLine(Strings.Get("MergeLog",
                    this.MergeStopwatch.ElapsedMilliseconds,
                    this.MergeFilePath));
            }

            // シーンアニメ最適化
            foreach (IfSceneAnimOptimizer optimizer in this.Optimizers)
            {
                builder.AppendLine("    " + optimizer.ToString());
            }

            builder.Append(GetWriteLog());
            return builder.ToString();
        }

        //=====================================================================
        private readonly string MergeFilePath;
        private readonly bool RepairFile = false;
        private readonly bool CheckFile = false;
        protected readonly Stopwatch MergeStopwatch = new Stopwatch();
        private List<IfSceneAnimOptimizer> Optimizers =
            new List<IfSceneAnimOptimizer>();
    }
}
