﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using nw4f.meshlib;
using nw.g3d.iflib;
using nw.g3d.iflib.src.Optimize.Model;
using nw.g3d.toollib;
using nw.g3d.toollib.Ext;
using System;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.optcvtr
{
    // モデルオプティマイザ
    internal class ModelOptimizer : Optimizer
    {
        // コンストラクタ
        internal ModelOptimizer(g3doptcvtrParams fileOption)
            : base(fileOption)
        {
            // マテリアル圧縮
            if (fileOption.CompressMaterial)
            {
                this.Compressors.Add(new IfModelMaterialCompressor());
            }

            // リジッドボディをリジッドスキニングに変換
            if (fileOption.RigidBodyToRigidSkinning)
            {
                this.Compressors.Add(new IfRigidBodyToRigidSkinning());
            }

            // 刈り込みボーン圧縮
            if (fileOption.CompressBoneCull)
            {
                this.Compressors.Add(new IfModelBoneCullCompressor());
            }

            // マージボーン圧縮
            if (fileOption.CompressBoneMerge)
            {
                this.Compressors.Add(new IfModelBoneMergeCompressor());
            }

            // 親子合体ボーン圧縮
            if (fileOption.CompressBoneUniteChild)
            {
                this.Compressors.Add(new IfModelBoneUniteChildCompressor());
            }

            // 合体ボーン圧縮
            if (fileOption.CompressBoneUnite)
            {
                this.Compressors.Add(new IfModelBoneUniteCompressor());
            }

            // 完全合体ボーン圧縮
            if (fileOption.CompressBoneUniteAll)
            {
                this.Compressors.Add(new IfModelBoneUniteAllCompressor());
            }

            // シェイプ圧縮
            if (fileOption.CompressShape)
            {
                var ignoreSkinningCount = fileOption.CompressShapeOptions != null &&
                                          fileOption.CompressShapeOptions.IgnoreSkinningCount;
                this.Compressors.Add(new IfModelShapeCompressor(ignoreSkinningCount));
            }

#if false
            // 中間ファイルの修正はデバッグ目的のみ使用
            ArgumentOption repair = fileOption.GetOption("--repair");
            if (repair != null)
            {
                RepairFile = true;
            }

            // 中間ファイルのチェックはデバッグ目的のみ使用
            ArgumentOption check = fileOption.GetOption("--check_stream");
            if (check != null)
            {
                CheckFile = true;
            }
#endif

            // マージ
            if (fileOption.Merge)
            {
                if (fileOption.MergeOptions == null)
                {
                    Strings.Throw("Error_OptionNotSpecified", fileOption.GetOptionName(() => fileOption.MergeOptions));
                }

                var mergeOptions = fileOption.MergeOptions;

                // マージファイルの設定
                if (!File.Exists(mergeOptions.MergeFile))
                {
                    Strings.Throw("Error_OptionFileNotFound", mergeOptions.MergeFile);
                }
                if (!G3dPath.IsModelPath(mergeOptions.MergeFile))
                {
                    Strings.Throw("Error_Merge_InvalidFileType", mergeOptions.MergeFile);
                }

                this.MergeFilePath = mergeOptions.MergeFile;

                // サンプラーのマージ優先度設定
                this.SamplerMergePriorityToNewModel = mergeOptions.SamplerMergePriority;

                // テクスチャー SRT のマージ優先度設定
                this.TextureSrtMergePriorityToNewModel = mergeOptions.TextureSrtMergePriority;

                // ボーンビジビリティのマージ優先度設定
                this.BoneVisibilityMergePriorityToNewModel = mergeOptions.BoneVisibilityMergePriority;
            }

            // 同一頂点の削除
            if (fileOption.DeleteNearVertex)
            {
                var value = string.Empty;
                var options = fileOption.DeleteNearVertexOptions;
                if (options != null)
                {
                    if (options.Position.HasValue)
                    {
                        value += string.Format("position: {0:R},", options.Position.Value);
                    }

                    if (options.Normal.HasValue)
                    {
                        value += string.Format("normal: {0:R},", options.Normal.Value);
                    }

                    if (options.Tangent.HasValue)
                    {
                        value += string.Format("tangent: {0:R},", options.Tangent.Value);
                    }

                    if (options.Binormal.HasValue)
                    {
                        value += string.Format("binormal: {0:R},", options.Binormal.Value);
                    }

                    if (options.Color.HasValue)
                    {
                        value += string.Format("color: {0:R},", options.Color.Value);
                    }

                    if (options.Uv.HasValue)
                    {
                        value += string.Format("uv: {0:R},", options.Uv.Value);
                    }

                    if (options.BlendIndex.HasValue)
                    {
                        value += string.Format("blendindex: {0:R},", options.BlendIndex.Value);
                    }

                    if (options.BlendWeight.HasValue)
                    {
                        value += string.Format("blendweight: {0:R},", options.BlendWeight.Value);
                    }

                    // 末尾の , を除く
                    if (value.Length > 0)
                    {
                        value = value.Substring(0, value.Length - 1);
                    }
                }
                this.Optimizers.Add(new IfModelDeleteEqualVertexOptimizer(value));
            }

            // LOD モデル削除
            // サブメッシュ分割が LOD モデルありに未対応なため、先に削除する
            if (fileOption.TrimLodModel)
            {
                this.Optimizers.Add(new IfModelTrimLodModelOptimizer());
            }

            // サブメッシュ分割
            if (fileOption.UniteSubmesh)
            {
                this.Optimizers.Add(new IfModelUniteSubmeshOptimizer());
            }
            if (fileOption.DivideSubmesh)
            {
                var value = "kd(";
                var options = fileOption.DivideSubmeshOptions;
                if (options != null)
                {
                    string prefix = string.Empty;
                    if (options.MaxBoundingBoxWidth.HasValue)
                    {
                        value += prefix + string.Format("max_width:{0:R}", options.MaxBoundingBoxWidth.Value);
                        prefix = ",";
                    }

                    if (options.MaxDivisionDepth.HasValue)
                    {
                        value += prefix + string.Format("max_depth:{0}", options.MaxDivisionDepth.Value);
                        prefix = ",";
                    }

                    if (options.MinTriangleCount.HasValue)
                    {
                        value += prefix + string.Format("min_triangle_count:{0}", options.MinTriangleCount.Value);
                        prefix = ",";
                    }
                }
                value += ")";
                this.Optimizers.Add(new IfModelDivideSubmeshOptimizer(value));
            }

            // PolygonReduction
            if (fileOption.PolygonReduction)
            {
                if (fileOption.PolygonReductionOptions == null)
                {
                    Strings.Throw("Error_OptionNotSpecified", fileOption.GetOptionName(() => fileOption.PolygonReductionOptions));
                }

                var options = new IfModelPolygonReductionOptimizer.Options();
                options.TargetRates = new SortedDictionary<int, float>();
                var polygonReductionOptions = fileOption.PolygonReductionOptions;
                if (polygonReductionOptions.DistanceOffset.HasValue)
                {
                    options.DistOffset = polygonReductionOptions.DistanceOffset.Value;
                }

                if (polygonReductionOptions.TargetRatePerMaterialShapes != null)
                {
                    // TODO: この辺のパース処理は g3doptcvtrParams.PolygonReductionSubOptions にコンバーターを作ってそこに実装する
                    options.SetRateForMaterialShapes(polygonReductionOptions.TargetRatePerMaterialShapes);
                }

                if (polygonReductionOptions.Smoothing)
                {
                    options.Smoothing = polygonReductionOptions.Smoothing;
                }

                // 新旧のパラメータは排他的にする
                // 移行期間終了後に削除予定
                if (polygonReductionOptions.LockUvHardEdgeAngle.HasValue)
                {
                    options.LockUvHardEdgeAngle = polygonReductionOptions.LockUvHardEdgeAngle.Value;
                }
                else if (polygonReductionOptions.OldLockUvHardEdge.HasValue)
                {
                    IfModelPolygonReductionOptimizer.WriteLogLine(Strings.Get("Warning_Depricated_Option", "lock-uv-hard-edge"));
                    options.OldLockUvHardEdgeAngle = polygonReductionOptions.OldLockUvHardEdge.Value;
                }

                if (polygonReductionOptions.PropertyFlag != null)
                {
                    // TODO: この辺のパース処理は g3doptcvtrParams.PolygonReductionSubOptions にコンバーターを作ってそこに実装する
                    options.SetPropertyFlags(polygonReductionOptions.PropertyFlag);
                }

                if (polygonReductionOptions.Policy != null)
                {
                    options.SetPolicy(polygonReductionOptions.Policy);
                }

                if (polygonReductionOptions.Quality.HasValue)
                {
                    options.Quality = polygonReductionOptions.Quality.Value;
                }

                if (polygonReductionOptions.OptimizePositionPolicy != null)
                {
                    options.SetOptPosPolicy(polygonReductionOptions.OptimizePositionPolicy);
                }

                // 新旧のパラメータは排他的にする
                // 移行期間終了後に削除予定
                if (polygonReductionOptions.LockOpenEdgeAngle.HasValue)
                {
                    options.LockOpenEdgeAngle = polygonReductionOptions.LockOpenEdgeAngle.Value;
                }
                else if (polygonReductionOptions.OldOpenEdgeLockAngle.HasValue)
                {
                    IfModelPolygonReductionOptimizer.WriteLogLine(Strings.Get("Warning_Depricated_Option", "open-edge-lock-angle"));
                    options.OldLockOpenEdgeAngle = polygonReductionOptions.OldOpenEdgeLockAngle.Value;
                }

                // 新旧のパラメータは排他的にする
                // 移行期間終了後に削除予定
                if (polygonReductionOptions.FeatureFaceAngle.HasValue)
                {
                    options.FeatureFaceAngle = polygonReductionOptions.FeatureFaceAngle.Value;
                }
                else if (polygonReductionOptions.OldFeatureAngle.HasValue)
                {
                    IfModelPolygonReductionOptimizer.WriteLogLine(Strings.Get("Warning_Depricated_Option", "feature-angle"));
                    options.OldFeatureFaceAngle = polygonReductionOptions.OldFeatureAngle.Value;
                }

                if (polygonReductionOptions.UvWeight.HasValue)
                {
                    options.UvWeight = polygonReductionOptions.UvWeight.Value;
                }

                if (polygonReductionOptions.NormalWeight.HasValue)
                {
                    options.NormalWeight = polygonReductionOptions.NormalWeight.Value;
                }

                if (polygonReductionOptions.AnimBlendWeight.HasValue)
                {
                    options.AnimBlendWeight = polygonReductionOptions.AnimBlendWeight.Value;
                }

                if (polygonReductionOptions.ClosedVertexThreshold.HasValue)
                {
                    options.ClosedVertexThreshold = polygonReductionOptions.ClosedVertexThreshold.Value;
                }

                if (polygonReductionOptions.ExcludeShapes != null)
                {
                    // TODO: この辺のパース処理は g3doptcvtrParams.PolygonReductionSubOptions にコンバーターを作ってそこに実装する
                    options.SetExcludeShapeList(polygonReductionOptions.ExcludeShapes);
                }

                // 隠しオプション --polygon_reduction_prop_border_check_by_index には対応しない
                //if (fileOption.HasOption("--polygon_reduction_prop_border_check_by_index"))
                //    options.PropBorderCheckByIndex = fileOption.HasOption("--polygon_reduction_prop_border_check_by_index");
                {
                    if (polygonReductionOptions.TargetRateLodLevel1 == null && polygonReductionOptions.TargetRate == null)
                    {
                        Strings.Throw("Error_PolygonReduction_TargetRateNotSpecified");
                    }

                    float?[] rates = new[] {
                            null,
                            polygonReductionOptions.TargetRateLodLevel1,
                            polygonReductionOptions.TargetRateLodLevel2,
                            polygonReductionOptions.TargetRateLodLevel3,
                            polygonReductionOptions.TargetRateLodLevel4,
                            polygonReductionOptions.TargetRateLodLevel5,
                            polygonReductionOptions.TargetRateLodLevel6,
                            polygonReductionOptions.TargetRateLodLevel7,
                            polygonReductionOptions.TargetRateLodLevel8,
                            polygonReductionOptions.TargetRateLodLevel9,
                        };

                    if (polygonReductionOptions.TargetRate != null)
                    {
                        const int MaxLodLevel = 9;
                        char[] TargetRateSeparator = { ' ' };
                        string[] targetRateStrings = polygonReductionOptions.TargetRate.Split(TargetRateSeparator, StringSplitOptions.RemoveEmptyEntries);
                        if (targetRateStrings.Length > MaxLodLevel)
                        {
                            Strings.Throw("Error_PolygonReduction_TargetRateTooManyLevels", polygonReductionOptions.TargetRate);
                        }

                        if (targetRateStrings.Length == 0 && polygonReductionOptions.TargetRateLodLevel1 == null)
                        {
                            Strings.Throw("Error_PolygonReduction_TargetRateNotSpecified");
                        }

                        int specifiedMaxLevel = targetRateStrings.Length;
                        for (int lodLevel = 1; lodLevel <= specifiedMaxLevel; ++lodLevel)
                        {
                            float rate;
                            string rateString = targetRateStrings[lodLevel - 1];
                            bool isParseSucceeded = float.TryParse(rateString, out rate);
                            if (!isParseSucceeded)
                            {
                                Strings.Throw("Error_PolygonReduction_TargetRateInvalidFormat", rateString);
                            }

                            rates[lodLevel] = rate;
                        }
                    }

                    bool full = true;
                    for (int index = 1; index < 10; index++)
                    {
                        if (rates[index].HasValue)
                        {
                            if (full)
                            {
                                options.TargetRates.Add(index, rates[index].Value);
                            }
                            else
                            {
                                // 途中のレベル指定が抜けている
                                Strings.Throw("Error_PolygonReduction_TargetRateLodLevelSkipped", index);
                            }
                        }
                        else
                        {
                            full = false;
                        }
                    }
                }

                if (polygonReductionOptions.MergeShape)
                {
                    IfModelPolygonReductionOptimizer.WriteLogLine(Strings.Get("Warning_Depricated_Option", "--merge-shape"));
                }

                if (polygonReductionOptions.DisableMergeShape)
                {
                    options.IsMargeShape = (!polygonReductionOptions.DisableMergeShape);
                }

                // --polygonr_reduction="値" はもう使わない
                //this.Optimizers.Add(new IfModelPolygonReductionOptimizer(fileOption.GetOption("--polygon_reduction").Value, options));
                this.Optimizers.Add(new IfModelPolygonReductionOptimizer(polygonReductionOptions.CreateProcessLogArgument(), options));
            }

            // LOD モデル結合
            if (fileOption.UniteLodModel)
            {
                if (fileOption.UniteLodOptions == null)
                {
                    Strings.Throw("Error_OptionNotSpecified", fileOption.GetOptionName(() => fileOption.UniteLodOptions));
                }

                Dictionary<int, string> lodModelFiles = new Dictionary<int, string>();
                var subOptions = fileOption.UniteLodOptions;
                string[] lodModelFilePaths = new[] {
                        null,
                        subOptions.LodLevel1,
                        subOptions.LodLevel2,
                        subOptions.LodLevel3,
                        subOptions.LodLevel4,
                        subOptions.LodLevel5,
                        subOptions.LodLevel6,
                        subOptions.LodLevel7,
                        subOptions.LodLevel8,
                        subOptions.LodLevel9,
                    };

                for (int lodLevel = 1; lodLevel <= IfModelUniteLodModelOptimizer.MaxLodLevel; ++lodLevel)
                {
                    string lodModelFilePath = lodModelFilePaths[lodLevel];
                    if (lodModelFilePath != null)
                    {
                        lodModelFiles.Add(lodLevel, lodModelFilePath);
                    }
                }
                if (lodModelFiles.Count == 0)
                {
                    Strings.Throw("Error_Lod_LodModelNotSpecified");
                }

                this.Optimizers.Add(new IfModelUniteLodModelOptimizer(
                    lodModelFiles, fileOption.UniteLodOptions.CreateProcessLogArgument(), G3dToolUtility.GetXsdBasePath()));
            }
            else
            {
                // もうエラーにしない
                /*
                //LOD モデル結合が指定されていないのに LOD モデル指定があったらエラーにする
                for (int i = 1; i <= IfModelUniteLodModelOptimizer.MaxLodLevel; ++i)
                {
                    string optString = String.Format("--lod_model_level_{0}", i);
                    if (fileOption.HasOption(optString))
                    {
                        Strings.Throw("Error_Lod_OnlyLodModelSpecified");
                    }
                }
                 */
            }

            // プリミティブ最適化
            if (fileOption.OptimizePrimitive)
            {
                // "--sort_submesh_index" はもう使わない
                bool useSortSubmeshIndex = false;
                string value = null;
                var subOption = fileOption.OptimizePrimitiveOptions;
                if (subOption != null)
                {
                    switch (subOption.Mode)
                    {
                        case g3doptcvtrParams.OptimizePrimitiveSubOptions.OptimizeMode.Force:
                            value = "force";
                            break;
                        case g3doptcvtrParams.OptimizePrimitiveSubOptions.OptimizeMode.BruteForce:
                            value = "brute_force";
                            break;
                        case g3doptcvtrParams.OptimizePrimitiveSubOptions.OptimizeMode.Forsyth:
                            value = "forsyth";
                            break;
                        case g3doptcvtrParams.OptimizePrimitiveSubOptions.OptimizeMode.Normal:
                            value = "default";
                            break;
                        default:
                            throw new Exception($"Unexpected OptimizeMode {subOption.Mode}");
                    }
                }
                // サブオプション指定なしの場合、デフォルト挙動として forsyth にする
                else
                {
                    value = "forsyth";
                }
                this.Optimizers.Add(new IfModelPrimitiveOptimizer(
                    value, useSortSubmeshIndex));
            }

            // --optimize_primitive_old はもう使わない
            /*
            if (fileOption.HasOption("--optimize_primitive_old"))
            {
                this.Optimizers.Add(new IfModelPrimitiveOptimizerOld(
                    fileOption.GetOption("--optimize_primitive_old").Value));
            }
            */

            // マテリアルアニメ統合
            CheckUnsupportedOption(fileOption, () => fileOption.Combine);
            CheckUnsupportedOption(fileOption, () => fileOption.CombineOptions);

            // 量子化分析
            if (fileOption.QuantizationAnalysis)
            {
                this.Optimizers.Add(new IfModelQuantizationAnalysisOptimizer());
            }
        }

        //=====================================================================
        // 最適化
        internal override void Optimize()
        {
            // 読み込み
            Read();
            modelType model = this.nw4f_3dif.Item as modelType;

            // モデル圧縮
            foreach (IfModelCompressor compressor in this.Compressors)
            {
                compressor.Compress(model, this.Streams);

                // 最適化ログの追加
                IfProcessLogUtility.Add(this.nw4f_3dif,
                    compressor.Process, compressor.Argument, compressor.GetResult());
            }

            // データの正当性チェック
            Check(model);

            // 修正
            Repair(model);

            // マージ
            Merge(model);

            // モデル最適化
            foreach (IfModelOptimizer optimizer in this.Optimizers)
            {
                optimizer.Optimize(model, this.Streams);

                // 最適化ログの追加
                if (optimizer.EnableProcessLog)
                {
                    IfProcessLogUtility.Add(this.nw4f_3dif,
                        optimizer.Process, optimizer.Argument, optimizer.GetResult());
                }
            }

            IfProcessLogUtility.ToDistinctLatest(this.nw4f_3dif);

            // model_info を更新する
            IfModelUpdateUtility.UpdateModelInfo(model, this.Streams);

            // 書き出し
            Write();
        }

        //---------------------------------------------------------------------
        // 確認
        private void Check(modelType model)
        {
            if (!CheckFile) { return; }

            // ストリームインデックスの正当性を確認する
            string checkLog = string.Empty;
            bool checkResult =
                StreamChecker.CheckStreamIndex(out checkLog, model, this.Streams);
            // チェックが失敗したらログを例外として投げて表示させる
            if (!checkResult)
            {
                throw new Exception(checkLog);
            }
            else
            {
                Console.WriteLine("ok.");
            }
        }

        //---------------------------------------------------------------------
        // 修正
        private void Repair(modelType model)
        {
            if (!RepairFile) { return; }

            StreamUtility.SortStream(model, this.Streams);
        }

        //---------------------------------------------------------------------
        // マージ
        private void Merge(modelType newModel)
        {
            if (this.MergeFilePath == null) { return; }

            this.MergeStopwatch.Start();

            // マージファイルの読み込み
            List<G3dStream> oldStreams = new List<G3dStream>();
            nw4f_3difType oldIf = IfReadUtility.Read(
                oldStreams, this.MergeFilePath, this.XsdBasePath);
            modelType oldModel = oldIf.Item as modelType;

            // マージオプションの設定
            var mergerInfo = new IfModelMerger.IfModelMergerInfo();
            mergerInfo.SamplerMergePriorityToNewModel = this.SamplerMergePriorityToNewModel;
            mergerInfo.TextureSrtMergePriorityToNewModel = this.TextureSrtMergePriorityToNewModel;
            mergerInfo.BoneVisibilityMergePriorityToNewModel = this.BoneVisibilityMergePriorityToNewModel;

            // マージ
            IfModelMergeContext context = new IfModelMergeContext(
                newModel, this.Streams, oldModel, oldStreams);
            context.Setup(true);
            IfModelMerger.Merge(context, mergerInfo);

            // 最適化ログの追加
            IfProcessLogUtility.Add(this.nw4f_3dif, "merge",
                Path.GetFullPath(this.MergeFilePath).Replace('\\', '/'), string.Empty);

            this.MergeStopwatch.Stop();
        }

        //=====================================================================
        // ログの取得
        internal override string GetLog()
        {
            StringBuilder builder = new StringBuilder();
            builder.AppendLine(base.GetLog());
            builder.AppendLine(GetReadLog());

            // モデル圧縮
            foreach (IfModelCompressor compressor in this.Compressors)
            {
                builder.AppendLine("    " + compressor.ToString());
            }

            // マージ
            if (this.MergeFilePath != null)
            {
                builder.AppendLine(Strings.Get("MergeLog",
                    this.MergeStopwatch.ElapsedMilliseconds,
                    this.MergeFilePath));
            }

            // モデル最適化
            foreach (IfModelOptimizer optimizer in this.Optimizers)
            {
                builder.AppendLine("    " + optimizer.ToString());
            }

            builder.Append(GetWriteLog());
            return builder.ToString();
        }

        //---------------------------------------------------------------------
        // メモリ解放
        internal override void Dispose()
        {
            base.Dispose();

            this.Compressors.Clear();
            this.Optimizers.Clear();
        }

        //=====================================================================
        private List<IfModelCompressor> Compressors = new List<IfModelCompressor>();
        private readonly string MergeFilePath;
        private readonly bool SamplerMergePriorityToNewModel = false;
        private readonly bool TextureSrtMergePriorityToNewModel = false;
        private readonly bool BoneVisibilityMergePriorityToNewModel = false;
        private readonly bool RepairFile = false;
        private readonly bool CheckFile = false;
        protected readonly Stopwatch MergeStopwatch = new Stopwatch();
        private List<IfModelOptimizer> Optimizers = new List<IfModelOptimizer>();
    }
}
