﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public class StreamUtility
    {
        // モデルのストリームを並べ替えます。
        public static void SortStream(modelType model, List<G3dStream> streams)
        {
            List<G3dStream> newStreams = new List<G3dStream>();
            Dictionary<int, int> oldNewMap = new Dictionary<int, int>();

            // ストリームの順番を中間ファイル内で参照された順に直す。
            // その上で stream_index のインデックスを振り直す。

            // マテリアルのユーザーデータのストリームを変更する。
            if (model.material_array != null &&
                model.material_array.material != null)
            {
                foreach (materialType mat in model.material_array.material)
                {
                    SortStream(mat.user_data_array, newStreams, streams, oldNewMap);
                }
            }

            // スケルトンのユーザーデータのストリームを変更する。
            foreach (boneType bone in model.skeleton.bone_array.bone)
            {
                SortStream(bone.user_data_array, newStreams, streams, oldNewMap);
            }

            // vtx_attrib.stream_index を変更する。
            if (model.vertex_array != null &&
                model.vertex_array.vertex != null)
            {
                foreach (vertexType vertex in model.vertex_array.vertex)
                {
                    foreach (vtx_attribType vtxAttr in vertex.vtx_attrib_array.vtx_attrib)
                    {
                        SortStream(vtxAttr, newStreams, streams, oldNewMap);
                    }
                }
            }

            // mesh.stream_index を変更する。
            if (model.shape_array != null &&
                model.shape_array.shape != null)
            {
                foreach (shapeType shape in model.shape_array.shape)
                {
                    foreach (meshType mesh in shape.mesh_array.mesh)
                    {
                        SortStream(mesh, newStreams, streams, oldNewMap);
                    }
                }
            }

            // モデルのユーザーデータのストリームを変更する。
            SortStream(model.user_data_array, newStreams, streams, oldNewMap);

            streams.Clear();
            streams.AddRange(newStreams);
        }

        // マテリアルのストリームを並べ替えます。
        public static void SortStream(materialType material, List<G3dStream> streams)
        {
            List<G3dStream> newStreams = new List<G3dStream>();
            Dictionary<int, int> oldNewMap = new Dictionary<int, int>();

            // ストリームの順番を中間ファイル内で参照された順に直す。
            // その上で stream_index のインデックスを振り直す。

            // マテリアルのユーザーデータのストリームを変更する。
            SortStream(material.user_data_array, newStreams, streams, oldNewMap);

            streams.Clear();
            streams.AddRange(newStreams);
        }

        // スケルタルアニメーションのストリームを並べ替えます。
        public static void SortStream(skeletal_animType skeletalAnim, List<G3dStream> streams)
        {
            List<G3dStream> newStreams = new List<G3dStream>();
            Dictionary<int, int> oldNewMap = new Dictionary<int, int>();

            // 使用されているストリームのフラグを false に設定する
            if (skeletalAnim.bone_anim_array != null &&
                skeletalAnim.bone_anim_array.bone_anim != null)
            {
                foreach (bone_animType boneAnim in skeletalAnim.bone_anim_array.bone_anim)
                {
                    foreach (bone_anim_targetType animTarget in boneAnim.bone_anim_target)
                    {
                        SortStream(animTarget.Curve, newStreams, streams, oldNewMap);
                    }
                }
            }

            // ユーザーデータのストリームを並べ替える
            SortStream(skeletalAnim.user_data_array, newStreams, streams, oldNewMap);

            streams.Clear();
            streams.AddRange(newStreams);
        }

        // ボーンビジビリティアニメーションのストリームを並べ替えます。
        public static void SortStream(bone_visibility_animType boneVisAnim, List<G3dStream> streams)
        {
            List<G3dStream> newStreams = new List<G3dStream>();
            Dictionary<int, int> oldNewMap = new Dictionary<int, int>();

            // ストリームの順番を中間ファイル内で参照された順に直す。
            // その上で stream_index のインデックスを振り直す。
            if (boneVisAnim.bone_vis_bone_anim_array != null &&
                boneVisAnim.bone_vis_bone_anim_array.bone_vis_bone_anim != null)
            {
                foreach (bone_vis_bone_animType bone in
                    boneVisAnim.bone_vis_bone_anim_array.bone_vis_bone_anim)
                {
                    SortStream(bone.Curve, newStreams, streams, oldNewMap);
                }
            }

            // ユーザーデータのストリームを並べ替える
            SortStream(boneVisAnim.user_data_array, newStreams, streams, oldNewMap);

            streams.Clear();
            streams.AddRange(newStreams);
        }

        // マテリアルビジビリティアニメーションのストリームを並べ替えます。
        public static void SortStream(mat_visibility_animType matVisAnim, List<G3dStream> streams)
        {
            List<G3dStream> newStreams = new List<G3dStream>();
            Dictionary<int, int> oldNewMap = new Dictionary<int, int>();

            // ストリームの順番を中間ファイル内で参照された順に直す。
            // その上で stream_index のインデックスを振り直す。
            if (matVisAnim.mat_vis_mat_anim_array != null &&
                matVisAnim.mat_vis_mat_anim_array.mat_vis_mat_anim != null)
            {
                foreach (mat_vis_mat_animType matVis in
                    matVisAnim.mat_vis_mat_anim_array.mat_vis_mat_anim)
                {
                    SortStream(matVis.Curve, newStreams, streams, oldNewMap);
                }
            }

            // ユーザーデータのストリームを並べ替える
            SortStream(matVisAnim.user_data_array, newStreams, streams, oldNewMap);

            streams.Clear();
            streams.AddRange(newStreams);
        }

        // シーンアニメーションのストリームを並べ替えます。
        public static void SortStream(scene_animType sceneAnim, List<G3dStream> streams)
        {
            List<G3dStream> newStreams = new List<G3dStream>();
            Dictionary<int, int> oldNewMap = new Dictionary<int, int>();

            // ストリームの順番を中間ファイル内で参照された順に直す。
            // その上で stream_index のインデックスを振り直す。

            // camera_anim のストリームを並び替え
            if (sceneAnim.camera_anim_array != null &&
                sceneAnim.camera_anim_array.camera_anim != null)
            {
                foreach (camera_animType cameraAnim in
                    sceneAnim.camera_anim_array.camera_anim)
                {
                    foreach (camera_anim_targetType cameraTarget in
                        cameraAnim.camera_anim_target ?? Enumerable.Empty<camera_anim_targetType>())
                    {
                        SortStream(cameraTarget.Curve, newStreams, streams, oldNewMap);
                    }

                    // ユーザーデータのストリームを並べ替える
                    SortStream(cameraAnim.user_data_array, newStreams, streams, oldNewMap);
                }
            }

            // light_anim のストリームを並び替え
            if (sceneAnim.light_anim_array != null &&
                sceneAnim.light_anim_array.light_anim != null)
            {
                foreach (light_animType lightAnim in
                    sceneAnim.light_anim_array.light_anim)
                {
                    foreach (light_anim_targetType lightTarget in
                        lightAnim.light_anim_target ?? Enumerable.Empty<light_anim_targetType>())
                    {
                        SortStream(lightTarget.Curve, newStreams, streams, oldNewMap);
                    }

                    // ユーザーデータのストリームを並べ替える
                    SortStream(lightAnim.user_data_array, newStreams, streams, oldNewMap);
                }
            }

            // fog_anim のストリームを並び替え
            if (sceneAnim.fog_anim_array != null &&
                sceneAnim.fog_anim_array.fog_anim != null)
            {
                foreach (fog_animType fogAnim in
                    sceneAnim.fog_anim_array.fog_anim)
                {
                    foreach (fog_anim_targetType fogTarget in
                        fogAnim.fog_anim_target ?? Enumerable.Empty<fog_anim_targetType>())
                    {
                        SortStream(fogTarget.Curve, newStreams, streams, oldNewMap);
                    }

                    // ユーザーデータのストリームを並べ替える
                    SortStream(fogAnim.user_data_array, newStreams, streams, oldNewMap);
                }
            }

            // ユーザーデータのストリームを並べ替える
            SortStream(sceneAnim.user_data_array, newStreams, streams, oldNewMap);

            streams.Clear();
            streams.AddRange(newStreams);
        }

        // マテリアルアニメーションのストリームを並べ替えます。
        public static void SortStream(material_animType materialAnim, List<G3dStream> streams)
        {
            var newStreams = new List<G3dStream>();
            var oldNewMap = new Dictionary<int, int>();

            // per_material の並べ替え
            if (materialAnim.HasMaterials())
            {
                foreach (var material in materialAnim.GetPerMaterialAnims())
                {
                    if (material.HasShaderParameterAnim())
                    {
                        // shader_param_anim のストリームを並び替え
                        var shaderAnims =
                            material.shader_param_anim_array.param_anim
                                .Where(x => x.param_anim_target != null && x.param_anim_target.Length > 0)
                                .SelectMany(x => x.param_anim_target)
                                .Where(x => x.Curve != null);

                        foreach (var paramAnim in shaderAnims)
                        {
                            SortStream(paramAnim.Curve, newStreams, streams, oldNewMap);
                        }
                    }

                    if (material.HasTexturePatternAnim())
                    {
                        // pattern_anim のストリームを並び替え
                        var texPatternAnims =
                            material.tex_pattern_anim_array.pattern_anim
                                .Where(x => x.Curve != null);
                        foreach (var patternAnim in texPatternAnims)
                        {
                            SortStream(patternAnim.Curve, newStreams, streams, oldNewMap);
                        }
                    }

                    if (material.HasVisibilityAnim())
                    {
                        // material_visibility_anim のストリームを並び替え
                        var visibilityAnims =
                            materialAnim.per_material_anim_array.per_material_anim.Where(
                                x => x.material_visibility_anim != null && x.material_visibility_anim.step_curve != null);
                        if (material.material_visibility_anim.step_curve != null)
                        {
                            SortStream(material.material_visibility_anim.step_curve, newStreams, streams, oldNewMap);
                        }
                    }
                }
            }

            // original_per_material の並べ替え
            if (materialAnim.HasOriginalMaterials())
            {
                foreach (var orgMaterial in materialAnim.GetOriginalMaterials())
                {
                    if (orgMaterial.HasOriginalColorAnim())
                    {
                        foreach (var colorAnim in orgMaterial.original_color_anim_array.original_color_anim)
                        {
                            foreach (var target in colorAnim.original_color_anim_target)
                            {
                                SortStream(target.Curve, newStreams, streams, oldNewMap);
                            }
                        }
                    }
                    if (orgMaterial.HasOriginalTexsrtAnim())
                    {
                        foreach (var texsrtAnim in orgMaterial.original_texsrt_anim_array.original_texsrt_anim)
                        {
                            foreach (var target in texsrtAnim.original_texsrt_anim_target)
                            {
                                SortStream(target.Curve, newStreams, streams, oldNewMap);
                            }
                        }
                    }
                }
            }

            // ユーザーデータのストリームを並べ替える
            SortStream(materialAnim.user_data_array, newStreams, streams, oldNewMap);

            streams.Clear();
            streams.AddRange(newStreams);
        }

        // シェーダーパラメーターアニメーションのストリームを並べ替えます。
        public static void SortStream(shader_param_animType shaderParamAnim, List<G3dStream> streams)
        {
            List<G3dStream> newStreams = new List<G3dStream>();
            Dictionary<int, int> oldNewMap = new Dictionary<int, int>();

            // ストリームの順番を中間ファイル内で参照された順に直す。
            // その上で stream_index のインデックスを振り直す。

            // shader_param_mat_anim のストリームを並び替え
            if (shaderParamAnim.shader_param_mat_anim_array != null &&
                shaderParamAnim.shader_param_mat_anim_array.shader_param_mat_anim != null)
            {
                foreach (shader_param_mat_animType matAnim in
                    shaderParamAnim.shader_param_mat_anim_array.shader_param_mat_anim)
                {
                    foreach (param_animType paramAnim in matAnim.param_anim_array.param_anim)
                    {
                        foreach (param_anim_targetType target in paramAnim.param_anim_target)
                        {
                            SortStream(target.Curve, newStreams, streams, oldNewMap);
                        }
                    }
                }
            }

            // original_material_anim のストリームを並び替え
            if (shaderParamAnim.original_material_anim_array != null &&
                shaderParamAnim.original_material_anim_array.original_material_anim != null)
            {
                foreach (original_material_animType orgMatAnim in
                    shaderParamAnim.original_material_anim_array.original_material_anim)
                {
                    if (orgMatAnim.original_color_anim_array != null &&
                        orgMatAnim.original_color_anim_array.original_color_anim != null)
                    {
                        foreach (original_color_animType colorAnim in
                            orgMatAnim.original_color_anim_array.original_color_anim)
                        {
                            foreach (original_color_anim_targetType target in
                                colorAnim.original_color_anim_target)
                            {
                                SortStream(target.Curve, newStreams, streams, oldNewMap);
                            }
                        }
                    }

                    if (orgMatAnim.original_texsrt_anim_array != null &&
                        orgMatAnim.original_texsrt_anim_array.original_texsrt_anim != null)
                    {
                        foreach (original_texsrt_animType texsrtAnim in
                            orgMatAnim.original_texsrt_anim_array.original_texsrt_anim)
                        {
                            foreach (original_texsrt_anim_targetType target in
                                texsrtAnim.original_texsrt_anim_target)
                            {
                                SortStream(target.Curve, newStreams, streams, oldNewMap);
                            }
                        }
                    }
                }
            }

            // ユーザーデータのストリームを並べ替える
            SortStream(shaderParamAnim.user_data_array, newStreams, streams, oldNewMap);

            streams.Clear();
            streams.AddRange(newStreams);
        }

        // シェイプアニメーションのストリームを並べ替えます。
        public static void SortStream(shape_animType shapeAnim, List<G3dStream> streams)
        {
            List<G3dStream> newStreams = new List<G3dStream>();
            Dictionary<int, int> oldNewMap = new Dictionary<int, int>();

            // ストリームの順番を中間ファイル内で参照された順に直す。
            // その上で stream_index のインデックスを振り直す。

            // shape_anim のストリームを並び替え
            if (shapeAnim.vertex_shape_anim_array != null &&
                shapeAnim.vertex_shape_anim_array.vertex_shape_anim != null)
            {
                foreach (vertex_shape_animType vertexShapeAnim in
                    shapeAnim.vertex_shape_anim_array.vertex_shape_anim)
                {
                    foreach (shape_anim_targetType target in
                        vertexShapeAnim.shape_anim_target)
                    {
                        SortStream(target.Curve, newStreams, streams, oldNewMap);
                    }
                }
            }

            // ユーザーデータのストリームを並べ替える
            SortStream(shapeAnim.user_data_array, newStreams, streams, oldNewMap);

            streams.Clear();
            streams.AddRange(newStreams);
        }

        // テクスチャパターンアニメーションのストリームを並べ替えます。
        public static void SortStream(tex_pattern_animType texPatternAnim, List<G3dStream> streams)
        {
            List<G3dStream> newStreams = new List<G3dStream>();
            Dictionary<int, int> oldNewMap = new Dictionary<int, int>();

            // ストリームの順番を中間ファイル内で参照された順に直す。
            // その上で stream_index のインデックスを振り直す。
            if (texPatternAnim.tex_pattern_mat_anim_array != null &&
                texPatternAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim != null)
            {
                foreach (tex_pattern_mat_animType matAnim in
                    texPatternAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim)
                {
                    foreach (pattern_anim_targetType target in matAnim.pattern_anim_target)
                    {
                        SortStream(target.Curve, newStreams, streams, oldNewMap);
                    }
                }
            }

            // ユーザーデータのストリームを並べ替える
            SortStream(texPatternAnim.user_data_array, newStreams, streams, oldNewMap);

            streams.Clear();
            streams.AddRange(newStreams);
        }

        // テクスチャのストリームを並べ替えます。
        public static void SortStream(textureType texture, List<G3dStream> streams)
        {
            List<G3dStream> newStreams = new List<G3dStream>();
            Dictionary<int, int> oldNewMap = new Dictionary<int, int>();

            // ストリームの順番を中間ファイル内で参照された順に直す。
            // その上で stream_index のインデックスを振り直す。
            SortStream(texture.texture_info, newStreams, streams, oldNewMap);

            if (texture.original_image_array != null &&
                texture.original_image_array.original_image != null)
            {
                foreach (original_imageType image in
                    texture.original_image_array.original_image)
                {
                    SortStream(image, newStreams, streams, oldNewMap);
                }
            }

            // ユーザーデータのストリームを並べ替える
            SortStream(texture.user_data_array, newStreams, streams, oldNewMap);

            streams.Clear();
            streams.AddRange(newStreams);
        }

        // ユーザーデータのストリームを並べ替えます。
        private static void SortStream(
            user_data_arrayType userDataArray,
            List<G3dStream> newStreams,
            List<G3dStream> oldStreams,
            Dictionary<int, int> oldNewMap)
        {
            if (userDataArray == null || userDataArray.user_data == null) { return; }

            foreach (user_dataType userData in userDataArray.user_data)
            {
                SortStream(userData.Item as user_streamType, newStreams, oldStreams, oldNewMap);
            }
        }

        // ストリームを並び替えます。
        private static void SortStream(IG3dStreamReference streamRef, List<G3dStream> newStreams,
           List<G3dStream> oldStreams, Dictionary<int, int> oldNewMap)
        {
            if (streamRef != null)
            {
                int oldIndex = streamRef.stream_index;
                if (oldIndex >= 0)
                {
                    if (oldNewMap.ContainsKey(oldIndex))
                    {
                        streamRef.stream_index = oldNewMap[oldIndex];
                    }
                    else
                    {
                        streamRef.stream_index = newStreams.Count;
                        newStreams.Add(oldStreams[oldIndex]);
                        oldNewMap.Add(oldIndex, streamRef.stream_index);
                    }
                }
            }
        }
    }
}
