﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // テクスチャパターンアニメーションの量子化分析
    public class IfTexPatternAnimQuantizationAnalysisOptimizer
        : IfTexPatternAnimOptimizer
    {
        // コンストラクタ
        public IfTexPatternAnimQuantizationAnalysisOptimizer() :
            base("IfTexPatternAnimQuantizationAnalysisOptimizer_Log") { }

        // プロセス
        public override string Process
        {
            get { return "quantization_analysis"; }
        }

        // 結果の取得
        public override string GetResult()
        {
            return string.Empty;
        }

        private int FrameSize;
        private int QuantizedFrameSize;

        // 最適化
        protected override void Optimize()
        {
            if (this.Target.tex_pattern_mat_anim_array == null) { return; }

            tex_pattern_animType texPatAnim = this.Target;
            tex_pattern_anim_infoType texPatAnimInfo = texPatAnim.tex_pattern_anim_info;
            IDictionary<pattern_anim_targetType, IfAnimCurve> curveDic =
                new Dictionary<pattern_anim_targetType, IfAnimCurve>();

            // カーブデータを構築する
            foreach (tex_pattern_mat_animType patAnim in texPatAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim)
            {
                if (patAnim.pattern_anim_target == null) { continue; }

                foreach (pattern_anim_targetType target in patAnim.pattern_anim_target)
                {
                    if (target.step_curve != null)
                    {
                        IfAnimCurve animCurve = new IfAnimCurve();
                        animCurve.curveType = IfAnimCurve.CurveType.Int;
                        animCurve.BuildIfAnimCurve(target.step_curve, false, this.Streams);
                        curveDic.Add(target, animCurve);
                    }
                }
            }

            // frame の量子化を評価して type を選択する。
            foreach (var curve in curveDic)
            {
                IfQuantizationAnalysisResult result =
                    IfTexPatternAnimQuantizationAnalysis.Analyse(
                        texPatAnimInfo,
                        curve.Value,
                        false);

                this.EnableProcessLog |= !result.skipped;

                ApplyQuantize(curve.Key, result);
            }
        }

        /// <summary>
        /// 量子化タイプを適応します。
        /// </summary>
        private void ApplyQuantize(pattern_anim_targetType target, IfQuantizationAnalysisResult analysisResult)
        {
            curve_frame_typeType frameType = curve_frame_typeType.none;
            curve_key_typeType keyType = curve_key_typeType.none;
            int frameCount;

            step_curveType stepCurve = target.step_curve;
            stepCurve.frame_type = analysisResult.frameType;
            stepCurve.key_type = analysisResult.keyType;
            stepCurve.scale = analysisResult.scale;
            stepCurve.offset = analysisResult.offset;

            frameType = stepCurve.frame_type;
            keyType = stepCurve.key_type;
            frameCount = stepCurve.count;

            this.FrameSize += frameCount * sizeof(float);
            switch (frameType)
            {
                case curve_frame_typeType.none:
                case curve_frame_typeType.frame32:
                    this.QuantizedFrameSize += frameCount * sizeof(float);
                    break;
                case curve_frame_typeType.frame16:
                    this.QuantizedFrameSize += frameCount * sizeof(ushort);
                    break;
                case curve_frame_typeType.frame8:
                    this.QuantizedFrameSize += frameCount * sizeof(byte);
                    break;
            }
        }
    }
}
