﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    public class IfSceneAnimQuantizationAnalysis
    {
        // カメラアニメーションの量子化分析
        public static IfQuantizationAnalysisResult Analyse(
            scene_anim_infoType sceneAnimInfo,
            camera_anim_target_targetType type,
            IfAnimCurve curve,
            bool forceAnalyse)
        {
            IfQuantizationAnalysisResult result = new IfQuantizationAnalysisResult();

            // 量子化判定を途中で打ち切っても元の量子化設定が反映されるように
            // 最初の段階の設定を結果にコピーしておく。
            result.frameType = curve.frameType;
            result.keyType = curve.keyType;
            result.scale = curve.scale;
            result.offset = curve.offset;
            result.skipped = true;

            // keytype が none 以外の場合は key 量子化の判定を行わない。
            if (!forceAnalyse &&
                (curve.keyType != curve_key_typeType.none || !curve.HasCurve()))
            {
                return result;
            }
            result.skipped = false;
            curve.EvalQuantizeFrame(0.5f / sceneAnimInfo.frame_resolution);
            result.frameType = curve.frameType;

            // カーブを量子化します。
            curve.QuantizeCurve();

            if (type == camera_anim_target_targetType.rotate_x ||
                type == camera_anim_target_targetType.rotate_y ||
                type == camera_anim_target_targetType.rotate_z)
            {
                // 回転量の誤差視認性はアニメーション全体における変化量との依存が
                // 小さいと考えられるため、量子化誤差チェックには絶対誤差を用いる。
                bool useAbsEval = true;
                float tolerance_rotate =
                    FloatUtility.DegreeToRadian(sceneAnimInfo.quantize_tolerance_rotate);
                curve.EvalQuantizeFloat(
                    tolerance_rotate,
                    useAbsEval,
                    sceneAnimInfo.frame_resolution);
            }
            else if (type == camera_anim_target_targetType.aim_x ||
                     type == camera_anim_target_targetType.aim_y ||
                     type == camera_anim_target_targetType.aim_z ||
                     type == camera_anim_target_targetType.position_x ||
                     type == camera_anim_target_targetType.position_y ||
                     type == camera_anim_target_targetType.position_z)
            {
                bool useAbsEval = false;
                curve.EvalQuantizeFloat(
                    sceneAnimInfo.quantize_tolerance_translate,
                    useAbsEval,
                    sceneAnimInfo.frame_resolution);
            }

            result.keyType = curve.keyType;
            result.scale = curve.scale;
            result.offset = curve.offset;

            return result;
        }

        // ライトアニメーションの量子化分析
        public static IfQuantizationAnalysisResult Analyse(
            scene_anim_infoType sceneAnimInfo,
            light_anim_target_targetType type,
            IfAnimCurve curve,
            bool forceAnalyse)
        {
            IfQuantizationAnalysisResult result = new IfQuantizationAnalysisResult();

            // 量子化判定を途中で打ち切っても元の量子化設定が反映されるように
            // 最初の段階の設定を結果にコピーしておく。
            result.frameType = curve.frameType;
            result.keyType = curve.keyType;
            result.scale = curve.scale;
            result.offset = curve.offset;
            result.skipped = true;

            // keytype が none 以外の場合は key 量子化の判定を行わない。
            if (!forceAnalyse &&
                (curve.keyType != curve_key_typeType.none || !curve.HasCurve()))
            {
                return result;
            }
            result.skipped = false;
            curve.EvalQuantizeFrame(0.5f / sceneAnimInfo.frame_resolution);
            result.frameType = curve.frameType;

            // カーブを量子化します。
            curve.QuantizeCurve();

            if (type == light_anim_target_targetType.aim_x ||
                type == light_anim_target_targetType.aim_y ||
                type == light_anim_target_targetType.aim_z ||
                type == light_anim_target_targetType.position_x ||
                type == light_anim_target_targetType.position_y ||
                type == light_anim_target_targetType.position_z)
            {
                bool useAbsEval = false;
                curve.EvalQuantizeFloat(
                    sceneAnimInfo.quantize_tolerance_translate,
                    useAbsEval,
                    sceneAnimInfo.frame_resolution);

                result.keyType = curve.keyType;
                result.scale = curve.scale;
                result.offset = curve.offset;
            }

            return result;
        }

        // フォグアニメーションの量子化分析
        public static IfQuantizationAnalysisResult Analyse(
            scene_anim_infoType sceneAnimInfo,
            fog_anim_target_targetType type,
            IfAnimCurve curve,
            bool forceAnalyse)
        {
            IfQuantizationAnalysisResult result = new IfQuantizationAnalysisResult();

            // 量子化判定を途中で打ち切っても元の量子化設定が反映されるように
            // 最初の段階の設定を結果にコピーしておく。
            result.frameType = curve.frameType;
            result.keyType = curve.keyType;
            result.scale = curve.scale;
            result.offset = curve.offset;
            result.skipped = true;

            // keytype が none 以外の場合は key 量子化の判定を行わない。
            if (!forceAnalyse &&
                (curve.keyType != curve_key_typeType.none || !curve.HasCurve()))
            {
                return result;
            }
            result.skipped = false;
            curve.EvalQuantizeFrame(0.5f / sceneAnimInfo.frame_resolution);
            result.frameType = curve.frameType;

            // key は量子化しない
            return result;
        }
    }
}
