﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections;
using System.Collections.Generic;
using System.Linq;

namespace nw.g3d.iflib
{
    /// <summary>
    /// Tipsify方式のポリゴン最適化クラスです。
    /// </summary>
    public class IfOptimizeAlgorithmTipsify : IfOptimizeAlgorithm
    {
        private const int AdjacencyMax = short.MaxValue;
        private const int DeadEndStackSize = 32768;
        /// <summary>
        /// トライアングルリストの情報を事前に解析します。
        /// </summary>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        public void Analyze(IfPolygonMesh polygonMesh)
        {
            polygonMesh.AnalyzeVertices();
        }

        /// <summary>
        /// トライアングルリストの頂点キャッシュ最適化を行います。
        /// </summary>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <param name="optimizedPolygonPrimitive">結果が格納されます。</param>
        public void Optimize(
            IfPolygonMesh polygonMesh,
            IList<IfPolygonPrimitive> optimizedPolygonPrimitive,
            int startIndex)
        {
            this.Tipsify(polygonMesh, optimizedPolygonPrimitive, IfPseudoVertexCacheFIFO.CafeCacheSize, startIndex);
        }

        /// <summary>
        /// Tipsify方式の頂点キャッシュ最適化を行います。
        /// </summary>
        /// <param name="polygonMesh">ポリゴンメッシュです。</param>
        /// <param name="optimizedPolygonPrimitive">結果が格納されます。</param>
        /// <param name="cacheSize">頂点キャッシュのサイズです。</param>
        private void Tipsify(IfPolygonMesh polygonMesh, IList<IfPolygonPrimitive> optimizedPolygonPrimitive, int cacheSize, int startIndex)
        {
            CircularList<int> deadEndStack
                = new CircularList<int>(DeadEndStackSize);

            // 頂点ごとの情報
            int[] liveTriangles = polygonMesh.Occurrences.ToArray();
            int[] cacheTimeStamps = new int[polygonMesh.VerticesNumber];

            int cacheStamp = cacheSize + 1;

            // 次の頂点候補のリスト
            List<int> nextCandidates = new List<int>(polygonMesh.MaxAdjacency * IfTriangle.VertexCount);

            int index = 0;

            // 任意の頂点からスタート
            int f = startIndex;
            IfPolygonPrimitive polygonPrimitive = new IfPolygonPrimitive();

            while (f >= 0)
            {
                foreach (int triangleIndex in polygonMesh.Adjacencys[f])
                {
                    // 任意の頂点(頂点番号f)を持つトライアングルを探して、
                    // まだ登録前であれば、最適化後リストに登録。
                    if (!polygonMesh.AnalyzedTriangles[triangleIndex].IsUsed)
                    {
                        IfTriangle triangle = polygonMesh.AnalyzedTriangles[triangleIndex].Triangle;
                        polygonPrimitive.Triangles.Add(triangle);

                        // 登録したトライアングルの頂点をスタックに追加
                        foreach (int v in triangle.PositionIds)
                        {
                            deadEndStack.Add(v);

                            // 頂点情報更新
                            --liveTriangles[v];
                            if (0 < liveTriangles[v])
                            {
                                // 次に選択する頂点の候補を追加する。
                                nextCandidates.Add(v);
                            }

                            if ((cacheStamp - cacheTimeStamps[v]) > cacheSize)
                            {
                                // キャッシュに入っていなかったらキャッシュに入れる。
                                // キャッシュ登録番号を頂点に保存
                                cacheTimeStamps[v] = cacheStamp;
                                ++cacheStamp;
                            }
                        }

                        polygonMesh.AnalyzedTriangles[triangleIndex].IsUsed = true;
                    }
                }

                f = this.GetNextVertex(
                    cacheSize,
                    nextCandidates,
                    cacheTimeStamps,
                    cacheStamp,
                    liveTriangles,
                    deadEndStack);

                if (f == -1)
                {
                    optimizedPolygonPrimitive.Add(polygonPrimitive);
                    polygonPrimitive = new IfPolygonPrimitive();

                    // 行き止まり頂点スタックにもない場合は順番に空いている頂点を探す。
                    // 擬似コードでは"SkipDeadEnd"の中にあった処理です。
                    // メンテナンス性の観点からここに移動しました。
                    while ((index + 1) < polygonMesh.VerticesNumber)
                    {
                        ++index;
                        if (liveTriangles[index] > 0)
                        {
                            f = index;
                            break;
                        }
                    }
                }

                nextCandidates.Clear();
            }

            optimizedPolygonPrimitive.Add(polygonPrimitive);
        }

        /// <summary>
        /// 次の頂点候補をチェックして、一番古いキャッシュに入っている頂点を返します。
        /// </summary>
        /// <param name="cacheSize">キャッシュサイズです。</param>
        /// <param name="nextCandidates">次候補群です。</param>
        /// <param name="cacheTime">cacheTime です。</param>
        /// <param name="cacheStamp">cacheStamp です。</param>
        /// <param name="liveTriangles">liveTriangles です。</param>
        /// <param name="deadEndStack">スタックしてある「行き止まり」（デッドエンド）頂点です。</param>
        /// <returns>一番古いキャッシュに入っている頂点を返します。</returns>
        private int GetNextVertex(
            int cacheSize,
            List<int> nextCandidates,
            int[] cacheTime,
            int cacheStamp,
            int[] liveTriangles,
            CircularList<int> deadEndStack)
        {
            int n = -1;
            int m = -1;

            foreach (int v in nextCandidates)
            {
                int p = 0;

                if (cacheStamp - cacheTime[v] + (2 * liveTriangles[v]) <= cacheSize)
                {
                    p = cacheStamp - cacheTime[v];
                }

                if (p > m)
                {
                    m = p;
                    n = v;
                }
            }

            if (n == -1)
            {
                // キャッシュに入っている頂点がなくなった場合は行き止まりスタックをチェック
                n = this.SkipDeadEnd(liveTriangles, deadEndStack);
            }

            return n;
        }

        /// <summary>
        /// スタックしてある「行き止まり」（デッドエンド）頂点の中から
        /// 一番最近に計算した頂点を返します。
        /// </summary>
        /// <param name="liveTriangles">liveTrianglesです。</param>
        /// <param name="deadEndStack">スタックしてある「行き止まり」（デッドエンド）頂点です。</param>
        /// <returns>一番最近に計算した頂点を返します。</returns>
        private int SkipDeadEnd(int[] liveTriangles, CircularList<int> deadEndStack)
        {
            int f = -1;

            while (!deadEndStack.IsEmpty())
            {
                int d = deadEndStack.Pop();

                if (liveTriangles[d] > 0)
                {
                    f = d;
                    break;
                }
            }

            return f;
        }

        /// <summary>
        /// 環状リストを表すクラスです。
        /// <para>
        /// 例えば、サイズを3とした時、初期状態から1,2,3を追加(Add)して、
        /// さらに4を追加した場合の配列メモリの状態は下記の通りです。
        /// </para>
        /// <code>
        /// Add(1)-> |1| -Add(2)-> |1|2| -Add(3)-> |1|2|3| -Add(4)-> |2|3|4|
        /// </code>
        /// <para>
        /// サイズを超えて追加すると、先頭の要素が削除されます。
        /// 配列用のメモリ確保は最初のサイズ指定時のみ行われます。
        /// Add関数などの内部で配列メモリを確保することはありません。
        /// </para>
        /// </summary>
        /// <typeparam name="T">リスト要素の型</typeparam>
        public class CircularList<T> : IEnumerable<T>
        {
            private List<T> bufferList = new List<T>();
            private int position = 0;
            private int start = 0;

            /// <summary>
            /// バッファのサイズを指定してオブジェクトを構築します。
            /// </summary>
            /// <param name="size">バッファのサイズ</param>
            public CircularList(int size)
            {
                // 先頭識別の末端用として、一つ余分に確保する。
                int s = size + 1;
                this.bufferList.Capacity = s;
                for (int i = 0; i < s; ++i)
                {
                    this.bufferList.Add(default(T));
                }
            }

            /// <summary>
            /// 配列であるかのようなアクセス方法を提供します。
            /// </summary>
            /// <param name="index">インデックスです。</param>
            /// <returns>要素を返します。</returns>
            public T this[int index]
            {
                get
                {
                    if (index < 0 || this.GetNumEntries() <= index)
                    {
                        throw new System.IndexOutOfRangeException();
                    }

                    int i = this.start + index + 1;
                    if (this.bufferList.Count <= i)
                    {
                        i = i % this.bufferList.Count;
                    }

                    return this.bufferList[i];
                }

                set
                {
                    if (index < 0 || this.GetNumEntries() <= index)
                    {
                        throw new System.IndexOutOfRangeException();
                    }

                    int i = this.start + index + 1;
                    if (this.bufferList.Count <= i)
                    {
                        i = i % this.bufferList.Count;
                    }

                    this.bufferList[i] = value;
                }
            }

            /// <summary>
            /// 列挙子を取得します。
            /// </summary>
            /// <returns>列挙子を帰します。</returns>
            public IEnumerator<T> GetEnumerator()
            {
                int end = this.position + 1;

                if (end == this.bufferList.Count)
                {
                    end = 0;
                }

                for (int i = this.start + 1; i != this.position; ++i)
                {
                    if (i == this.bufferList.Count)
                    {
                        i = 0;
                    }

                    yield return this.bufferList[i];
                }
            }

            /// <summary>
            /// 列挙子を取得します。
            /// </summary>
            /// <returns>列挙子を帰します。</returns>
            IEnumerator IEnumerable.GetEnumerator()
            {
                throw new System.NotImplementedException();
            }

            /// <summary>
            /// 要素を末尾に追加します。
            /// </summary>
            /// <param name="x">追加する要素です。</param>
            public void Add(T x)
            {
                ++this.position;
                if (this.bufferList.Count <= this.position)
                {
                    this.position = 0;
                }

                this.bufferList[this.position] = x;
                if (this.position == this.start)
                {
                    this.start = this.position + 1;
                    if (this.bufferList.Count <= this.start)
                    {
                        this.start = 0;
                    }
                }
            }

            /// <summary>
            /// 末尾の要素の値を返し、リストから削除します。
            /// </summary>
            /// <returns>末尾の要素の値を返します。</returns>
            public T Pop()
            {
                if (this.IsEmpty())
                {
                    throw new System.Exception("This is empty!");
                }

                T last = default(T);
                if (this.position != this.start)
                {
                    last = this.bufferList[this.position];
                    --this.position;
                    if (this.position < 0)
                    {
                        this.position = this.bufferList.Count - 1;
                    }
                }

                return last;
            }

            /// <summary>
            /// 全ての要素を消去します。
            /// メモリの解放は行いません。
            /// </summary>
            public void Clear()
            {
                this.position = 0;
                this.start = 0;
            }

            /// <summary>
            /// 要素の数を取得します。
            /// </summary>
            /// <returns>要素の数を返します。</returns>
            public int GetNumEntries()
            {
                int num = 0;

                if (this.start <= this.position)
                {
                    num = this.position - this.start;
                }
                else
                {
                    num = this.position + (this.bufferList.Count - this.start);
                }

                return num;
            }

            /// <summary>
            /// 要素がないか調べます。
            /// </summary>
            /// <returns>要素がないとtrueを返します。</returns>
            public bool IsEmpty()
            {
                bool isEmpty = false;

                if (this.position == this.start)
                {
                    isEmpty = true;
                }

                return isEmpty;
            }
        }
    }
}
