﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;
using System.Collections.ObjectModel;

namespace nw.g3d.iflib
{
    internal class IfSubmeshPrimitiveOptimizer : ISubmeshPrimitiveOptimizer
    {
        // コンストラクタ
        internal IfSubmeshPrimitiveOptimizer(
            modelType model, List<G3dStream> streams, shapeType shape,
            List<int> indexStream, submeshType submesh,
            string mode)
        {
            this.model = model;
            this.Streams = streams;
            this.shape = shape;
            this.IndexStream = indexStream;
            this.submesh = submesh;
            this.Mode = mode;
        }

        //---------------------------------------------------------------------
        // 初期化
        public void Initialize()
        {
            // 最適化対象のインデックスストリームを取り出す
            int[] source = new int[this.submesh.count];
            for (int i = 0; i < this.submesh.count; i++)
            {
                source[i] = this.IndexStream[i + submesh.offset];
            }
            this.Source = Array.AsReadOnly(source);

            this.TriangleCount = this.Source.Count / 3;
            this.SourceProcessingVertexCount = G3dProcessingVertexCounter.CountTrianglesNX(this.Source);
            this.ProcessingVertexCount = this.SourceProcessingVertexCount;
        }

        //---------------------------------------------------------------------
        // 最適化
        public void Optimize()
        {
            int[] result = null;
            // 大量に並列化するとメモリがあふれるので、Default のみ並列化する
            if (this.Mode != "force" && this.Mode != "brute_force" && this.Mode != "forsyth")
            {
                List<IfOptimizePrimitiveAlgorithm> algorithms = new List<IfOptimizePrimitiveAlgorithm>();
                algorithms.AddRange(BuildAlgorithms());
                G3dParallel.ForEach(algorithms,
                    delegate(IfOptimizePrimitiveAlgorithm algorithm)
                    {
                        algorithm.Optimize(this.ProcessingVertexCount);
                    });
                foreach (var algorithm in algorithms)
                {
                    if (algorithm.ProcessingVertexCount < this.ProcessingVertexCount)
                    {
                        result = algorithm.Result;
                        this.ProcessingVertexCount = algorithm.ProcessingVertexCount;
                    }
                }
            }
            else if (this.Mode == "forsyth")
            {
                IfOptimizePrimitiveAlgorithmForsyth forsyth = new IfOptimizePrimitiveAlgorithmForsyth(this.Source);
                forsyth.Optimize(this.ProcessingVertexCount);
                result = forsyth.Result;
                this.ProcessingVertexCount = forsyth.ProcessingVertexCount;
            }
            else
            {
                // 複数ファイルと複数シェイプによりスレッドが大量になっている可能性がある
                // メモリ対策のためアルゴリズムの列挙パターンを保持しない
                foreach (var algorithm in BuildAlgorithms())
                {
                    //Stopwatch sw = new Stopwatch();
                    //sw.Start();
                    algorithm.Optimize(this.ProcessingVertexCount);
                    if (algorithm.ProcessingVertexCount < this.ProcessingVertexCount)
                    {
                        result = algorithm.Result;
                        this.ProcessingVertexCount = algorithm.ProcessingVertexCount;
                    }
                    //sw.Stop();
                    //Debug.WriteLine("{0,5}ms {1,4} {2} {3}",
                    //    sw.ElapsedMilliseconds, i, algorithm, minProcessingVertexCount);
                }
            }

            this.isSuccess = result != null;
            ApplyResult(result);
        }

        // アルゴリズムの構築
        private IEnumerable<IfOptimizePrimitiveAlgorithm> BuildAlgorithms()
        {
            if (this.Mode == "force")
            {
                return IfOptimizePrimitiveAlgorithmCombo.EnumerateForce(this.Source);
            }
            else if (this.Mode == "brute_force")
            {
                return IfOptimizePrimitiveAlgorithmCombo.EnumerateBruteForce(this.Source);
            }
            return IfOptimizePrimitiveAlgorithmCombo.EnumerateDefault(this.Source);
        }

        // 結果の適用
        private void ApplyResult(int[] result)
        {
            // インデックスストリームへの最適化結果書き込み
            if (IsSuccess())
            {
                for (int i = 0; i < this.submesh.count; i++)
                {
                    this.IndexStream[i + submesh.offset] = result[i];
                }
            }

            this.Source = null;
        }

        //---------------------------------------------------------------------
        // 最適化が成功したか
        public bool IsSuccess() { return (this.isSuccess); }

        //---------------------------------------------------------------------
        private readonly modelType model;
        private readonly List<G3dStream> Streams;
        private readonly shapeType shape;
        private readonly List<int> IndexStream;
        private readonly submeshType submesh;
        private readonly string Mode;

        private ReadOnlyCollection<int> Source;
        public int TriangleCount { get; private set; }
        public int SourceProcessingVertexCount { get; private set; }

        public int ProcessingVertexCount { get; private set; }
        private bool isSuccess = false;
    }
}
