﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // テクスチャパターンアニメーションマージャ
    public static class IfTexPatternAnimMerger
    {
        // マージ用の情報
        public class IfTexPatternAnimMergerInfo
        {
            public IEnumerable<IfMergeSrcDstPair> MaterialNamePairTable{ get; set; }
            public IEnumerable<IfMergeSrcDstPair> SamplerNamePairTable{ get; set; }
        }

        // マージ
        public static void Merge(
            tex_pattern_animType newTexPatternAnim,
            List<G3dStream> newStream,
            tex_pattern_animType oldTexPatternAnim,
            List<G3dStream> oldStream)
        {
            Merge(newTexPatternAnim, newStream, oldTexPatternAnim, oldStream, null);
        }

        // オプション付きマージ
        public static void Merge(
            tex_pattern_animType newTexPatternAnim,
            List<G3dStream> newStream,
            tex_pattern_animType oldTexPatternAnim,
            List<G3dStream> oldStream,
            IfTexPatternAnimMergerInfo info)
        {
            // <user_data_array> <tool_data> <user_tool_data> <comment>
            IfMergeUtility.MergeRootObject(
                newTexPatternAnim, oldTexPatternAnim,
                newStream, oldStream);

            // マージファイルにカーブが存在していない時は処理を行う必要がない
            if (oldTexPatternAnim.tex_pattern_mat_anim_array == null) { return; }

            // テクスチャパターンがなく、カーブのみが存在するデータが入力されたときはエラーとする。
            if (newTexPatternAnim.tex_pattern_array == null &&
                newTexPatternAnim.tex_pattern_mat_anim_array != null)
            {
                throw new Exception("<tex_pattern> doesn't exist in the input file.");
            }
            if (oldTexPatternAnim.tex_pattern_array == null &&
                oldTexPatternAnim.tex_pattern_mat_anim_array != null)
            {
                throw new Exception("<tex_pattern> doesn't exist in the merge file.");
            }

            // テクスチャパターンアニメーションの存在を確認する。
            bool hasNewTexPatternAnim = HasTexturePatternAnim(newTexPatternAnim);
            bool hasOldTexPatternAnim = HasTexturePatternAnim(oldTexPatternAnim);

            if (hasNewTexPatternAnim && hasOldTexPatternAnim)
            {
                // oldTexPatternAnim から newTexPatternAnim にコピーしたテクスチャパターンの
                // 新しいインデックスを記録した配列。
                int[] newTexPatternIdx;

                // <tex_pattern_mat_anim> のテーブルを作成する
                // キーとして oldTexPatternAnim 内の <tex_pattern_mat_anim> を、
                // 値としてキーと同じ mat_name を持つ newTexPatternAnim 内の <tex_pattern_mat_anim> を格納する。
                Dictionary<tex_pattern_mat_animType, tex_pattern_mat_animType> texPatternMatTable =
                    new Dictionary<tex_pattern_mat_animType, tex_pattern_mat_animType>();
                IfMergeUtility.SetupTableByFunc<tex_pattern_mat_animType>(
                    texPatternMatTable,
                    oldTexPatternAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim,
                    newTexPatternAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim,
                    delegate(tex_pattern_mat_animType newObject, tex_pattern_mat_animType oldObject)
                    {
                        return (newObject.mat_name == oldObject.mat_name);
                    });

                // テクスチャパターンをマージする
                MergeTexturePattern(
                    newTexPatternAnim,
                    oldTexPatternAnim,
                    texPatternMatTable,
                    oldStream,
                    out newTexPatternIdx);

                // <tex_pattern_mat_anim> をマージする。
                // <tex_pattern_mat_anim> mat_name と <pattern_anim_target> sampler_name
                // が重複したら newTexPatternAnim を優先する。
                List<tex_pattern_mat_animType> newTexPatternMatAnimArray =
                    new List<tex_pattern_mat_animType>();
                newTexPatternMatAnimArray.AddRange(
                    newTexPatternAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim);
                foreach (tex_pattern_mat_animType oldTexPatternMatAnim in
                    oldTexPatternAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim)
                {
                    tex_pattern_mat_animType newTexPatternMatAnim;
                    if (!texPatternMatTable.ContainsKey(oldTexPatternMatAnim))
                    {
                        // newTexPatternAnim に oldTexPatternAnim の <tex_pattern_mat_anim>
                        // が存在していない場合コピーする。
                        newTexPatternMatAnim = new tex_pattern_mat_animType();
                        newTexPatternMatAnimArray.Add(newTexPatternMatAnim);
                    }
                    else
                    {
                        newTexPatternMatAnim = texPatternMatTable[oldTexPatternMatAnim];
                    }

                    MergeTexPatternAnim(
                        newTexPatternMatAnim,
                        newStream,
                        oldTexPatternMatAnim,
                        oldStream,
                        newTexPatternIdx);
                }

                // <tex_pattern_mat_anim> を mat_name でソートし、インデックスを振り直す。
                newTexPatternMatAnimArray.Sort(
                    delegate (tex_pattern_mat_animType lhs, tex_pattern_mat_animType rhs)
                    {
                        return string.CompareOrdinal(lhs.mat_name, rhs.mat_name);
                    });
                for (int i = 0; i < newTexPatternMatAnimArray.Count; i++)
                {
                    newTexPatternMatAnimArray[i].index = i;
                }

                // <tex_pattern_mat_anim_array> を置きかえる。
                newTexPatternAnim.tex_pattern_mat_anim_array.length =
                    newTexPatternMatAnimArray.Count;
                newTexPatternAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim =
                    newTexPatternMatAnimArray.ToArray();

                // 中間ファイルの出現順にストリームをソートする。
                StreamUtility.SortStream(newTexPatternAnim, newStream);
            }
            else if (hasOldTexPatternAnim)
            {
                // oldTexPatternAnim にのみ <tex_pattern> が存在している場合、
                // oldTexPatternAnim の <tex_pattern> をそのままコピーする。

                newTexPatternAnim.tex_pattern_array =
                    oldTexPatternAnim.tex_pattern_array;
                newTexPatternAnim.tex_pattern_mat_anim_array =
                    oldTexPatternAnim.tex_pattern_mat_anim_array;

                newStream.Clear();
                newStream.AddRange(oldStream);

                // 中間ファイルの出現順にストリームをソートする。
                StreamUtility.SortStream(newTexPatternAnim, newStream);
            }
            // newTexPatternAnim にのみアニメーションが存在している場合は
            // マージする必要がないので処理を行わない。
        }

        // テクスチャパターンアニメーションを持つか調べます。
        private static bool HasTexturePatternAnim(tex_pattern_animType texPattern)
        {
            if (texPattern.tex_pattern_array == null ||
                texPattern.tex_pattern_array.tex_pattern == null)
            {
                return false;
            }

            if (texPattern.tex_pattern_mat_anim_array == null ||
                texPattern.tex_pattern_mat_anim_array.tex_pattern_mat_anim == null)
            {
                return false;
            }

            return true;
        }

        // <tex_pattern> を混ぜ合わせる。
        // newTexPatternAnim 内の <tex_pattern> はそのままにし、oldTexPatternAnim
        // でのみ使用されている <tex_pattern> を newTexPatternAnim の <tex_pattern_array> に追加する。
        // 追加する <tex_pattern> は順番を保つようにする。
        private static void MergeTexturePattern(
            tex_pattern_animType newTexPatternAnim,
            tex_pattern_animType oldTexPatternAnim,
            Dictionary<tex_pattern_mat_animType, tex_pattern_mat_animType> texPatternMatTable,
            List<G3dStream> oldStream,
            out int[] newTexPatternIdx)
        {
            // <tex_pattern> の使用フラグ
            bool[] useOldTexPattern = new bool[oldTexPatternAnim.tex_pattern_array.length];
            // oldTexPatternAnim にのみ存在しているテクスチャパターンアニメーションが
            // 参照している <tex_pattern> の useOldTexPattern を true にする。
            foreach (tex_pattern_mat_animType texPatternMatAnim in
                oldTexPatternAnim.tex_pattern_mat_anim_array.tex_pattern_mat_anim)
            {
                // マージファイル内と引数ファイルで同じ mat_name を持つ <tex_pattern_mat_anim>
                // が存在するか確認する。
                tex_pattern_mat_animType newTexPatternMatAnim;
                if (texPatternMatTable.ContainsKey(texPatternMatAnim))
                {
                    newTexPatternMatAnim = texPatternMatTable[texPatternMatAnim];
                }
                else
                {
                    newTexPatternMatAnim = null;
                }

                foreach (pattern_anim_targetType target in texPatternMatAnim.pattern_anim_target)
                {
                    // カーブが存在していなければ tex_pattern をマージする必要がない。
                    if (target.Curve == null)
                    {
                        continue;
                    }
                    // 重複する hint をもつ pattern_anim_target が存在していないか検索し、
                    // 存在していれば newTexPatternAnim のアニメーションが優先されるため、
                    // oldTexPatternAnim の tex_pattern をマージしない。
                    if (newTexPatternMatAnim != null)
                    {
                        int index = Array.FindIndex(newTexPatternMatAnim.pattern_anim_target,
                            delegate(pattern_anim_targetType tgt) { return tgt.hint == target.hint; });
                        if (index >= 0)
                        {
                            continue;
                        }
                    }

                    // カーブが参照するテクスチャパターンに使用している印をつける。
                    G3dStream stream = oldStream[target.Curve.stream_index];
                    int numKey = stream.FloatData.Count / stream.column;
                    for (int iKey = 0; iKey < numKey; iKey++)
                    {
                        int texPatternIndex = (int)stream.FloatData[iKey * stream.column + 1];
                        useOldTexPattern[texPatternIndex] = true;
                    }
                }
            }

            // oldTexPatternAnim の <tex_pattern> で使用されている
            List<tex_patternType> newTexPatternArray =
                new List<tex_patternType>(newTexPatternAnim.tex_pattern_array.tex_pattern);
            List<tex_patternType> oldTexPatternArray =
                new List<tex_patternType>(oldTexPatternAnim.tex_pattern_array.tex_pattern);
            newTexPatternIdx = new int[oldTexPatternAnim.tex_pattern_array.length];

            for (int i = 0; i < oldTexPatternArray.Count; i++)
            {
                // 使用されていない <tex_pattern> はコピーしない。
                if (!useOldTexPattern[i])
                {
                    continue;
                }

                tex_patternType texPattern = oldTexPatternArray[i];

                int index = newTexPatternArray.FindIndex(
                    delegate(tex_patternType tp) { return tp.tex_name == texPattern.tex_name; });
                if (index >= 0)
                {
                    // newTexPatternAnim にテクスチャパターンが存在していたら
                    // 既存のパターンのインデックスを使用する。
                    newTexPatternIdx[i] = index;
                }
                else
                {
                    // newTexPatternAnim にテクスチャパターンが存在していなければ末尾に追加する。
                    newTexPatternIdx[i] = newTexPatternArray.Count;
                    newTexPatternArray.Add(texPattern);
                }
            }
            // <tex_pattern_array> を置きかえる。
            newTexPatternAnim.tex_pattern_array.tex_pattern = newTexPatternArray.ToArray();
            newTexPatternAnim.tex_pattern_array.length = newTexPatternArray.Count;
        }

        // <tex_pattern_mat_anim> をコピーします。
        private static void MergeTexPatternAnim(
            tex_pattern_mat_animType newTexPatternAnim,
            List<G3dStream> newStream,
            tex_pattern_mat_animType oldTexPatternAnim,
            List<G3dStream> oldStream,
            int[] newTexPatternIdx)
        {
            List<pattern_anim_targetType> newPatternAnimTargetArray;
            if (newTexPatternAnim.pattern_anim_target != null)
            {
                newPatternAnimTargetArray =
                    new List<pattern_anim_targetType>(newTexPatternAnim.pattern_anim_target);
            }
            else
            {
                newPatternAnimTargetArray =
                    new List<pattern_anim_targetType>();
            }

            foreach (pattern_anim_targetType oldPatternAnimTarget in
                oldTexPatternAnim.pattern_anim_target)
            {
                // 同じ hint をもつターゲットを検索する。
                // ターゲットが存在していなければ oldPatternAnimTarget をコピーする。
                int targetIdx = newPatternAnimTargetArray.FindIndex(
                    delegate(pattern_anim_targetType tgt)
                    {
                        return tgt.hint == oldPatternAnimTarget.hint;
                    });
                if (targetIdx >= 0)
                {
                    continue;
                }
                pattern_anim_targetType newPatternAnimTarget =
                    new pattern_anim_targetType();
                newPatternAnimTargetArray.Add(newPatternAnimTarget);

                newPatternAnimTarget.sampler_name = oldPatternAnimTarget.sampler_name;
                newPatternAnimTarget.hint = oldPatternAnimTarget.hint;
                newPatternAnimTarget.base_value =
                    (float)newTexPatternIdx[(int)oldPatternAnimTarget.base_value];
                // カーブが存在していたらカーブもコピーする
                if (oldPatternAnimTarget.Curve != null)
                {
                    step_curveType newCurve;
                    step_curveType oldCurve = oldPatternAnimTarget.Curve as step_curveType;
                    G3dStream newTexPatternStream = new G3dStream();
                    G3dStream oldTexPatternStream = oldStream[oldCurve.stream_index];

                    // カーブをコピーする。
                    // コピーする際に、カーブの内容を新しい newStream 内のインデックスに置きかえる。
                    CopyTexPatternAnimCurveAndStream(
                        out newCurve,
                        newTexPatternStream,
                        newStream.Count,
                        oldCurve,
                        oldTexPatternStream,
                        newTexPatternIdx);

                    // ストリームを newStream に追加する。
                    newStream.Add(newTexPatternStream);
                    newPatternAnimTarget.Curve = newCurve;
                }
            }

            newTexPatternAnim.pattern_anim_target = newPatternAnimTargetArray.ToArray();
            newTexPatternAnim.mat_name = oldTexPatternAnim.mat_name;
            // destTexPatternAnim.index はソート後に設定するのでここではコピーしない。
        }

        // テクスチャパターンアニメーションのカーブとカーブが参照するストリームをコピーします。
        // ストリームをコピーする際にはインデックスを新しい値に置き換えます。
        internal static void CopyTexPatternAnimCurveAndStream(
            out step_curveType newCurve,
            G3dStream newStream,
            int streamIndex,
            step_curveType oldCurve,
            G3dStream oldStream,
            int[] newTexPatternIdx)
        {
            // カーブの情報をコピーする。
            newCurve = IfMergeUtility.CopyCurve(oldCurve) as step_curveType;
            // 新しいストリームのインデックスを設定する
            newCurve.stream_index = streamIndex;
            // ストリームをコピーする。
            newStream.type = oldStream.type;
            newStream.column = oldStream.column;
            // ストリームの内容をコピーする。
            int numKey = oldStream.FloatData.Count / oldStream.column;
            for (int iKey = 0; iKey < numKey; iKey++)
            {
                float time = oldStream.FloatData[iKey * oldStream.column];
                int value = (int)oldStream.FloatData[iKey * oldStream.column + 1];
                // テクスチャパターンのインデックスを置きかえる。
                value = newTexPatternIdx[value];

                newStream.FloatData.Add(time);
                newStream.FloatData.Add((float)value);
            }
        }
    }
}
