﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;
using System.Diagnostics;
using System.Linq;

namespace nw.g3d.iflib
{
    // モデルマージャ
    public static class IfModelMerger
    {
        // マージ用の情報
        public class IfModelMergerInfo
        {
            public bool IsMergeShape { get; set; }
            public bool IsMergeVertex { get; set; }
            public bool IsMergeScaleEnable { get; set; }
            public bool SamplerMergePriorityToNewModel { get; set; }
            public bool TextureSrtMergePriorityToNewModel { get; set; }

            public bool BoneVisibilityMergePriorityToNewModel { get; set; }

            public IEnumerable<IfMergeSrcDstPair> BoneNamePairTable { get; set; }
            public IEnumerable<IfMergeSrcDstPair> MaterialNamePairTable { get; set; }
            public IEnumerable<IfMergeSrcDstPair> ShapeNamePairTable { get; set; }

            public IfModelMergerInfo()
            {
                IsMergeShape = true;
                IsMergeVertex = true;
                IsMergeScaleEnable = true;
                SamplerMergePriorityToNewModel = false;
                TextureSrtMergePriorityToNewModel = false;
                BoneVisibilityMergePriorityToNewModel = false;

                BoneNamePairTable = null;
                MaterialNamePairTable = null;
            }
        }

        //=====================================================================
        // マージ
        public static void Merge(IfModelMergeContext context)
        {
            Merge(context, null);
        }

        //=====================================================================
        // オプション付きマージ
        public static void Merge(IfModelMergeContext context, IfModelMergerInfo info)
        {
            // TODO: Unite モデル対応
            // これより下は Unite されたモデルにまだ対応していない。
            foreach (shapeType shape in context.NewModel.shape_array.shape)
            {
                if (shape.mesh_array.mesh.Length > 1)
                {
                    IfStrings.Throw("IfModelOptimizer_Error_UnitedModelUnsupported");
                }
            }

            IfModelMergerInfo mergeInfo;
            if (info == null)
            {
                // オプションが指定されなかった場合は全てのマージを実行し、
                // ボーンとマテリアルは全て同一名によりマージを行うデフォルトオプションを作成する。
                mergeInfo = new IfModelMergerInfo();
            }
            else
            {
                mergeInfo = info;
            }

            // model
            // <user_data_array> <tool_data> <user_tool_data> <comment>
            IfMergeUtility.MergeRootObject(
                context.NewModel, context.OldModel,
                context.NewStreams, context.OldStreams);

            // <model_info> unite_pos_quantize
            context.NewModel.model_info.unite_pos_quantize =
                context.OldModel.model_info.unite_pos_quantize;

            // <skeleton_info> scale_enable
            if (mergeInfo.IsMergeScaleEnable)
            {
                context.NewModel.skeleton.skeleton_info.scale_enable =
                    context.OldModel.skeleton.skeleton_info.scale_enable;
            }

            context.NewModel.skeleton.skeleton_info.motion_mirroring_enable = context.OldModel.skeleton.skeleton_info.motion_mirroring_enable;

            // material
            // <material_info> visibility
            // <render_state>
            // <sampler>
            // <shader_assign>
            // <user_data_array> <tool_data> <user_tool_data>
            if (context.NewModel.material_array != null &&
                context.NewModel.material_array.material != null &&
                context.OldModel.material_array != null &&
                context.OldModel.material_array.material != null)
            {
                Dictionary<string, original_materialType> newOriginalMaterials =
                    context.NewModel.original_material_array.GetItems().ToDictionary(x => x.mat_name);
                if (mergeInfo.MaterialNamePairTable == null)
                {
                    IfMergeUtility.MergeByTable(
                        context.NewModel.material_array.material,
                        context.MaterialTable,
                        (x, y) => MergeMaterial(
                            x,
                            context.NewStreams,
                            y,
                            context.OldStreams,
                            mergeInfo,
                            newOriginalMaterials));
                }
                else
                {
                    // マテリアルのマージするペアが指定された場合は、そのペア同士でマージを行う。
                    // ペアの配列に含まれていないマテリアルはマージを行わない。
                    var materialTable = new Dictionary<materialType, materialType>();
                    materialType[] newMaterials = context.NewModel.material_array.material;
                    materialType[] oldMaterials = context.OldModel.material_array.material;
                    IfMergeUtility.SetupTableByName(
                        materialTable, newMaterials, oldMaterials,
                        mergeInfo.MaterialNamePairTable);

                    // マージを実行する。
                    IfMergeUtility.MergeByTable(
                        context.NewModel.material_array.material,
                        materialTable,
                        (x, y) => MergeMaterial(
                            x,
                            context.NewStreams,
                            y,
                            context.OldStreams,
                            mergeInfo,
                            newOriginalMaterials));
                }
            }

            // bone
            // <bone> billboard visibility
            // <user_data_array> <tool_data> <user_tool_data>
            if (mergeInfo.BoneNamePairTable == null)
            {
                IfMergeUtility.MergeByTable(
                    context.NewModel.skeleton.bone_array.bone,
                    context.BoneTable,
                    context.NewStreams,
                    context.OldStreams,
                    MergeBone,
                    mergeInfo);
            }
            else
            {
                // ボーンのマージするペアが指定された場合は、そのペア同士でマージを行う。
                // ペアの配列に含まれていないボーンはマージしない。
                var boneTable = new Dictionary<boneType, boneType>();
                boneType[] newBones = context.NewModel.skeleton.bone_array.bone;
                boneType[] oldBones = context.OldModel.skeleton.bone_array.bone;
                IfMergeUtility.SetupTableByName(
                    boneTable, newBones, oldBones,
                    mergeInfo.BoneNamePairTable);

                // マージを実行する。
                IfMergeUtility.MergeByTable(
                    newBones,
                    boneTable,
                    context.NewStreams,
                    context.OldStreams,
                    MergeBone,
                    mergeInfo);
            }

            // shape
            // × <submesh>
            // <user_data_array> <tool_data> <user_tool_data>
            // <shape_info> optimize_primitive_mode original_shape_hash Streams
            // <input> <mesh> quantize_type
            // <vtx_buffer> all
            if (context.NewModel.shape_array != null &&
                context.NewModel.shape_array.shape != null &&
                context.NewModel.vertex_array != null &&
                context.NewModel.vertex_array.vertex != null &&
                context.OldModel.shape_array != null &&
                context.OldModel.shape_array.shape != null &&
                context.OldModel.vertex_array != null &&
                context.OldModel.vertex_array.vertex != null)
            {
                if (mergeInfo.ShapeNamePairTable == null)
                {
                    foreach (Dictionary<string, IfMergeShapeGroup> shapeDict in context.ShapeTable.Values)
                    {
                        foreach (IfMergeShapeGroup shapeGroup in shapeDict.Values)
                        {
                            if (shapeGroup.IsStructChanged()) { continue; }

                            // <shape> のマージを実行する。
                            if (mergeInfo.IsMergeShape)
                            {
                                foreach (shapeType newObject in context.NewModel.shape_array.shape)
                                {
                                    if (!shapeGroup.ShapeTable.ContainsKey(newObject)) { continue; }
                                    shapeType oldObject = shapeGroup.ShapeTable[newObject];
                                    MergeShape(
                                        context.NewModel,
                                        newObject,
                                        context.NewStreams,
                                        context.OldModel,
                                        oldObject,
                                        context.OldStreams,
                                        mergeInfo);
                                }
                            }
                        }
                    }
                }
                else
                {
                    // シェイプのマージするペアが指定された場合は、そのペア同士でマージを行う。
                    // ペアの配列に含まれていないシェイプはマージしない。
                    var shapeTable = new Dictionary<shapeType, shapeType>();
                    shapeType[] newShapes = context.NewModel.shape_array.shape;
                    shapeType[] oldShapes = context.OldModel.shape_array.shape;
                    IfMergeUtility.SetupTableByName(
                        shapeTable, newShapes, oldShapes,
                        mergeInfo.BoneNamePairTable);

                    // マージを実行する。
                    foreach (shapeType newObject in newShapes)
                    {
                        if (!shapeTable.ContainsKey(newObject)) { continue; }
                        shapeType oldObject = shapeTable[newObject];
                        MergeShape(
                            context.NewModel,
                            newObject,
                            context.NewStreams,
                            context.OldModel,
                            oldObject,
                            context.OldStreams,
                            mergeInfo);
                    }
                }
            }

            // ストリームを並べ替える
            StreamUtility.SortStream(context.NewModel, context.NewStreams);

            // Merge は 3DEditor から直接呼び出されるため、
            // この関数内で model_info を更新する
            IfModelUpdateUtility.UpdateModelInfo(context.NewModel, context.NewStreams);
        }

        //---------------------------------------------------------------------
        // マテリアルのマージ
        private static void MergeMaterial(
            materialType newMaterial, List<G3dStream> newStreams,
            materialType oldMaterial, List<G3dStream> oldStreams,
            IfModelMergerInfo info,
            Dictionary<string, original_materialType> newOriginalMaterials)
        {
            newMaterial.material_info.visibility = oldMaterial.material_info.visibility;
            newMaterial.material_info.mesh_adjacency = oldMaterial.material_info.mesh_adjacency;
            newMaterial.render_state = oldMaterial.render_state;
            newMaterial.shader_assign = oldMaterial.shader_assign;

            // 新しいモデルのサンプラを優先する指定がある場合、
            // サンプラのマージ処理を行います。
            if (info.SamplerMergePriorityToNewModel)
            {
                List<samplerType> result = MergeSampler(newMaterial.sampler_array, oldMaterial.sampler_array);
                if (result.Count > 0)
                {
                    if (newMaterial.sampler_array == null)
                    {
                        newMaterial.sampler_array = new sampler_arrayType();
                    }

                    newMaterial.sampler_array.length = result.Count;
                    newMaterial.sampler_array.sampler = result.ToArray();
                }
                else if (newMaterial.sampler_array != null)
                {
                    newMaterial.sampler_array = new sampler_arrayType();
                }
            }
            else
            {
                newMaterial.sampler_array = oldMaterial.sampler_array;
            }

            if (info.TextureSrtMergePriorityToNewModel)
            {
                // texsrt, texsrt_ex について newMaterial の original_texsrt の値を使う
                AssignOriginalTexsrtToShaderParam(newMaterial, newOriginalMaterials);
            }

            user_data_arrayType newUserData = newMaterial.user_data_array;
            IfMergeUtility.MergeUserDataArray(
                ref newUserData,
                oldMaterial.user_data_array,
                newStreams, oldStreams);
            newMaterial.user_data_array = newUserData;

            if (newMaterial.comment != null &&
                oldMaterial.comment != null)
            {
                IfMergeUtility.MergeComment(
                    newMaterial.comment,
                    oldMaterial.comment);
            }
            else if (oldMaterial.comment != null)
            {
                newMaterial.comment = oldMaterial.comment;
            }

            newMaterial.tool_data = oldMaterial.tool_data;
            newMaterial.user_tool_data = oldMaterial.user_tool_data;
        }

        private static void AssignOriginalTexsrtToShaderParam(materialType material, Dictionary<string, original_materialType> originalMaterials)
        {
            original_materialType originalMaterial;
            if (material.shader_assign != null &&
                originalMaterials.TryGetValue(material.name, out originalMaterial))
            {
                foreach (var shaderParam in material.shader_assign.shader_param_array.GetItems())
                {
                    if ((shaderParam.type == shader_param_typeType.texsrt)
                        && !string.IsNullOrEmpty(shaderParam.original_hint))
                    {
                        var originalTexSrt = originalMaterial.original_texsrt_array.GetItems()
                            .FirstOrDefault(x => x.hint == shaderParam.original_hint);

                        if (originalTexSrt != null)
                        {
                            shaderParam.Value = IfShaderParameterUtility.MakeTexsrtValueFromOriginalTexsrt(originalTexSrt);
                        }
                    }
                }
            }
        }

        //---------------------------------------------------------------------
        // サンプラのパラメーターをコピーする
        private static void CopySamplerParameter(
            samplerType dest, samplerType src)
        {
            dest.wrap = new wrapType
            {
                u = src.wrap.u,
                v = src.wrap.v,
                w = src.wrap.w
            };
            dest.filter = new filterType
            {
                mag = src.filter.mag,
                min = src.filter.min,
                mip = src.filter.mip,
                max_aniso = src.filter.max_aniso
            };
            dest.lod = new lodType
            {
                min = src.lod.min,
                max = src.lod.max,
                bias = src.lod.bias
            };
        }

        //---------------------------------------------------------------------
        // サンプラのパラメーターをマージする
        private static void MergeSamplerParameter(
            samplerType dest, samplerType src)
        {
            // wrap の u, v は DCC ツールで設定されたものを維持する
            dest.wrap.w = src.wrap.w;

            dest.filter = new filterType
            {
                mag = src.filter.mag,
                min = src.filter.min,
                mip = src.filter.mip,
                max_aniso = src.filter.max_aniso
            };
            dest.lod = new lodType
            {
                min = src.lod.min,
                max = src.lod.max,
                bias = src.lod.bias
            };
        }

        //---------------------------------------------------------------------
        // <sampler> をコピー
        internal static samplerType CopySampler(samplerType src)
        {
            var copied = new samplerType
            {
                sampler_index = src.sampler_index,
                name = src.name,
                hint = src.hint,
                tex_name = src.tex_name
            };
            CopySamplerParameter(copied, src);

            return copied;
        }

        //---------------------------------------------------------------------
        // サンプラの名前が予約名かどうかを取得する
        private static bool IsReservedSamplerName(string name)
        {
            if (string.IsNullOrEmpty(name))
            {
                return false;
            }

            return name[0] == '_';
        }

        //---------------------------------------------------------------------
        // サンプラのマージ
        internal static List<samplerType> MergeSampler(
            sampler_arrayType newSamplers,
            sampler_arrayType oldSamplers)
        {
            // サンプラのマージ動作
            // newSamplers : マージ先データ。DCC ツールから出力したもの。
            // oldSamplers : マージ元データ。3DEditor で編集したもの。
            //
            // サンプラは名前ではなく hint でマッチングを行います。
            // 以下はマッチしたサンプラのマージ処理の挙動です。
            //
            // ルール：名前とヒントの両方が一致するものを同一のサンプラとみなす
            //         下のパターンに一致しないものはマージしない
            //
            // パターン 1  名前が予約名のサンプラで newSamplers に同一のサンプラがある
            // 結果 :      サンプラのパラメーターのみマージする
            //
            // パターン 2  名前が非予約名のサンプラで newSamplers に同一のサンプラがある
            // 結果 :      サンプラのパラメーターのみマージする
            //
            // パターン 3  名前が非予約名のサンプラで newSamplers に同じ名前か同じヒントのサンプラがない
            //             ※ ヒントが空の場合はヒントの重複をチェックしない
            // 結果 :      マージする
            //
            // パターン 4  名前が予約名でヒントのないサンプラで newSamplers に同じ名前のサンプラがない
            //             ※ 過去の3DEditorで追加されたサンプラ
            // 結果 :      マージする

            var mergeResult = new List<samplerType>();
            var mergeSampler = new List<samplerType>();
            bool isNewSamplersNull = (newSamplers == null || newSamplers.sampler == null);
            bool isOldSamplersNull = (oldSamplers == null || oldSamplers.sampler == null);

            if (!isNewSamplersNull)
            {
                mergeResult.AddRange(newSamplers.sampler);
            }

            // マージ元データがない場合はマージ先データをそのまま使用する
            if (isOldSamplersNull)
            {
                return mergeResult;
            }

            // パターンに照らし合わせながらマージを実行する
            foreach (var oldSampler in oldSamplers.sampler)
            {
                var isHintNull = string.IsNullOrEmpty(oldSampler.hint);
                if (IsReservedSamplerName(oldSampler.name))
                {
                    // 予約名のサンプラはマージ先データがなければマージしない
                    if (isNewSamplersNull)
                    {
                        if (!isHintNull)
                        {
                            continue;
                        }
                    }

                    int sameSamplerIndex = mergeResult.FindIndex(
                        sampler => (string.CompareOrdinal(sampler.name, oldSampler.name) == 0) &&
                                   (string.CompareOrdinal(sampler.hint, oldSampler.hint) == 0));
                    var sameNameIndex = mergeResult.FindIndex(
                        sampler => (string.CompareOrdinal(sampler.name, oldSampler.name) == 0));
                    if (sameSamplerIndex >= 0)
                    {
                        // パターン 1 に該当
                        MergeSamplerParameter(mergeResult[sameSamplerIndex], oldSampler);
                    }
                    else if (isHintNull && sameNameIndex < 0)
                    {
                        // 名前が予約名でヒントが空の場合は、過去の3DEditorで追加されたサンプラ
                        // 同じ予約名でヒントが異なるサンプラが有る場合はマージしない。
                        // パターン 4 に該当
                        var copied = CopySampler(oldSampler);
                        copied.sampler_index = mergeResult.Count;
                        mergeSampler.Add(copied);
                    }
                }
                else
                {
                    if (!isNewSamplersNull)
                    {
                        samplerType matchNewSampler = null;
                        samplerType mismatchNewSampler = null;
                        foreach (var newSampler in mergeResult)
                        {
                            bool isSameName = string.CompareOrdinal(newSampler.name, oldSampler.name) == 0;
                            bool isSameHint = string.CompareOrdinal(newSampler.hint, oldSampler.hint) == 0;
                            if (isSameName && isSameHint)
                            {
                                matchNewSampler = newSampler;
                                break;
                            }
                            // ヒントが空の場合は重複をチェックしない
                            if (isSameName || (isSameHint && !isHintNull))
                            {
                                mismatchNewSampler = newSampler;
                                break;
                            }
                        }

                        if (matchNewSampler != null)
                        {
                            // パターン 2 に該当
                            MergeSamplerParameter(matchNewSampler, oldSampler);
                        }
                        else if (mismatchNewSampler == null)
                        {
                            // パターン 3 に該当
                            var copied = CopySampler(oldSampler);
                            copied.sampler_index = mergeResult.Count;
                            mergeSampler.Add(copied);
                        }
                    }
                    else
                    {
                        // newSamplers が存在していない場合はすべてパターン 3 に該当
                        var copied = CopySampler(oldSampler);
                        copied.sampler_index = mergeResult.Count;
                        mergeSampler.Add(copied);
                    }
                }
            }

            mergeResult.AddRange(mergeSampler);
            return mergeResult;
        }

        //---------------------------------------------------------------------
        // ボーンのマージ
        internal static void MergeBone(
            boneType newBone, List<G3dStream> newStreams,
            boneType oldBone, List<G3dStream> oldStreams,
            IfModelMergerInfo info)
        {
            newBone.billboard = oldBone.billboard;
            if (!info.BoneVisibilityMergePriorityToNewModel)
            {
                newBone.visibility = oldBone.visibility;
            }

            user_data_arrayType newUserData = newBone.user_data_array;
            IfMergeUtility.MergeUserDataArray(
                ref newUserData,
                oldBone.user_data_array,
                newStreams, oldStreams);
            newBone.user_data_array = newUserData;

            if (newBone.comment != null &&
                oldBone.comment != null)
            {
                IfMergeUtility.MergeComment(
                    newBone.comment,
                    oldBone.comment);
            }
            else if (oldBone.comment != null)
            {
                newBone.comment = oldBone.comment;
            }

            newBone.tool_data = oldBone.tool_data;
            newBone.user_tool_data = oldBone.user_tool_data;
        }

        //---------------------------------------------------------------------
        // 頂点データのマージ
        private static void MergeVertexBuffer(vertexType newVertex, vertexType oldVertex, IfModelMergerInfo mergeInfo)
        {
            vtx_attribType[] newVtxAttribs = newVertex.vtx_attrib_array.vtx_attrib;
            vtx_attribType[] oldVtxAttribs = oldVertex.vtx_attrib_array.vtx_attrib;

            // vtx_attrib_array の構成が同じであるか確認する。
            // vtx_attrib の数と hint が一致すれば構成が同じとみなす。
            if (newVtxAttribs.Length != oldVtxAttribs.Length) { return; }
            for (int iVtx = 0; iVtx < newVtxAttribs.Length; iVtx++)
            {
                vtx_attribType newAttrib = newVtxAttribs[iVtx];
                vtx_attribType oldAttrib = oldVtxAttribs[iVtx];
                if (newAttrib.hint != oldAttrib.hint) { return; }
            }

            // vtx_buffer をマージする
            if (oldVertex.vtx_buffer_array != null)
            {
                newVertex.vtx_buffer_array = oldVertex.vtx_buffer_array;
            }
        }

        //---------------------------------------------------------------------
        // シェイプのマージ
        private static void MergeShape(
            modelType newModel, shapeType newShape, List<G3dStream> newStreams,
            modelType oldModel, shapeType oldShape, List<G3dStream> oldStreams,
            IfModelMergerInfo mergeInfo)
        {
            // MEMO: 現在 newShape に LOD を持つデータが入力されることはありません。

            shape_infoType newInfo = newShape.shape_info;
            shape_infoType oldInfo = oldShape.shape_info;

            // シェイプをマージしたら頂点列もマージする
            vertexType newVertex = newModel.vertex_array.vertex[newInfo.vertex_index];
            vertexType oldVertex = oldModel.vertex_array.vertex[oldInfo.vertex_index];
            MergeVertexBuffer(newVertex, oldVertex, mergeInfo);

            // 最適化結果のマージ、最適化が行われていなければマージしない
            if (oldInfo.original_shape_hash == string.Empty) { return; }

            // DCC から出力した状態のハッシュを求める
            string newHash = newInfo.original_shape_hash;
            if (newHash == string.Empty)
            {
                // 通常の利用形態であれば、常に string.Empty なはず
                newHash = IfOptimizePrimitiveHashCalculator.Calculate(
                    newModel, newStreams, newShape);
            }

            // 頂点列とインデックス列が変更されていれば、マージしない
            if (oldInfo.original_shape_hash != newHash) { return; }

            newInfo.original_shape_hash = oldInfo.original_shape_hash;
            newInfo.original_shape_count = oldInfo.original_shape_count;
            newInfo.original_material_name = oldInfo.original_material_name;
            newInfo.original_bone_name = oldInfo.original_bone_name;
            newInfo.delete_near_vertex_mode = oldInfo.delete_near_vertex_mode;
            newInfo.divide_submesh_mode = oldInfo.divide_submesh_mode;
            newInfo.optimize_primitive_mode = oldInfo.optimize_primitive_mode;
            newInfo.polygon_reduction_mode = oldInfo.polygon_reduction_mode;

            // <vtx_attrib> の quantize_type もマージします。
            int vtx_attribLength = newVertex.vtx_attrib_array.vtx_attrib.Length;
            Nintendo.Foundation.Contracts.Assertion.Operation.True(vtx_attribLength == oldVertex.vtx_attrib_array.vtx_attrib.Length);
            for (int i = 0; i < vtx_attribLength; i++)
            {
                newVertex.vtx_attrib_array.vtx_attrib[i].quantize_type =
                    oldVertex.vtx_attrib_array.vtx_attrib[i].quantize_type;
            }

            bool hasLOD = (oldShape.mesh_array.mesh.Length > 1) ? true : false;
            if (hasLOD)
            {
                if (oldInfo.polygon_reduction_mode.Length == 0)
                {   // polygon_reduction_mode が空の場合(LOD が手動生成された)は LOD を含めずにマージします。
                    MergeWithoutLOD(newModel, newShape, newStreams,
                                    oldModel, oldShape, oldStreams);
                }
                else
                {   // LOD を含めてマージします。
                    MergeLOD(newModel, newShape, newStreams,
                             oldModel, oldShape, oldStreams);
                }
            }
            else
            {
                // LOD を含まない mesh のマージ処理です。
                // このマージでは以下の要素をコピーします。
                //
                // - <mesh> の全要素
                // - <mesh> が参照する <stream>
                // - <vtx_attrib> の count
                // - <vtx_attrib> が参照する <stream>

                meshType newMesh = newShape.mesh_array.mesh[0];
                meshType oldMesh = oldShape.mesh_array.mesh[0];

                // メッシュは全要素をマージする
                newShape.mesh_array.mesh[0] = CopyMesh(oldMesh);
                newShape.mesh_array.mesh[0].stream_index = newMesh.stream_index;

                // インデックス列のマージ
                // -dnv でインデックス数が変わる
                MergeStream(newStreams[newMesh.stream_index], oldStreams[oldMesh.stream_index]);

                // 頂点列のマージ
                for (int i = 0; i < vtx_attribLength; i++)
                {
                    // -dnv で頂点数が変わる
                    newVertex.vtx_attrib_array.vtx_attrib[i].count =
                        oldVertex.vtx_attrib_array.vtx_attrib[i].count;
                    MergeStream(
                        newStreams[newVertex.vtx_attrib_array.vtx_attrib[i].stream_index],
                        oldStreams[oldVertex.vtx_attrib_array.vtx_attrib[i].stream_index]);
                }
            }
        }

        //---------------------------------------------------------------------
        // LOD を含めたメッシュのマージ
        private static void MergeLOD(
            modelType newModel, shapeType newShape, List<G3dStream> newStreams,
            modelType oldModel, shapeType oldShape, List<G3dStream> oldStreams)
        {
            shape_infoType newInfo = newShape.shape_info;
            shape_infoType oldInfo = oldShape.shape_info;

            // シェイプをマージしたら頂点列もマージする
            vertexType newVertex = newModel.vertex_array.vertex[newInfo.vertex_index];
            vertexType oldVertex = oldModel.vertex_array.vertex[oldInfo.vertex_index];

            // lod_offset をマージ
            newVertex.lod_offset = oldVertex.lod_offset;

            // 頂点列のマージ
            // LOD 込みでマージを行うので、ストリームをそのままマージします。
            int vtx_attribLength = newVertex.vtx_attrib_array.vtx_attrib.Length;
            Nintendo.Foundation.Contracts.Assertion.Operation.True(vtx_attribLength == oldVertex.vtx_attrib_array.vtx_attrib.Length);
            for (int i = 0; i < vtx_attribLength; i++)
            {
                newVertex.vtx_attrib_array.vtx_attrib[i].count =
                    oldVertex.vtx_attrib_array.vtx_attrib[i].count;
                MergeStream(
                    newStreams[newVertex.vtx_attrib_array.vtx_attrib[i].stream_index],
                    oldStreams[oldVertex.vtx_attrib_array.vtx_attrib[i].stream_index]);
            }

            List<meshType> newMeshList = new List<meshType>();

            // LOD のメッシュをマージします。
            foreach (meshType oldMesh in oldShape.mesh_array.mesh)
            {
                // LOD のメッシュはマージファイルのメッシュを残します。
                meshType newMesh = CopyMesh(oldMesh);
                newMeshList.Add(newMesh);

                // メッシュのインデックスを追加する
                newMesh.stream_index = newStreams.Count;
                newStreams.Add(oldStreams[oldMesh.stream_index]);
            }

            // メッシュ配列を更新
            newShape.mesh_array.mesh = newMeshList.ToArray();
            newShape.mesh_array.length = newMeshList.Count;
        }

        //---------------------------------------------------------------------
        // LOD を含めないメッシュのマージ
        private static void MergeWithoutLOD(
            modelType newModel, shapeType newShape, List<G3dStream> newStreams,
            modelType oldModel, shapeType oldShape, List<G3dStream> oldStreams)
        {
            // LOD を除いた mesh のマージ処理です。
            // このマージでは以下の要素をコピーします。
            //
            // - <mesh> が参照する <stream>
            // - <submesh> の配列
            //
            // <vtx_attrib> と参照する頂点ストリームはベースの mesh が使用する頂点のみをコピーします。

            shape_infoType newInfo = newShape.shape_info;
            shape_infoType oldInfo = oldShape.shape_info;
            meshType newBaseMesh = newShape.mesh_array.mesh[0];
            meshType oldBaseMesh = oldShape.mesh_array.mesh[0];
            meshType newMesh = CopyMesh(oldBaseMesh);
            vertexType newVertex = newModel.vertex_array.vertex[newInfo.vertex_index];
            vertexType oldVertex = oldModel.vertex_array.vertex[oldInfo.vertex_index];

            // ベースの mesh が使用する頂点ストリームのみコピーするアルゴリズムです。
            // 1. oldShape.mesh_array.mesh[0].stream_index が参照するインデクスのストリームで
            //    使用されているインデックスにマークを付けます。
            // 2. oldVertex.vtx_attrib_array が参照するストリームのなかで 1. でマークされた
            //    要素のみ newStream にコピーします。
            // 3. マークされたインデックスのみ newShape.mesh_array.mesh[0].stream_index が
            //    参照するストリームにコピーします。その際にインデックスの値を 2. でコピーした
            //    頂点に合わせて修正します。

            // 使用しない頂点インデックスをマークし、新しいインデックスを計算するための
            // 一時的なインデックスリストを作成します。
            // インデックスリストは初期化時に使用していない頂点を表す -1 を設定します。
            //
            // 頂点を削除しても順番が変わらないようにします。

            int numVertices = oldVertex.vtx_attrib_array.vtx_attrib[0].count;
            List<int> useVetices = new List<int>(numVertices);
            for (int i = 0; i < numVertices; i++)
            {
                useVetices.Add(-1);
            }

            // 使用しているインデックスに 0 を設定してマークを付けます。
            List<int> oldIndices = oldStreams[oldBaseMesh.stream_index].IntData;
            foreach (int i in oldIndices)
            {
                useVetices[i] = 0;
            }
            // useVetices の中でマークされた要素のみインデクスを割り当てます。
            int curIndex = 0;
            for (int i = 0; i < useVetices.Count; i++)
            {
                if (useVetices[i] == 0)
                {
                    useVetices[i] = curIndex++;
                }
            }

            // useVetices でマークされている要素のみ vtx_attrib をコピーします。
            int vtx_attribLength = newVertex.vtx_attrib_array.vtx_attrib.Length;
            Nintendo.Foundation.Contracts.Assertion.Operation.True(vtx_attribLength == oldVertex.vtx_attrib_array.vtx_attrib.Length);
            for (int i = 0; i < vtx_attribLength; i++)
            {
                vtx_attribType newVtxAttrib = newVertex.vtx_attrib_array.vtx_attrib[i];
                vtx_attribType oldVtxAttrib = oldVertex.vtx_attrib_array.vtx_attrib[i];

                int stride = GetStreamStride(oldVtxAttrib);
                int copyCount = CopyStreamIf(newStreams[newVtxAttrib.stream_index],
                                             oldStreams[oldVtxAttrib.stream_index],
                                             useVetices, stride);

                newVtxAttrib.count = copyCount / stride;
            }

            // インデックスストリームをマージします。
            List<int> oldIndexStream = oldStreams[oldBaseMesh.stream_index].IntData;
            List<int> newIndexStream = new List<int>(oldIndexStream.Count);
            for (int i = 0; i < oldIndexStream.Count; i++)
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(useVetices[oldIndexStream[i]] >= 0);
                newIndexStream.Add(useVetices[oldIndexStream[i]]);
            }
            newStreams[newBaseMesh.stream_index].IntData.Clear();
            newStreams[newBaseMesh.stream_index].IntData.AddRange(newIndexStream);
            newMesh.stream_index = newBaseMesh.stream_index;
            newShape.mesh_array.mesh[0] = newMesh;
        }

        private static int GetStreamStride(vtx_attribType vtxAttrib)
        {
            int stride = 1;
            switch (vtxAttrib.type)
            {
                case vtx_attrib_typeType.@int:
                case vtx_attrib_typeType.@uint:
                case vtx_attrib_typeType.@float:
                    stride = 1;
                    break;
                case vtx_attrib_typeType.int2:
                case vtx_attrib_typeType.uint2:
                case vtx_attrib_typeType.float2:
                    stride = 2;
                    break;
                case vtx_attrib_typeType.int3:
                case vtx_attrib_typeType.uint3:
                case vtx_attrib_typeType.float3:
                    stride = 3;
                    break;
                case vtx_attrib_typeType.int4:
                case vtx_attrib_typeType.uint4:
                case vtx_attrib_typeType.float4:
                    stride = 4;
                    break;
                default:
                    Nintendo.Foundation.Contracts.Assertion.Operation.True(false);
                    break;
            }
            return stride;
        }

        private static int CopyStreamIf(G3dStream newStream, G3dStream oldStream, List<int> mark, int stride)
        {
            int numCopyElements = 0;

            // mark が 0 以上の要素のみ newStream から oldStream へコピーを行います。
            Nintendo.Foundation.Contracts.Assertion.Argument.True(newStream.type == oldStream.type);
            if (newStream.type == stream_typeType.@int)
            {
                Nintendo.Foundation.Contracts.Assertion.Argument.True(oldStream.IntData.Count == (mark.Count * stride));

                List<int> newIntData = new List<int>(oldStream.IntData.Count);
                int numVertex = oldStream.IntData.Count / stride;
                for (int i = 0; i < numVertex; i++)
                {
                    if (mark[i] >= 0)
                    {
                        for (int j = 0; j < stride; j++)
                        {
                            newIntData.Add(oldStream.IntData[(i * stride) + j]);
                        }
                    }
                }

                newStream.IntData.Clear();
                newStream.IntData.AddRange(newIntData);
                numCopyElements = newIntData.Count;
            }
            else if (newStream.type == stream_typeType.@float)
            {
                Nintendo.Foundation.Contracts.Assertion.Argument.True(oldStream.FloatData.Count == (mark.Count * stride));

                List<float> newFloatData = new List<float>(oldStream.FloatData.Count);
                int numVertex = oldStream.FloatData.Count / stride;
                for (int i = 0; i < numVertex; i++)
                {
                    if (mark[i] >= 0)
                    {
                        for (int j = 0; j < stride; j++)
                        {
                            newFloatData.Add(oldStream.FloatData[(i * stride) + j]);
                        }
                    }
                }

                newStream.FloatData.Clear();
                newStream.FloatData.AddRange(newFloatData);
                numCopyElements = newFloatData.Count;
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Fail($"Unexpected stream type {newStream.type}");
            }

            newStream.column = oldStream.column;

            return numCopyElements;
        }

        private static void MergeStream(G3dStream newStream, G3dStream oldStream)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(newStream.type == oldStream.type);
            if (newStream.type == stream_typeType.@int)
            {
                // 要素数の変わる -op 以外の最適化が含まれるようになった
                //Nintendo.Foundation.Contracts.Assertion.Operation.True(newStream.IntData.Count == oldStream.IntData.Count);
                newStream.IntData.Clear();
                newStream.IntData.AddRange(oldStream.IntData);
            }
            else if (newStream.type == stream_typeType.@float)
            {
                // 要素数の変わる -op 以外の最適化が含まれるようになった
                //Nintendo.Foundation.Contracts.Assertion.Operation.True(newStream.FloatData.Count == oldStream.FloatData.Count);
                newStream.FloatData.Clear();
                newStream.FloatData.AddRange(oldStream.FloatData);
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(false);
            }
        }

        private static meshType CopyMesh(meshType src)
        {
            meshType newMesh = new meshType
            {
                index = src.index,
                stream_index = src.stream_index,
                mode = src.mode,
                index_hint = src.index_hint,
                quantize_type = src.quantize_type,
                submesh_array = src.submesh_array,
                count = src.count,
            };
            return newMesh;
        }

        //=====================================================================
    }
}
