﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // マージユーティリティ
    internal static    class IfMergeUtility
    {
        // ルートオブジェクトのマージ
        internal static    void MergeRootObject(
            IG3dRootElement newObject, IG3dRootElement oldObject,
            List<G3dStream>    newStream, List<G3dStream> oldStream)
        {
            user_data_arrayType    newUserData    = newObject.user_data_array;
            MergeUserDataArray(ref newUserData,
                               oldObject.user_data_array,
                               newStream, oldStream);
            newObject.user_data_array =    newUserData;

            // <comment> が両方に存在する場合は引数ファイルの内容を優先してマージする。
            // マージファイルのみ存在する場合は    <comment> を引数ファイルにコピーする。
            if (newObject.comment != null &&
                oldObject.comment != null)
            {
                MergeComment(newObject.comment,    oldObject.comment);
            }
            else if    (oldObject.comment != null)
            {
                newObject.comment =    oldObject.comment;
            }

            newObject.tool_data    = oldObject.tool_data;
            newObject.user_tool_data = oldObject.user_tool_data;
        }

        //---------------------------------------------------------------------
        // 関数によるテーブルセットアップ
        internal static    void SetupTableByFunc<TType>(
            Dictionary<TType, TType> table,
            TType[]    newArray, TType[] oldArray,
            Func<TType,    TType, bool> func)
        {
            table.Clear();
            foreach    (TType newObject in    newArray)
            {
                TType oldObject    = Array.Find<TType>(oldArray,
                    delegate(TType target)
                    {
                        return func(newObject, target);
                    });
                if (oldObject == null) { continue; }
                table.Add(newObject, oldObject);
            }
        }

        // 名前によるテーブルセットアップ
        internal static    void SetupTableByName<TType>(
            Dictionary<TType, TType> table,
            TType[]    newArray, TType[] oldArray)
            where TType    : IG3dNamedElement
        {
            IfMergeUtility.SetupTableByFunc<TType>(
                table, newArray, oldArray,
                delegate(TType newObject, TType    oldObject)
                {
                    return (newObject.name == oldObject.name);
                });
        }

        // 関数とペアによるテーブルセットアップ
        internal static    void SetupTableByFunc<TType>(
            Dictionary<TType, TType> table,
            TType[]    newArray, TType[] oldArray,
            IEnumerable<IfMergeSrcDstPair> pair,
            Func<TType,    string,    bool> func)
            where TType    : IG3dNamedElement
        {
            // 名前のペアリストから辞書に変換する。
            Dictionary<string, string> pairTable =
                new    Dictionary<string, string>();
            IfMergeUtility.ConvertPairToDictionary(pairTable, pair);

            // マージするデータの辞書を作成する。
            foreach    (TType newObject in    newArray)
            {
                // マテリアルのマージ元となる名前を決定する。
                // オプションによりマージするマテリアルのペアが存在していたら、その指定にしたがう。
                // ペアがなければそのオブジェクトはマージを行わない。
                if (!pairTable.ContainsKey(newObject.name))    { continue;    }

                string targetName =    pairTable[newObject.name];
                TType oldObject    = Array.Find<TType>(oldArray,
                    delegate(TType target)
                    {
                        return func(target,    targetName);
                    });

                if (oldObject == null) { continue; }
                table.Add(newObject, oldObject);
            }
        }

        // 名前とペアによるテーブルセットアップ
        internal static    void SetupTableByName<TType>(
            Dictionary<TType, TType> table,
            TType[]    newArray, TType[] oldArray,
            IEnumerable<IfMergeSrcDstPair> pair)
            where TType    : IG3dNamedElement
        {
            IfMergeUtility.SetupTableByFunc<TType>(
                table, newArray, oldArray, pair,
                delegate(TType oldObject, string name)
                {
                    return (oldObject.name == name);
                });
        }

        // インデックスによるテーブルセットアップ
        internal static    void SetupTableByIndex<TType>(
            Dictionary<TType, TType> table,
            TType[]    newArray, TType[] oldArray)
        {
            table.Clear();
            int    length = newArray.Length;
            if (length > oldArray.Length) {    length = oldArray.Length; }
            for    (int i = 0;    i <    length;    i++)
            {
                table.Add(newArray[i], oldArray[i]);
            }
        }

        // テーブルによるマージ
        internal static    void MergeByTable<TType>(
            TType[]    newObjects,
            Dictionary<TType, TType> mergeTable,
            Action<TType, TType> mergeMethod)
        {
            foreach    (TType newObject in    newObjects)
            {
                if (!mergeTable.ContainsKey(newObject))    { continue;    }
                TType oldObject    = mergeTable[newObject];
                mergeMethod(newObject, oldObject);
            }
        }

        // テーブルによるマージ
        internal static    void MergeByTable<TType, InfoType>(
            TType[]    newObjects,
            Dictionary<TType, TType> mergeTable,
            Action<TType, TType, InfoType> mergeMethod,
            InfoType mergeInfo)
        {
            foreach    (TType newObject in    newObjects)
            {
                if (!mergeTable.ContainsKey(newObject))    { continue;    }
                TType oldObject    = mergeTable[newObject];
                mergeMethod(newObject, oldObject, mergeInfo);
            }
        }

        // テーブルによるマージ
        internal static    void MergeByTable<TType>(
            TType[]    newObjects,
            Dictionary<TType, TType> mergeTable,
            List<G3dStream>    newStream,
            List<G3dStream>    oldStream,
            Action<TType, List<G3dStream>, TType, List<G3dStream>> mergeMethod)
        {
            foreach    (TType newObject in    newObjects)
            {
                if (!mergeTable.ContainsKey(newObject))    { continue;    }
                TType oldObject    = mergeTable[newObject];
                mergeMethod(newObject, newStream, oldObject, oldStream);
            }
        }

        // テーブルによるマージ
        internal static    void MergeByTable<TType, InfoType>(
            TType[]    newObjects,
            Dictionary<TType, TType> mergeTable,
            List<G3dStream>    newStream,
            List<G3dStream>    oldStream,
            Action<TType, List<G3dStream>, TType, List<G3dStream>, InfoType> mergeMethod,
            InfoType mergeInfo)
        {
            foreach    (TType newObject in    newObjects)
            {
                if (!mergeTable.ContainsKey(newObject))    { continue;    }
                TType oldObject    = mergeTable[newObject];
                mergeMethod(newObject, newStream, oldObject, oldStream,    mergeInfo);
            }
        }

        // テーブルによるマージ
        internal static    void MergeByTable<TRootType, TType>(
            TRootType newRoot, TRootType oldRoot,
            TType[]    newObjects,
            Dictionary<TType, TType> mergeTable,
            List<G3dStream>    newStreams,
            List<G3dStream>    oldStreams,
            Action<TRootType, TType, List<G3dStream>, TRootType, TType,    List<G3dStream>> mergeMethod)
        {
            foreach    (TType newObject in    newObjects)
            {
                if (!mergeTable.ContainsKey(newObject))    { continue;    }
                TType oldObject    = mergeTable[newObject];
                mergeMethod(newRoot, newObject,    newStreams,    oldRoot, oldObject,    oldStreams);
            }
        }

        // ペアの配列から辞書に変換する
        internal static    void ConvertPairToDictionary(
            Dictionary<string, string> dict,
            IEnumerable<IfMergeSrcDstPair> pair)
        {
            foreach    (IfMergeSrcDstPair p in    pair)
            {
                dict.Add(p.Dst,    p.Src);
            }
        }

        //---------------------------------------------------------------------
        // <comment> をコピー
        internal static    void MergeComment(commentType dest,    commentType    src)
        {
            if (dest ==    null ||    src    == null)
            {
                return;
            }

            if (dest.label == null || dest.label.Length == 0)
            {
                dest.label = src.label;
            }

            if (dest.color == null || dest.color.Length    == 0)
            {
                dest.color = src.color;
            }

            if (dest.text == null || dest.text.Length == 0)
            {
                dest.text =    src.text;
            }
        }

        //---------------------------------------------------------------------
        // カーブをコピー
        internal static    IG3dQuantizedCurve CopyCurve(IG3dQuantizedCurve srcCurve)
        {
            IG3dQuantizedCurve destCurve    = null;
            if (srcCurve.GetType() == typeof(hermite_curveType))
            {
                hermite_curveType srcHermiteCurve =    srcCurve as    hermite_curveType;
                hermite_curveType hermiteCurve = new hermite_curveType();
                hermiteCurve.count = srcCurve.count;
                hermiteCurve.post_wrap = srcCurve.post_wrap;
                hermiteCurve.pre_wrap =    srcCurve.pre_wrap;
                hermiteCurve.stream_index =    srcCurve.stream_index;
                hermiteCurve.frame_type    = srcCurve.frame_type;
                hermiteCurve.key_type =    srcCurve.key_type;
                hermiteCurve.scale = srcCurve.scale;
                hermiteCurve.offset    = srcCurve.offset;
                hermiteCurve.baked = srcHermiteCurve.baked;
                hermiteCurve.stream_index =    srcCurve.stream_index;

                destCurve =    hermiteCurve;
            }
            else if    (srcCurve.GetType()    == typeof(linear_curveType))
            {
                linear_curveType srcLinearCurve    = srcCurve as linear_curveType;
                linear_curveType linearCurve = new linear_curveType();
                linearCurve.count =    srcCurve.count;
                linearCurve.post_wrap =    srcCurve.post_wrap;
                linearCurve.pre_wrap = srcCurve.pre_wrap;
                linearCurve.stream_index = srcCurve.stream_index;
                linearCurve.frame_type = srcCurve.frame_type;
                linearCurve.key_type = srcCurve.key_type;
                linearCurve.scale =    srcCurve.scale;
                linearCurve.offset = srcCurve.offset;
                linearCurve.baked =    srcLinearCurve.baked;
                linearCurve.stream_index = srcCurve.stream_index;

                destCurve =    linearCurve;
            }
            else if    (srcCurve.GetType()    == typeof(step_curveType))
            {
                step_curveType srcStepCurve    = srcCurve as step_curveType;
                step_curveType stepCurve = new step_curveType();
                stepCurve.count    = srcCurve.count;
                stepCurve.post_wrap    = srcCurve.post_wrap;
                stepCurve.pre_wrap = srcCurve.pre_wrap;
                stepCurve.stream_index = srcCurve.stream_index;
                stepCurve.frame_type = srcCurve.frame_type;
                stepCurve.key_type = srcCurve.key_type;
                stepCurve.scale    = srcCurve.scale;
                stepCurve.offset = srcCurve.offset;
                stepCurve.baked    = srcStepCurve.baked;
                stepCurve.stream_index = srcCurve.stream_index;

                destCurve =    stepCurve;
            }
            return destCurve;
        }

        internal static    IG3dQuantizedCurve CopyCurve(IG3dCurve srcCurve)
        {
            IG3dQuantizedCurve destCurve    = null;
            if (srcCurve.GetType() == typeof(original_hermiteType))
            {
                original_hermiteType orgHermiteCurve = srcCurve    as original_hermiteType;
                hermite_curveType hermiteCurve = new hermite_curveType();
                hermiteCurve.count = srcCurve.count;
                hermiteCurve.post_wrap = srcCurve.post_wrap;
                hermiteCurve.pre_wrap =    srcCurve.pre_wrap;
                hermiteCurve.stream_index =    srcCurve.stream_index;
                hermiteCurve.frame_type    = curve_frame_typeType.none;
                hermiteCurve.key_type =    curve_key_typeType.none;
                hermiteCurve.scale = 1.0f;
                hermiteCurve.offset    = 0.0f;
                hermiteCurve.baked = orgHermiteCurve.baked;
                hermiteCurve.stream_index =    srcCurve.stream_index;

                destCurve =    hermiteCurve;
            }
            else if    (srcCurve.GetType()    == typeof(original_linearType))
            {
                original_linearType    orgLinearCurve = srcCurve as original_linearType;
                linear_curveType linearCurve = new linear_curveType();
                linearCurve.count =    srcCurve.count;
                linearCurve.post_wrap =    srcCurve.post_wrap;
                linearCurve.pre_wrap = srcCurve.pre_wrap;
                linearCurve.stream_index = srcCurve.stream_index;
                linearCurve.frame_type = curve_frame_typeType.none;
                linearCurve.key_type = curve_key_typeType.none;
                linearCurve.scale =    1.0f;
                linearCurve.offset = 0.0f;
                linearCurve.baked =    orgLinearCurve.baked;
                linearCurve.stream_index = srcCurve.stream_index;

                destCurve =    linearCurve;
            }
            else if    (srcCurve.GetType()    == typeof(original_stepType))
            {
                original_stepType orgStepCurve = srcCurve as original_stepType;
                step_curveType stepCurve = new step_curveType();
                stepCurve.count    = srcCurve.count;
                stepCurve.post_wrap    = srcCurve.post_wrap;
                stepCurve.pre_wrap = srcCurve.pre_wrap;
                stepCurve.stream_index = srcCurve.stream_index;
                stepCurve.frame_type = curve_frame_typeType.none;
                stepCurve.key_type = curve_key_typeType.none;
                stepCurve.scale    = 1.0f;
                stepCurve.offset = 0.0f;
                stepCurve.baked    = orgStepCurve.baked;
                stepCurve.stream_index = srcCurve.stream_index;

                destCurve =    stepCurve;
            }
            return destCurve;
        }

        //---------------------------------------------------------------------
        // ユーザーデータをコピー
        internal static    user_dataType CopyUserData(
            user_dataType userData,
            List<G3dStream>    destStream,
            List<G3dStream>    srcStream)
        {
            user_dataType newData =    new    user_dataType();
            newData.index =    userData.index;
            newData.name = userData.name;
            if (userData.Item != null)
            {
                object item    = null;

                if (userData.Item is user_intType)
                {
                    user_intType newIntData    = new user_intType();
                    user_intType intData = userData.Item as    user_intType;
                    newIntData.count = intData.count;
                    newIntData.Value = intData.Value;
                    item = newIntData;
                }
                else if    (userData.Item is user_floatType)
                {
                    user_floatType newFloatData    = new user_floatType();
                    user_floatType floatData = userData.Item as    user_floatType;
                    newFloatData.count = floatData.count;
                    newFloatData.Value = floatData.Value;
                    item = newFloatData;
                }
                else if    (userData.Item is user_stringType)
                {
                    user_stringType    newStringData =    new    user_stringType();
                    user_stringType    stringData = userData.Item as user_stringType;
                    newStringData.count    = stringData.count;
                    if (stringData.@string != null)
                    {
                        newStringData.@string =    new    string[stringData.@string.Length];
                        stringData.@string.CopyTo(newStringData.@string, 0);
                    }
                    item = newStringData;
                }
                else if    (userData.Item is user_wstringType)
                {
                    user_wstringType newWstringData    = new user_wstringType();
                    user_wstringType wstringData = userData.Item as    user_wstringType;
                    newWstringData.count = wstringData.count;
                    if (wstringData.wstring    != null)
                    {
                        newWstringData.wstring = new string[wstringData.wstring.Length];
                        wstringData.wstring.CopyTo(newWstringData.wstring, 0);
                    }
                    item = newWstringData;
                }
                else if    (userData.Item is user_streamType)
                {
                    user_streamType    newStreamData =    new    user_streamType();
                    user_streamType    streamData = userData.Item as user_streamType;
                    newStreamData.size = streamData.size;
                    if (streamData.stream_index    >= 0)
                    {
                        newStreamData.stream_index = destStream.Count;
                        destStream.Add(srcStream[streamData.stream_index]);
                    }
                    else
                    {
                        newStreamData.stream_index = -1;
                    }
                    item = newStreamData;
                }
                newData.Item = item;
            }
            else
            {
                newData.Item = null;
            }

            return newData;
        }

        //---------------------------------------------------------------------
        // ユーザーデータ配列をコピーします。
        internal static    user_data_arrayType    CopyUserDataArray(
            user_data_arrayType    src,
            List<G3dStream>    destStream,
            List<G3dStream>    srcStream)
        {
            if (src    == null    || src.user_data ==    null) {    return null; }

            List<user_dataType>    newArray = new List<user_dataType>();
            // ユーザーデータをコピーする
            foreach    (user_dataType srcData in src.user_data)
            {
                newArray.Add(CopyUserData(srcData, destStream, srcStream));
            }

            user_data_arrayType    newUserDataArray = new user_data_arrayType();
            newUserDataArray.length    = newArray.Count;
            newUserDataArray.user_data = newArray.ToArray();
            return newUserDataArray;
        }

        //---------------------------------------------------------------------
        // ユーザーデータ配列をマージします。
        // dest    が null で src がデータを持つ場合、dest    にユーザーデータを作成します。
        // そのため、dest は参照で受け取ります。
        internal static    void MergeUserDataArray(
            ref    user_data_arrayType    dest,
            user_data_arrayType    src,
            List<G3dStream>    destStream,
            List<G3dStream>    srcStream)
        {
            if (src    == null    || src.user_data ==    null) {    return;    }

            List<user_dataType>    newArray = new List<user_dataType>();
            // 既存のユーザーデータを保存する
            if (dest !=    null &&    dest.user_data != null)
            {
                foreach    (user_dataType destData    in dest.user_data)
                {
                    newArray.Add(destData);
                }
            }
            // ユーザーデータをマージする
            foreach    (user_dataType srcData in src.user_data)
            {
                int    index =    newArray.FindIndex(
                    delegate (user_dataType    data)
                    {
                        return (data.name == srcData.name);
                    });
                if (index <    0)
                {
                    newArray.Add(CopyUserData(srcData, destStream, srcStream));
                }
            }

            // 配列の内容を入れ替える
            if (newArray.Count > 0)
            {
                if (dest ==    null)
                {
                    dest = new user_data_arrayType();
                }

                dest.length    = newArray.Count;
                dest.user_data = newArray.ToArray();
            }
        }
    }
}
