﻿using System;
using System.Collections.Generic;
using System.Linq;

using Microsoft.Scripting.Utils;

using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // テクスチャパターンアニメーションマージャ
    public static class IfMaterialAnimMerger
    {
        // マージ用の情報
        public class IfMaterialAnimMergerInfo
        {
            public IEnumerable<IfMergeSrcDstPair> MaterialNamePairTable { get; set; }
            public IEnumerable<IfMergeSrcDstPair> SamplerNamePairTable { get; set; }
        }
        // マージ
        public static void Merge(
            material_animType newMaterialAnim,
            List<G3dStream> newStreams,
            material_animType oldMaterialAnim,
            List<G3dStream> oldStreams)
        {
            Merge(newMaterialAnim, newStreams, oldMaterialAnim, oldStreams, null);
        }

        // オプション付きマージ
        public static void Merge(
            material_animType newMaterialAnim,
            List<G3dStream> newStreams,
            material_animType oldMaterialAnim,
            List<G3dStream> oldStreams,
            IfMaterialAnimMergerInfo info)
        {
            // <user_data_array> <tool_data> <user_tool_data> <comment>
            IfMergeUtility.MergeRootObject(
                newMaterialAnim, oldMaterialAnim,
                newStreams, oldStreams);

            // quantize_tolerance_～ をマージする
            newMaterialAnim.material_anim_info.quantize_tolerance_tex_scale =
                oldMaterialAnim.material_anim_info.quantize_tolerance_tex_scale;
            newMaterialAnim.material_anim_info.quantize_tolerance_tex_rotate =
                oldMaterialAnim.material_anim_info.quantize_tolerance_tex_rotate;
            newMaterialAnim.material_anim_info.quantize_tolerance_tex_translate =
                oldMaterialAnim.material_anim_info.quantize_tolerance_tex_translate;

            // マージファイルにマテリアルが存在していない時は処理を行う必要がない
            if (!oldMaterialAnim.HasMaterials()) { return; }

            var oldMaterialArray = oldMaterialAnim.per_material_anim_array.per_material_anim;
            var newMaterialList = newMaterialAnim.per_material_anim_array.per_material_anim.ToList();

            //----------------------------------------------------------
            // シェーダーパラメーターアニメーション
            // オリジナルマテリアルアニメーションの存在を確認する
            var hasOriginalMaterialAnim = newMaterialAnim.HasOriginalMaterials() && oldMaterialAnim.HasOriginalMaterials();

            //----------------------------------------------------------
            // テクスチャパターンアニメーション
            // テクスチャパターンがなく、パターンアニメーションのみが存在するデータが入力されたときはエラーとする。
            if (newMaterialAnim.tex_pattern_array == null &&
                newMaterialList.Any(x => x.tex_pattern_anim_array != null))
            {
                throw new Exception("<tex_pattern> doesn't exist in the input file.");
            }
            if (oldMaterialAnim.tex_pattern_array == null &&
                oldMaterialArray.Any(x => x.tex_pattern_anim_array != null))
            {
                throw new Exception("<tex_pattern> doesn't exist in the merge file.");
            }

            // テクスチャパターンアニメーションの存在を確認する。
            var hasNewTexPatternAnim = newMaterialAnim.HasTexturePatternAnim();
            var hasOldTexPatternAnim = oldMaterialAnim.HasTexturePatternAnim();

            //----------------------------------------------------------
            // マテリアルビジビリティアニメーション
            // マテリアルビジビリティアニメーションの存在を確認する。
            var hasVisibilityAnim =
                oldMaterialArray.Any(x => x.material_visibility_anim != null);

            //----------------------------------------------------------
            // シェーダーパラメーターアニメーションのマージ
            // オリジナルマテリアルアニメーションが存在していなければマージ処理を行わない
            if (hasOriginalMaterialAnim)
            {
                // newMaterialAnim の <shader_param_anim_array> 削除し、oldMaterialAnim からコピーする。
                foreach (var newMat in newMaterialList)
                {
                    newMat.shader_param_anim_array = null;
                }
                foreach (var oldMat in oldMaterialArray)
                {
                    var newMat = newMaterialList.FirstOrDefault(x => x.mat_name == oldMat.mat_name);
                    if (newMat == null)
                    {
                        newMat = new per_material_animType { mat_name = oldMat.mat_name };
                        newMaterialList.Add(newMat);
                    }
                    newMat.shader_param_anim_array = oldMat.shader_param_anim_array;
                }
                // <original_per_material_anim> をキーに、<per_material_anim> を値に持つ辞書。
                // 同じ名前のマテリアル同士でアニメーションカーブをマージするために使用する。
                var matAnimDict =
                    new Dictionary<original_material_animType, per_material_animType>();

                if (newMaterialList.Any(x => x.shader_param_anim_array != null))
                {
                    //  shader_param_anim_array が使用する stream の移植。
                    foreach (var newMat in newMaterialList)
                    {
                        foreach (var paramAnim in newMat.shader_param_anim_array.Items)
                        {
                            foreach (var animTarget in paramAnim.param_anim_target)
                            {
                                IG3dCurve curve = animTarget.Curve;
                                if (curve == null)
                                {
                                    continue;
                                }
                                G3dStream copyStream = oldStreams[curve.stream_index];
                                curve.stream_index = newStreams.Count;
                                newStreams.Add(copyStream);
                            }
                        }
                    }

                    // original_per_material_anim_array をキーにした辞書を作成する
                    var shaderAnimMatArray =
                        newMaterialList.Where(x => x.shader_param_anim_array != null);
                    var originalMatAnimArray =
                        newMaterialAnim.original_per_material_anim_array.Items;
                    if (info == null || info.MaterialNamePairTable == null)
                    {
                        foreach (var shaderAnimMat in shaderAnimMatArray)
                        {
                            var matAnim =
                                Array.Find(originalMatAnimArray,
                                    mat => (mat.mat_name == shaderAnimMat.mat_name));
                            if (matAnim == null) { continue; }
                            matAnimDict.Add(matAnim, shaderAnimMat);
                        }
                    }
                    else
                    {
                        // マテリアル名のペアテーブルが指定されていたら、その指定に従って辞書を作成する

                        // 名前のペアリストから辞書に変換する。
                        var pairTable = new Dictionary<string, string>();
                        IfMergeUtility.ConvertPairToDictionary(pairTable, info.MaterialNamePairTable);

                        foreach (var shaderAnimMat in shaderAnimMatArray)
                        {
                            if (!pairTable.ContainsKey(shaderAnimMat.mat_name)) { continue; }

                            var matName = pairTable[shaderAnimMat.mat_name];
                            var matAnim =
                                Array.Find(originalMatAnimArray,
                                    mat => (mat.mat_name == matName));
                            if (matAnim == null) { continue; }
                            matAnimDict.Add(matAnim, shaderAnimMat);
                        }
                    }
                }

                // アニメーションカーブをマージする。
                // <per_material_anim> の mat_name と同じ名前の <original_per_material_anim> が持つカーブをマージする。
                // さらに <param_anim> の original_hint と同じ hint を持つ <original_color_anim>
                // 及び <original_texsrt_anim> が存在する場合のみマージが行われる。
                if (newMaterialAnim.original_per_material_anim_array != null)
                {
                    foreach (var originalMatAnim in newMaterialAnim.original_per_material_anim_array.Items)
                    {
                        // 同じ名前のマテリアルを取得する。
                        if (!matAnimDict.ContainsKey(originalMatAnim)) { continue; }

                        var shaderAnim = matAnimDict[originalMatAnim];

                        // <param_anim> の original_hint と同じ <original_color_anim> hint があれば
                        // カーブを再設定する。
                        if (originalMatAnim.original_color_anim_array != null)
                        {
                            foreach (var colorAnim in
                                originalMatAnim.original_color_anim_array.original_color_anim)
                            {
                                var paramAnim = GetParamAnim(shaderAnim, colorAnim.hint);
                                if (paramAnim != null)
                                {
                                    // カーブをマージする
                                    MergeColorAnim(
                                        paramAnim,
                                        newStreams,
                                        colorAnim,
                                        oldStreams,
                                        newMaterialAnim.material_anim_info);
                                }
                            }
                        }

                        // <param_anim> の original_hint と同じ <original_texsrt_anim> hint があれば
                        // カーブを再設定する。
                        if (originalMatAnim.original_texsrt_anim_array != null)
                        {
                            foreach (var texsrtAnim in
                                originalMatAnim.original_texsrt_anim_array.original_texsrt_anim)
                            {
                                var paramAnim = GetParamAnim(shaderAnim, texsrtAnim.hint);
                                if (paramAnim != null)
                                {
                                    // カーブをマージする
                                    MergeTexsrtAnim(
                                        paramAnim,
                                        newStreams,
                                        texsrtAnim,
                                        oldStreams,
                                        newMaterialAnim.material_anim_info);
                                }
                            }
                        }
                    }
                }
            }

            //----------------------------------------------------------
            // テクスチャパターンアニメーションのマージ
            if (hasOldTexPatternAnim)
            {
                // <per_material_animType> のテーブルを作成する
                // キーとして oldMaterialAnim 内の <per_material_animType> を、
                // 値としてキーと同じ mat_name を持つ newMaterialAnim 内の <per_material_animType> を格納する。
                var texPatternMatTable = new Dictionary<per_material_animType, per_material_animType>();
                var oldPatternAnimMatArray =
                    oldMaterialArray.Where(x => x.tex_pattern_anim_array != null).ToList();
                var newPatternAnimMatArray =
                    newMaterialList.Where(x => x.tex_pattern_anim_array != null).ToList();
                IfMergeUtility.SetupTableByFunc(
                    texPatternMatTable,
                    oldPatternAnimMatArray.ToArray(),
                    newPatternAnimMatArray.ToArray(),
                    (newObject, oldObject) => (newObject.mat_name == oldObject.mat_name));

                if (hasNewTexPatternAnim)
                {
                    // oldMaterialAnim から newMaterialAnim にコピーしたテクスチャパターンの
                    // 新しいインデックスを記録した配列。
                    int[] newTexPatternIdx;

                    // テクスチャパターンをマージする
                    MergeTexturePattern(
                        newMaterialAnim,
                        oldMaterialAnim,
                        texPatternMatTable,
                        oldStreams,
                        out newTexPatternIdx);

                    // <tex_pattern_mat_anim> をマージする。
                    // <tex_pattern_mat_anim> mat_name と <pattern_anim_target> sampler_name
                    // が重複したら newMaterialAnim を優先する。
                    foreach (var oldPatternAnimMat in oldPatternAnimMatArray)
                    {
                        per_material_animType newMat;
                        if (!texPatternMatTable.ContainsKey(oldPatternAnimMat))
                        {
                            // newMaterialAnim に oldMaterialAnim の <tex_pattern_mat_anim>
                            // が存在していない場合コピーする。
                            newMat = new per_material_animType();
                            newMaterialList.Add(newMat);
                        }
                        else
                        {
                            newMat = texPatternMatTable[oldPatternAnimMat];
                        }

                        MergeTexPatternAnim(
                            newMat,
                            newStreams,
                            oldPatternAnimMat,
                            oldStreams,
                            newTexPatternIdx);
                    }
                }
                else //if (hasOldTexPatternAnim)
                {
                    // oldMaterialAnim にのみ <tex_pattern> が存在している場合、
                    // oldMaterialAnim の <tex_pattern> をそのままコピーする。
                    newMaterialAnim.tex_pattern_array = oldMaterialAnim.tex_pattern_array;
                    foreach (var oldMat in oldPatternAnimMatArray)
                    {
                        var newMat = newMaterialList.FirstOrDefault(
                            x => x.mat_name == oldMat.mat_name);
                        if (newMat == null)
                        {
                            newMat = new per_material_animType { mat_name = oldMat.mat_name };
                            newMaterialList.Add(newMat);
                        }
                        newMat.tex_pattern_anim_array = oldMat.tex_pattern_anim_array;
                        foreach (var animTarget in newMat.tex_pattern_anim_array.pattern_anim)
                        {
                            var curve = animTarget.Curve;
                            if (curve == null)
                            {
                                continue;
                            }
                            var copyStream = oldStreams[curve.stream_index];
                            curve.stream_index = newStreams.Count;
                            newStreams.Add(copyStream);
                        }
                    }
                }
            }
            // newTexPatternAnim にのみアニメーションが存在している場合は
            // マージする必要がないので処理を行わない。

            if (hasVisibilityAnim)
            {
                var oldVisibilityAnimMatArray =
                    oldMaterialArray.Where(x => x.material_visibility_anim != null).ToList();
                var newVisibilityAnimMatArray =
                    newMaterialList.Where(x => x.material_visibility_anim != null).ToList();

                // マテリアルアニメーションはカーブをつけたアニメーションしか出力しないので、
                // newMatVisAnim 側に無い mat_name を持つ oldMatVisibilityAnim をマージする
                foreach (var oldMat in oldVisibilityAnimMatArray)
                {
                    if (newVisibilityAnimMatArray.Any(x => x.mat_name == oldMat.mat_name))
                    {
                        continue;
                    }
                    var newMat = newMaterialList.FirstOrDefault(
                        x => x.mat_name == oldMat.mat_name);

                    if (newMat == null)
                    {
                        newMat = new per_material_animType { mat_name = oldMat.mat_name };
                        newMaterialList.Add(newMat);
                    }
                    newMat.material_visibility_anim = oldMat.material_visibility_anim;
                    var curve = newMat.material_visibility_anim.step_curve;
                    if (curve == null)
                    {
                        continue;
                    }
                    var copyStream = oldStreams[curve.stream_index];
                    curve.stream_index = newStreams.Count;
                    newStreams.Add(copyStream);
                }
            }

            // <per_material_anim> を mat_name でソートし、インデックスを振り直す。
            newMaterialList.Sort((lhs, rhs) => string.CompareOrdinal(lhs.mat_name, rhs.mat_name));
            for (var i = 0; i < newMaterialList.Count; i++)
            {
                newMaterialList[i].index = i;
            }

            // <per_material_anim_array> を置きかえる。
            newMaterialAnim.per_material_anim_array.length = newMaterialList.Count;
            newMaterialAnim.per_material_anim_array.Items = newMaterialList.ToArray();

            // 使用されていないストリームを削除し、
            // 残ったストリームを中間ファイルの出現順にソートする。
            StreamUtility.SortStream(newMaterialAnim, newStreams);
            newMaterialAnim.per_material_anim_array.UpdateHint();
        }

        private static void MergeTexsrtAnim(param_animType paramAnim, List<G3dStream> newStreams, original_texsrt_animType texsrtAnim, List<G3dStream> oldStreams, material_anim_infoType materialAnimInfo)
        {
            var shaderAnimInfo = CopyToShaderAnimInfo(materialAnimInfo);

            IfShaderParamAnimMerger.MergeTexsrtAnim(
                paramAnim,
                newStreams,
                texsrtAnim,
                oldStreams,
                shaderAnimInfo);
        }

        private static void MergeColorAnim(param_animType paramAnim, List<G3dStream> newStreams, original_color_animType colorAnim, List<G3dStream> oldStreams, material_anim_infoType materialAnimInfo)
        {
            var shaderAnimInfo = CopyToShaderAnimInfo(materialAnimInfo);

            IfShaderParamAnimMerger.MergeColorAnim(
                paramAnim,
                newStreams,
                colorAnim,
                oldStreams,
                shaderAnimInfo);
        }

        internal static shader_param_anim_infoType CopyToShaderAnimInfo(material_anim_infoType materialAnimInfo)
        {
            var shaderAnimInfo = new shader_param_anim_infoType
                                     {
                                         frame_count = materialAnimInfo.frame_count,
                                         loop = materialAnimInfo.loop,
                                         frame_resolution = materialAnimInfo.frame_resolution,
                                         dcc_preset = materialAnimInfo.dcc_preset,
                                         dcc_start_frame = materialAnimInfo.dcc_start_frame,
                                         dcc_end_frame = materialAnimInfo.dcc_end_frame,
                                         dcc_fps = materialAnimInfo.dcc_fps,
                                         bake_all = materialAnimInfo.bake_all,
                                         bake_tolerance_color = materialAnimInfo.bake_tolerance_color,
                                         bake_tolerance_tex_scale =
                                             materialAnimInfo.bake_tolerance_tex_scale,
                                         bake_tolerance_tex_rotate =
                                             materialAnimInfo.bake_tolerance_tex_rotate,
                                         bake_tolerance_tex_translate =
                                             materialAnimInfo.bake_tolerance_tex_translate,
                                         quantize_tolerance_tex_scale =
                                             materialAnimInfo.quantize_tolerance_tex_scale,
                                         quantize_tolerance_tex_rotate =
                                             materialAnimInfo.quantize_tolerance_tex_rotate,
                                         quantize_tolerance_tex_translate =
                                             materialAnimInfo.quantize_tolerance_tex_translate
                                     };
            return shaderAnimInfo;
        }

        // 指定された original_hint を持つ param_anim を取得します。
        private static param_animType GetParamAnim(
            per_material_animType matAnim,
            string originalHint)
        {
            return matAnim.shader_param_anim_array.param_anim.FirstOrDefault(paramAnim => paramAnim.original_hint == originalHint);
        }

        // <tex_pattern> を混ぜ合わせる。
        // newTexPatternAnim 内の <tex_pattern> はそのままにし、oldTexPatternAnim
        // でのみ使用されている <tex_pattern> を newTexPatternAnim の <tex_pattern_array> に追加する。
        // 追加する <tex_pattern> は順番を保つようにする。
        private static void MergeTexturePattern(
            material_animType newTexPatternAnim,
            material_animType oldTexPatternAnim,
            Dictionary<per_material_animType, per_material_animType> texPatternMatTable,
            List<G3dStream> oldStreams,
            out int[] newTexPatternIdx)
        {
            // <tex_pattern> の使用フラグ
            var useOldTexPattern = new bool[oldTexPatternAnim.tex_pattern_array.length];
            // テクスチャパターンアニメーションを持つマテリアルのみを列挙
            var newPatternAnimMatArray = newTexPatternAnim.per_material_anim_array.Items.Where(x => x.tex_pattern_anim_array != null).ToArray();
            var oldPatternAnimMatArray = oldTexPatternAnim.per_material_anim_array.Items.Where(x => x.tex_pattern_anim_array != null).ToArray();
            // oldTexPatternAnim にのみ存在しているテクスチャパターンアニメーションが
            // 参照している <tex_pattern> の useOldTexPattern を true にする。
            foreach (var oldTexPatternAnimMat in oldPatternAnimMatArray)
            {
                // マージファイル内と引数ファイルで同じ mat_name を持つ <tex_pattern_mat_anim>
                // が存在するか確認する。
                var newTexPatternMatAnim = texPatternMatTable.ContainsKey(oldTexPatternAnimMat) ? texPatternMatTable[oldTexPatternAnimMat] : null;

                foreach (var pattern_anim in oldTexPatternAnimMat.tex_pattern_anim_array.pattern_anim)
                {
                    //foreach (pattern_anim_targetType target in texPatternMatAnim.pattern_anim_target)
                    {
                        // カーブが存在していなければ tex_pattern をマージする必要がない。
                        if (pattern_anim.Curve == null)
                        {
                            continue;
                        }
                        // 重複する hint をもつ pattern_anim_target が存在していないか検索し、
                        // 存在していれば newTexPatternAnim のアニメーションが優先されるため、
                        // oldTexPatternAnim の tex_pattern をマージしない。
                        if (newTexPatternMatAnim != null)
                        {
                            var index = newTexPatternMatAnim.tex_pattern_anim_array.pattern_anim.FindIndex(
                                tgt => tgt.hint == pattern_anim.hint);
                            if (index >= 0)
                            {
                                continue;
                            }
                        }

                        // カーブが参照するテクスチャパターンに使用している印をつける。
                        var stream = oldStreams[pattern_anim.Curve.stream_index];
                        var numKey = stream.FloatData.Count / stream.column;
                        for (var iKey = 0; iKey < numKey; iKey++)
                        {
                            var texPatternIndex = (int)stream.FloatData[iKey * stream.column + 1];
                            useOldTexPattern[texPatternIndex] = true;
                        }
                    }
                }
            }

            // oldTexPatternAnim の <tex_pattern> で使用されている
            var newTexPatternArray =
                new List<tex_patternType>(newTexPatternAnim.tex_pattern_array.tex_pattern);
            var oldTexPatternArray =
                new List<tex_patternType>(oldTexPatternAnim.tex_pattern_array.tex_pattern);
            newTexPatternIdx = new int[oldTexPatternAnim.tex_pattern_array.length];

            for (var i = 0; i < oldTexPatternArray.Count; i++)
            {
                // 使用されていない <tex_pattern> はコピーしない。
                if (!useOldTexPattern[i])
                {
                    continue;
                }

                var texPattern = oldTexPatternArray[i];

                var index = newTexPatternArray.FindIndex(tp => tp.tex_name == texPattern.tex_name);
                if (index >= 0)
                {
                    // newTexPatternAnim にテクスチャパターンが存在していたら
                    // 既存のパターンのインデックスを使用する。
                    newTexPatternIdx[i] = index;
                }
                else
                {
                    // newTexPatternAnim にテクスチャパターンが存在していなければ末尾に追加する。
                    newTexPatternIdx[i] = newTexPatternArray.Count;
                    newTexPatternArray.Add(texPattern);
                }
            }
            // <tex_pattern_array> を置きかえる。
            newTexPatternAnim.tex_pattern_array.tex_pattern = newTexPatternArray.ToArray();
            newTexPatternAnim.tex_pattern_array.length = newTexPatternArray.Count;
        }

        // <tex_pattern_mat_anim> をコピーします。
        private static void MergeTexPatternAnim(
            per_material_animType newTexPatternAnim,
            List<G3dStream> newStream,
            per_material_animType oldTexPatternAnim,
            List<G3dStream> oldStream,
            int[] newTexPatternIdx)
        {
            var newPatternAnimTargetArray = newTexPatternAnim.tex_pattern_anim_array != null
                                                 ? new List<pattern_anim_targetType>(newTexPatternAnim.tex_pattern_anim_array.pattern_anim)
                                                 : new List<pattern_anim_targetType>();

            foreach (var oldPatternAnimTarget in oldTexPatternAnim.tex_pattern_anim_array.pattern_anim)
            {
                // 同じ hint をもつターゲットを検索する。
                // ターゲットが存在していなければ oldPatternAnimTarget をコピーする。
                var targetIdx = newPatternAnimTargetArray.FindIndex(tgt => tgt.hint == oldPatternAnimTarget.hint);
                if (targetIdx >= 0)
                {
                    continue;
                }
                var newPatternAnimTarget = new pattern_anim_targetType();
                newPatternAnimTargetArray.Add(newPatternAnimTarget);

                newPatternAnimTarget.sampler_name = oldPatternAnimTarget.sampler_name;
                newPatternAnimTarget.hint = oldPatternAnimTarget.hint;
                newPatternAnimTarget.base_value =
                    (float)newTexPatternIdx[(int)oldPatternAnimTarget.base_value];
                // カーブが存在していたらカーブもコピーする
                var oldCurve = oldPatternAnimTarget.Curve as step_curveType;
                if (oldCurve != null)
                {
                    var newTexPatternStream = new G3dStream();
                    var oldTexPatternStream = oldStream[oldCurve.stream_index];

                    // カーブをコピーする。
                    // コピーする際に、カーブの内容を新しい newStream 内のインデックスに置きかえる。
                    step_curveType newCurve;
                    IfTexPatternAnimMerger.CopyTexPatternAnimCurveAndStream(
                        out newCurve,
                        newTexPatternStream,
                        newStream.Count,
                        oldCurve,
                        oldTexPatternStream,
                        newTexPatternIdx);

                    // ストリームを newStream に追加する。
                    newStream.Add(newTexPatternStream);
                    newPatternAnimTarget.Curve = newCurve;
                }
            }

            newTexPatternAnim.tex_pattern_anim_array = new tex_pattern_anim_arrayType();
            newTexPatternAnim.tex_pattern_anim_array.pattern_anim = newPatternAnimTargetArray.ToArray();
            newTexPatternAnim.tex_pattern_anim_array.length = newPatternAnimTargetArray.Count;
            newTexPatternAnim.mat_name = oldTexPatternAnim.mat_name;
        }
    }
}
