﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.IO;
using System.Linq;
using Nintendo.G3dTool.Entities;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // 中間ファイル書き出しユーティリティ
    public static class IfWriteUtility
    {
        public static void WriteIntermediateFile(IntermediateFile file, string xsdBasePath)
        {
            Ensure.Argument.True(!string.IsNullOrEmpty(file.Path));
            WriteIntermediateFile(file, file.Path, xsdBasePath);
        }

        public static void WriteIntermediateFile(IntermediateFile file, string filePath, string xsdBasePath)
        {
            if (G3dPath.IsTextPath(filePath))
            {
                IfTextWriteUtility.Write(file.CreateSerializableData(), filePath, xsdBasePath);
                file.Path = filePath;
            }
            else if (G3dPath.IsBinaryPath(filePath))
            {
                IfBinaryWriteUtility.WriteIntermediateFile(file, filePath, xsdBasePath);
            }
            else
            {
                throw new System.Exception($"Unexpected file extension. {filePath}");
            }
        }

        /// <summary>
        /// 中間ファイルの書き出しを行います。
        /// </summary>
        /// <param name="file">書き出すデータを指定します。</param>
        /// <param name="filePath">出力するファイルパスを指定します。</param>
        /// <param name="xsdBasePath">スキーマチェックを行う XSD フォルダーを指定します。null を指定するとスキーマチェックがスキップされます。</param>
        public static void Write(nw4f_3difType file, string filePath)
        {
            // 書き出しフォーマット自動判別
            Write(file, filePath, null);
        }

        /// <summary>
        /// 中間ファイルの書き出しを行います。
        /// </summary>
        /// <param name="file">書き出すデータを指定します。</param>
        /// <param name="filePath">出力するファイルパスを指定します。</param>
        /// <param name="xsdBasePath">スキーマチェックを行う XSD フォルダーを指定します。null を指定するとスキーマチェックがスキップされます。</param>
        public static void Write(nw4f_3difType file, string filePath, string xsdBasePath)
        {
            // 書き出しフォーマット自動判別
            if (G3dPath.IsTextPath(filePath))
            {
                IfTextWriteUtility.Write(file, filePath, xsdBasePath);
            }
            else
            {
                IfBinaryWriteUtility.Write(file, filePath, xsdBasePath);
            }
        }

        /// <summary>
        /// 中間ファイルの書き出しを行います。
        /// </summary>
        /// <param name="file">書き出すデータを指定します。</param>
        /// <param name="streams">書き出すストリームを指定します。</param>
        /// <param name="filePath">出力するファイルパスを指定します。</param>
        public static void Write(
            nw4f_3difType file, List<G3dStream> streams, string filePath)
        {
            Write(file, streams, filePath, null);
        }

        /// <summary>
        /// 中間ファイルの書き出しを行います。
        /// </summary>
        /// <param name="file">書き出すデータを指定します。</param>
        /// <param name="streams">書き出すストリームを指定します。</param>
        /// <param name="filePath">出力するファイルパスを指定します。</param>
        /// <param name="xsdBasePath">スキーマチェックを行う XSD フォルダーを指定します。null を指定するとスキーマチェックがスキップされます。</param>
        public static void Write(
            nw4f_3difType file, List<G3dStream> streams, string filePath, string xsdBasePath)
        {
            // 書き出しフォーマット自動判別
            if (G3dPath.IsTextPath(filePath))
            {
                file.RootElement.stream_array = G3dStreamUtility.ToStreamArray(streams);
                IfTextWriteUtility.Write(file, filePath, xsdBasePath);
                file.RootElement.stream_array = null;
            }
            else
            {
                IfBinaryWriteUtility.Write(file, streams, filePath, xsdBasePath);
            }
        }

        //---------------------------------------------------------------------
        // ファイルイメージの書き出し
        internal static void WriteFileImage(byte[] fileImage, string filePath)
        {
            using (FileStream fileStream = new FileStream(filePath, FileMode.Create))
            {
                fileStream.Write(fileImage, 0, fileImage.Length);
            }
        }
    }
}
