﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Xml;
using IronPython.Hosting;
using IronPython.Runtime;
using Microsoft.Scripting.Hosting;
using nw.g3d.nw4f_3dif;
using System.Collections.Generic;

namespace nw.g3d.iflib
{
    // 中間ファイルバージョンアップデータ
    public static class IfVersionUpdater
    {
        static private readonly int ReadBytes = 1024;

        // 中間ファイルのバージョンアップ
        // バージョンアップしなかったら null を返す
        public static byte[] Update(byte[] fileImage, string xsdBasePath)
        {
            Nintendo.Foundation.Contracts.Ensure.Argument.NotNull(xsdBasePath);

            // コンテナーフォーマット更新
            byte[] updatedImage1 = UpdateImpl(
                typeof(nw4f_3difType),
                "/nw4f_3dif",
                fileImage,
                System.IO.Path.Combine(xsdBasePath, "3dif"));
            byte[] nextInputImage = updatedImage1;
            if (nextInputImage == null)
            {
                nextInputImage = fileImage;
            }

            // 各拡張子ごとのフォーマット更新
            byte[] updatedImage2 = UpdateImpl(nextInputImage, xsdBasePath);
            if (updatedImage2 != null)
            {
                return updatedImage2;
            }
            else if (updatedImage1 != null)
            {
                return updatedImage1;
            }
            else
            {
                return null;
            }
        }

        private static string ExamineVersion(byte[] fileImage, string targetElemName)
        {
            // 高速化のため、ファイルの先頭のみを読む
            int readCount = ReadBytes;
            if (fileImage.Length < readCount)
            {
                readCount = fileImage.Length;
            }

            using (XmlTextReader rd = new XmlTextReader(
                new MemoryStream(fileImage, 0, readCount)))
            {
                while (rd.Read())
                {
                    if ((rd.NodeType == XmlNodeType.Element) &&
                        (rd.Name == targetElemName))
                    {
                        return rd.GetAttribute("version");
                    }
                }
            }
            return null;
        }

        private static string GetElemName(Type type)
        {
            return type.Name.Substring(0, type.Name.Length - "Type".Length);
        }

        private class IntermediateFileTypeInfo
        {
            public IntermediateFileTypeInfo(Type type, string ext)
            {
                this.Type = type;
                this.Extension = ext;
            }

            public Type Type { get; }
            public string Extension { get; }
        }

        private static byte[] UpdateImpl(byte[] fileImage, string xsdBasePath)
        {
            // スキーマパスが設定されていないとバージョンアップはできない
            if (xsdBasePath == null)
            {
                return null;
            }

            IntermediateFileTypeInfo[] typeAndExtTable = new IntermediateFileTypeInfo[]
                {
                    new IntermediateFileTypeInfo(typeof(modelType), "fmd"),
                    new IntermediateFileTypeInfo(typeof(materialType), "fmt"),
                    new IntermediateFileTypeInfo(typeof(textureType), "ftx"),
                    new IntermediateFileTypeInfo(typeof(skeletal_animType), "fsk"),
                    new IntermediateFileTypeInfo(typeof(material_animType), "fma"),
                    new IntermediateFileTypeInfo(typeof(shader_param_animType), "fsp"),
                    new IntermediateFileTypeInfo(typeof(tex_pattern_animType), "ftp"),
                    new IntermediateFileTypeInfo(typeof(bone_visibility_animType), "fvb"),
                    new IntermediateFileTypeInfo(typeof(mat_visibility_animType), "fvm"),
                    new IntermediateFileTypeInfo(typeof(shape_animType), "fsh"),
                    new IntermediateFileTypeInfo(typeof(scene_animType), "fsn"),
                    new IntermediateFileTypeInfo(typeof(shader_configType), "fsc"),
                    new IntermediateFileTypeInfo(typeof(shader_definitionType), "fsd"),
                    new IntermediateFileTypeInfo(typeof(shader_variationType), "fsv"),
                };

            // 高速化のため、ファイルの先頭のみを読み、ファイル種を判定
            int readCount = ReadBytes;
            if (fileImage.Length < readCount)
            {
                readCount = fileImage.Length;
            }
            using (XmlTextReader rd = new XmlTextReader(
                new MemoryStream(fileImage, 0, readCount)))
            {
                while (rd.Read())
                {
                    if (rd.NodeType != XmlNodeType.Element)
                    {
                        continue;
                    }

                    foreach (var item in typeAndExtTable)
                    {
                        string rootElemName = GetElemName(item.Type);
                        if (rd.Name == rootElemName)
                        {
                            return UpdateImpl(
                                item.Type,
                                $"/nw4f_3dif/{rootElemName}",
                                fileImage,
                                System.IO.Path.Combine(xsdBasePath, item.Extension));
                        }
                    }
                }
            }

            // 各拡張子のルートアイテムが見つからなかった
            return null;
        }

        private static byte[] UpdateImpl(Type type, string rootElemXPath, byte[] fileImage, string xsdBasePath)
        {
            string rootItemElemName = GetElemName(type);
            string version = ExamineVersion(fileImage, rootItemElemName);
            if (version == null)
            {
                // バージョン管理が拡張子ごとに分離する 3.7.0 より古い中間ファイル
                return null;
            }

            string updaterFolderPath =
                IfReadUtility.GetUpdaterFolderPath(xsdBasePath, version);
            string[] updaters = IfVersionUpdater.GetUpdaters(updaterFolderPath);

            // 最初のアップデータが無ければ、バージョンアップ無し
            if (updaters.Length == 0) { return null; }

            // アップデートファイルのテンポラリ保存初期化
            InitializeUpdateTemp();

            // ドキュメントの生成
            XmlDocument document = new XmlDocument();
            XmlReaderSettings readerSettings = new XmlReaderSettings();
            readerSettings.ValidationType = ValidationType.Schema;

            // nw4f_3dif と各拡張子ごとの xsd のバージョンの組み合わせを管理するのが大変なので、
            // 古いバージョンのスキーマチェックは一旦省く
            //readerSettings.Schemas.Add(IfTextReadUtility.GetXsd(fmdXsdPath, "nw4f_3dif", version));

            using (XmlReader reader = XmlReader.Create(
                new MemoryStream(fileImage), readerSettings))
            {
                /// TODO: エラー行数表示方法不明
                document.Load(reader);
            }
            // オリジナルもテンポラリに出力する
            SaveUpdateTemp(document);

            // ドキュメントの更新
            ScriptEngine engine = IfScriptingUtility.CreateEngine();
            UpdateDocument(engine, document, updaters, version, rootElemXPath, xsdBasePath);

            // ドキュメントをバイト列に戻す
            XmlWriterSettings writerSettings = new XmlWriterSettings();
            writerSettings.Indent = true;
            writerSettings.IndentChars = "\t";
            writerSettings.NewLineOnAttributes = true;
            MemoryStream memoryStream = new MemoryStream();
            using (XmlWriter xmlWriter = XmlWriter.Create(memoryStream, writerSettings))
            {
                document.Save(xmlWriter);
            }

            return memoryStream.ToArray();
        }

        // ドキュメントの更新
        private static void UpdateDocument(
            ScriptEngine engine, XmlDocument document,
            string[] updaters, string version, string versionElemXPath, string xsdBasePath)
        {
            // アップデート
            foreach (string updater in updaters)
            {
                IfVersionUpdater.Update(engine, updater, document);
            }

            // 再帰的なアップデート
            string nextVersion = document.SelectSingleNode($"{versionElemXPath}/@version").Value;

            // バージョンが変わらなければ終了（最新バージョンに対するバージョンアップ対応）
            if (version == nextVersion) { return; }

            string nextUpdaterFolderPath =
                IfReadUtility.GetUpdaterFolderPath(xsdBasePath, nextVersion);
            string[] nextUpdaters = IfVersionUpdater.GetUpdaters(nextUpdaterFolderPath);

            // アップデータが無ければ終了
            if (nextUpdaters.Length == 0) { return; }

            // 次のバージョンアップへ
            UpdateDocument(engine, document, nextUpdaters, nextVersion, versionElemXPath, xsdBasePath);
        }

        // アップデート
        private static void Update(
            ScriptEngine engine, string updaterPath, XmlDocument document)
        {
            ScriptSource source = engine.CreateScriptSourceFromFile(
                updaterPath, Encoding.UTF8);
            ScriptScope scope = engine.CreateScope();
            source.Execute(scope);

            object updateVariable;
            scope.TryGetVariable("Update", out updateVariable);
            PythonFunction updateFunc = updateVariable as PythonFunction;
            Nintendo.Foundation.Contracts.Assertion.Operation.True(updateFunc != null);

            scope.SetVariable("update_xml_document", document);
            engine.Execute("Update(update_xml_document)", scope);

            // アップデートファイルのテンポラリ保存
            SaveUpdateTemp(document);
        }

        // アップデータの取得
        private static string[] GetUpdaters(string updaterFolderPath)
        {
            // ディレクトリが無ければアップデータ無し
            if (!Directory.Exists(updaterFolderPath)) { return new string[0]; }

            string[] updaters = Directory.GetFiles(
                updaterFolderPath, "*.py", SearchOption.TopDirectoryOnly);

            // 適用順を選択できるようにアップデータをソートする
            Array.Sort<string>(updaters);
            return updaters;
        }

        //---------------------------------------------------------------------
        // テンポラリファイルの初期化
        private static void InitializeUpdateTemp()
        {
            if (tempPath != null) { return; }
            lock (tempCounterLock)
            {
                if (tempCounter == -1) { return; }

                // テンポラリ環境変数の取得
                tempPath = Environment.GetEnvironmentVariable("NW4F_3DIF_UPDATE_TEMP");
                if ((tempPath == null) || (!Directory.Exists(tempPath)))
                {
                    tempPath = null;
                    tempCounter = -1;
                    return;
                }
                foreach (string filePath in Directory.EnumerateFiles(tempPath))
                {
                    File.Delete(filePath);
                }
            }
        }

        // アップデートファイルのテンポラリ保存
        private static void SaveUpdateTemp(XmlDocument document)
        {
            if (tempPath == null) { return; }

            // 出力パスの作成
            XmlElement rootElement = document.SelectSingleNode("/nw4f_3dif") as XmlElement;
            string version = rootElement.GetAttribute("version").Replace('.', '_');
            string elementName = rootElement.LastChild.Name;
            string extension = G3dPath.GetTextExtensionByElement(elementName);

            string filePath;
            lock (tempCounterLock)
            {
                filePath = string.Format("{0}/Update{1:D2}-{2}{3}",
                    tempPath, tempCounter, version, extension);
                tempCounter++;
            }

            // テンポラリファイルの書き出し
            XmlWriterSettings writerSettings = new XmlWriterSettings();
            writerSettings.Indent = true;
            writerSettings.IndentChars = "\t";
            writerSettings.NewLineOnAttributes = true;
            using (XmlWriter xmlWriter = XmlWriter.Create(filePath, writerSettings))
            {
                document.Save(xmlWriter);
            }
        }

        private static string tempPath;
        private static int tempCounter;
        private static object tempCounterLock = new object();
    }
}
