﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Xml;
using Nintendo.G3dTool.Entities;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // 中間ファイル読み込みユーティリティ
    public static class IfReadUtility
    {
        public static IntermediateFile ReadIntermediateFile(string filePath, string xsdBasePath)
        {
            bool updated;
            return ReadIntermediateFile(filePath, xsdBasePath, out updated);
        }

        /// <summary>
        /// 中間ファイルの読み込みを行います。
        /// </summary>
        /// <param name="filePath">読み込む中間ファイルのパスを指定します。</param>
        /// <param name="xsdBasePath">スキーマチェックを行う XSD フォルダーを指定します。null を指定するとスキーマチェックをスキップします。</param>
        /// <param name="updated">読み込み時に中間ファイルバージョンが更新されたかどうかを格納します。</param>
        /// <returns>読み込まれたファイルを返します。</returns>
        public static IntermediateFile ReadIntermediateFile(string filePath, string xsdBasePath, out bool updated)
        {
            // 読み込みフォーマット自動判別
            if (G3dPath.IsTextPath(filePath))
            {
                var fileData = IfTextReadUtility.Read(filePath, xsdBasePath, out updated);
                return new IntermediateFile(fileData) { Path = filePath };
            }
            else if (G3dPath.IsBinaryPath(filePath))
            {
                return IfBinaryReadUtility.ReadIntermediateFile(filePath, xsdBasePath, out updated);
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(false);
                updated = false;
                return null;
            }
        }

        /// <summary>
        /// 中間ファイルの読み込みを行います。
        /// </summary>
        /// <param name="filePath">読み込む中間ファイルのパスを指定します。</param>
        /// <param name="xsdBasePath">スキーマチェックを行う XSD フォルダーを指定します。null を指定するとスキーマチェックをスキップします。</param>
        /// <returns>読み込まれたファイルを返します。</returns>
        public static nw4f_3difType Read(string filePath, string xsdBasePath)
        {
            bool updated;
            return IfReadUtility.Read(filePath, xsdBasePath, out updated);
        }

        /// <summary>
        /// 中間ファイルの読み込みを行います。
        /// </summary>
        /// <param name="filePath">読み込む中間ファイルのパスを指定します。</param>
        /// <param name="xsdBasePath">スキーマチェックを行う XSD フォルダーを指定します。null を指定するとスキーマチェックをスキップします。</param>
        /// <param name="updated">読み込み時に中間ファイルバージョンが更新されたかどうかを格納します。</param>
        /// <returns>読み込まれたファイルを返します。</returns>
        public static nw4f_3difType Read(
            string filePath, string xsdBasePath, out bool updated)
        {
            // 読み込みフォーマット自動判別
            if (G3dPath.IsTextPath(filePath))
            {
                return IfTextReadUtility.Read(filePath, xsdBasePath, out updated);
            }
            else if (G3dPath.IsBinaryPath(filePath))
            {
                return IfBinaryReadUtility.Read(filePath, xsdBasePath, out updated);
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(false);
                updated = false;
                return null;
            }
        }

        /// <summary>
        /// 中間ファイルの読み込みを行います。
        /// </summary>
        /// <param name="streams">読み込んだファイルのストリームを格納します。</param>
        /// <param name="filePath">読み込む中間ファイルのパスを指定します。</param>
        /// <param name="xsdBasePath">スキーマチェックを行う XSD フォルダーを指定します。null を指定するとスキーマチェックをスキップします。</param>
        /// <returns>読み込まれたファイルを返します。</returns>
        public static nw4f_3difType Read(List<G3dStream> streams,
            string filePath, string xsdBasePath)
        {
            bool updated;
            return IfReadUtility.Read(streams, filePath, xsdBasePath, out updated);
        }

        /// <summary>
        /// 中間ファイルの読み込みを行います。
        /// </summary>
        /// <param name="streams">読み込んだファイルのストリームを格納します。</param>
        /// <param name="filePath">読み込む中間ファイルのパスを指定します。</param>
        /// <param name="xsdBasePath">スキーマチェックを行う XSD フォルダーを指定します。null を指定するとスキーマチェックをスキップします。</param>
        /// <param name="updated">読み込み時に中間ファイルバージョンが更新されたかどうかを格納します。</param>
        /// <returns>読み込まれたファイルを返します。</returns>
        public static nw4f_3difType Read(List<G3dStream> streams,
            string filePath, string xsdBasePath, out bool updated)
        {
            // 読み込みフォーマット自動判別
            if (G3dPath.IsTextPath(filePath))
            {
                nw4f_3difType nw4f_3dif = IfTextReadUtility.Read(
                    filePath, xsdBasePath, out updated);

                // ストリーム配列の取得、パースするので重い
                if (G3dStreamUtility.HasStreamArray(nw4f_3dif))
                {
                    G3dStreamUtility.ToStreams(streams, nw4f_3dif.RootElement.stream_array);
                    nw4f_3dif.RootElement.stream_array = null;
                }

                return nw4f_3dif;
            }
            else if (G3dPath.IsBinaryPath(filePath))
            {
                return IfBinaryReadUtility.Read(
                    streams, filePath, xsdBasePath, out updated);
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(false);
                updated = false;
                return null;
            }
        }

        //---------------------------------------------------------------------
        // バージョンの取得
        public static string GetVersion(byte[] fileImage)
        {
            // 高速化のため、ファイルの先頭のみを読む
            int readCount = 256;
            if (fileImage.Length < readCount) { readCount = fileImage.Length; }

            using (XmlTextReader rd = new XmlTextReader(
                new MemoryStream(fileImage, 0, readCount)))
            {
                while (rd.Read())
                {
                    if ((rd.NodeType == XmlNodeType.Element) &&
                        (rd.Name == "nw4f_3dif"))
                    {
                        return rd.GetAttribute("version");
                    }
                }
            }
            return null;
        }

        //---------------------------------------------------------------------
        // スキーマフォルダパスの取得
        public static string GetXsdFolderPath(string xsdBasePath, string version)
        {
            return System.IO.Path.Combine(xsdBasePath, string.Format("ver_{0}/", version.Replace('.', '_')));
        }

        // スキーマパスの取得
        public static string GetXsdPath(string xsdBasePath, string version)
        {
            return GetXsdPath(System.IO.Path.Combine(xsdBasePath, "3dif"), "nw4f_3dif", version);
        }

        internal static string GetXsdPath(string xsdBasePath, string xsdFileName, string version)
        {
            return GetXsdFolderPath(xsdBasePath, version) + $"{xsdFileName}.xsd";
        }

        // アップデータフォルダパスの取得
        public static string GetUpdaterFolderPath(string xsdBasePath, string version)
        {
            return GetXsdFolderPath(xsdBasePath, version) + "updater/";
        }
    }
}
