﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using Nintendo.G3dTool.Entities;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // バイナリ中間ファイル読み込みユーティリティ
    public static class IfBinaryReadUtility
    {
        public static IntermediateFile ReadIntermediateFile(System.IO.Stream inputStream, string xsdBasePath)
        {
            bool updated;
            return ReadIntermediateFile(inputStream, xsdBasePath, out updated);
        }

        /// <summary>
        /// 中間ファイルの読み込みを行います。
        /// </summary>
        /// <param name="inputStream">読み込む中間ファイルのバイトストリームを指定します。</param>
        /// <param name="xsdBasePath">スキーマチェックを行う XSD フォルダーを指定します。null を指定するとスキーマチェックをスキップします。</param>
        /// <param name="updated">読み込み時に中間ファイルバージョンが更新されたかどうかを格納します。</param>
        /// <returns>読み込まれたファイルを返します。</returns>
        public static IntermediateFile ReadIntermediateFile(System.IO.Stream inputStream, string xsdBasePath, out bool updated)
        {
            byte[] fileImage = StreamToByteArray(inputStream);
            List<G3dStream> streams = new List<G3dStream>();
            var fileData = Read(streams, fileImage, xsdBasePath, out updated);
            var file = new IntermediateFile(fileData);
            file.Streams.Add(streams.Select(x => ConvertToStream(x)));
            file.ResolveInternalReferences();
            return file;
        }

        public static IntermediateFile ReadIntermediateFile(string filePath, string xsdBasePath, out bool updated)
        {
            // 読み込みフォーマット自動判別
            List<G3dStream> streams = new List<G3dStream>();
            var fileData = Read(streams, filePath, xsdBasePath, out updated);
            var file = new IntermediateFile(fileData) { Path = filePath };
            file.Streams.Add(streams.Select(x => ConvertToStream(x)));
            file.ResolveInternalReferences();
            return file;
        }

        public static IntermediateFile ReadIntermediateFileWithoutStreams(string filePath, string xsdBasePath)
        {
            var fileData = ReadTextPartOnly(filePath, xsdBasePath);
            var file = new IntermediateFile(fileData) { Path = filePath };
            return file;
        }

        private static Nintendo.G3dTool.Entities.Stream ConvertToStream(G3dStream sourceStream)
        {
            switch (sourceStream.type)
            {
                case stream_typeType.@byte:
                    {
                        var stream = new StreamByte()
                        {
                            Column = sourceStream.column,
                        };
                        stream.Values.Add(sourceStream.ByteData);
                        return stream;
                    }
                case stream_typeType.@float:
                    {
                        var stream = new StreamFloat()
                        {
                            Column = sourceStream.column,
                        };
                        stream.Values.Add(sourceStream.FloatData);
                        return stream;
                    }
                case stream_typeType.@int:
                    {
                        var stream = new StreamInt()
                        {
                            Column = sourceStream.column,
                        };
                        stream.Values.Add(sourceStream.IntData);
                        return stream;
                    }
                case stream_typeType.@string:
                    {
                        var stream = new StreamString()
                        {
                            Column = sourceStream.column,
                        };
                        stream.Value = sourceStream.StringData;
                        return stream;
                    }
                case stream_typeType.wstring:
                    {
                        var stream = new StreamWstring()
                        {
                            Column = sourceStream.column,
                        };
                        stream.Value = sourceStream.StringData;
                        return stream;
                    }
                default:
                    throw new System.Exception("Unexpected default.");
            }
        }

        // バイナリ中間ファイルの読み込み
        // 重いので非推奨
        public static nw4f_3difType Read(string filePath, string xsdBasePath)
        {
            bool updated;
            return IfBinaryReadUtility.Read(filePath, xsdBasePath, out updated);
        }

        // バイナリ中間ファイルの読み込み
        // 重いので非推奨
        public static nw4f_3difType Read(
            string filePath, string xsdBasePath, out bool updated)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(G3dPath.IsBinaryPath(filePath));

            if (G3dPath.IsStreamBinaryPath(filePath))
            {
                List<G3dStream> streams = new List<G3dStream>();
                nw4f_3difType nw4f_3dif = IfBinaryReadUtility.Read(
                    streams, filePath, xsdBasePath, out updated);
                nw4f_3dif.RootElement.stream_array = G3dStreamUtility.ToStreamArray(streams);
                return nw4f_3dif;
            }
            else
            {
                return IfTextReadUtility.Read(
                    File.ReadAllBytes(filePath), xsdBasePath, out updated);
            }
        }

        // ---------------------------------------------------------------------
        // G3dStream
        // バイナリ中間ファイルの読み込み
        public static nw4f_3difType Read(
            List<G3dStream> stream_array, string filePath, string xsdBasePath)
        {
            bool updated;
            return IfBinaryReadUtility.Read(
                stream_array, filePath, xsdBasePath, out updated);
        }

        // バイナリ中間ファイルの読み込み
        public static nw4f_3difType Read(
            List<G3dStream> stream_array,
            string filePath, string xsdBasePath, out bool updated)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(G3dPath.IsStreamBinaryPath(filePath));
            /// TODO: 拡張子に対する内容チェック

            byte[] fileImage = File.ReadAllBytes(filePath);
            return Read(stream_array, fileImage, xsdBasePath, out updated);
        }

        // メモリ上のバイナリ中間ファイルの読み込み
        public static nw4f_3difType Read(
            List<G3dStream> stream_array, byte[] fileImage, string xsdBasePath)
        {
            bool updated;
            return IfBinaryReadUtility.Read(
                stream_array, fileImage, xsdBasePath, out updated);
        }

        // メモリ上のバイナリ中間ファイルの読み込み
        public static nw4f_3difType Read(
            List<G3dStream> streamArray,
            byte[] fileImage, string xsdBasePath, out bool updated)
        {
            int binaryIndex;
            var nw4F3Dif = Read(fileImage, xsdBasePath, out binaryIndex, out updated);

            // バイナリ部の読み込み
            if (binaryIndex != -1)
            {
                IfBinaryReadUtility.ReadStreamArray(streamArray, fileImage, binaryIndex);
            }

            return nw4F3Dif;
        }

        // ストリーム配列の読み込み
        public static void ReadStreamArray(
            List<G3dStream> streamArray, byte[] binaryImage, int binaryOffset = 0)
        {
            // ストリーム配列チャンクヘッダの読み込み
            int streamCount;
            uint[] streamOffsets;
            uint[] streamSizes;
            int binaryLength = binaryImage.Length - binaryOffset;
            using (BinaryReader rd = new BinaryReader(new MemoryStream(binaryImage, binaryOffset, binaryLength)))
            {
                ulong chunkID = rd.ReadUInt64();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(chunkID == G3dConstant.StreamArrayChunkID);

                streamCount = rd.ReadInt32();
                streamOffsets = new uint[streamCount];
                streamSizes = new uint[streamCount];
                for (int i = 0; i < streamCount; i++)
                {
                    streamOffsets[i] = rd.ReadUInt32() + (uint)binaryOffset;
                    streamSizes[i] = rd.ReadUInt32();
                }
            }

            // stream の読み込み
            G3dStream[] streams = new G3dStream[streamCount];
            Exception exception = null;
            G3dParallel.For(0, streamCount, delegate(int i)
            {
                try
                {
                    using (BinaryReader streamReader = new BinaryReader(
                        new MemoryStream(binaryImage, (int)streamOffsets[i], (int)streamSizes[i])))
                    {
                        G3dStream stream = new G3dStream();
                        stream.Read(streamReader);
                        streams[i] = stream;
                    }
                }
                catch (Exception exp)
                {
                    exception = exp;
                }
            });
            if (!object.ReferenceEquals(exception, null)) { throw exception; }
            streamArray.AddRange(streams);
        }

        // ---------------------------------------------------------------------
        // IntermediateFileStream
        // バイナリ中間ファイルの読み込み
        public static nw4f_3difType Read(
            List<IntermediateFileStream> streamArray, string filePath, string xsdBasePath)
        {
            bool updated;
            return IfBinaryReadUtility.Read(
                streamArray, filePath, xsdBasePath, out updated);
        }

        // バイナリ中間ファイルの読み込み
        public static nw4f_3difType Read(
            List<IntermediateFileStream> streamArray,
            string filePath, string xsdBasePath, out bool updated)
        {
            Nintendo.Foundation.Contracts.Assertion.Operation.True(G3dPath.IsStreamBinaryPath(filePath));
            /// TODO: 拡張子に対する内容チェック

            byte[] fileImage = File.ReadAllBytes(filePath);
            return Read(streamArray, fileImage, xsdBasePath, out updated);
        }

        // メモリ上のバイナリ中間ファイルの読み込み
        public static nw4f_3difType Read(
            List<IntermediateFileStream> streamArray, byte[] fileImage, string xsdBasePath)
        {
            bool updated;
            return IfBinaryReadUtility.Read(
                streamArray, fileImage, xsdBasePath, out updated);
        }

        // メモリ上のバイナリ中間ファイルの読み込み
        public static nw4f_3difType Read(
            List<IntermediateFileStream> streamArray,
            byte[] fileImage, string xsdBasePath, out bool updated)
        {
            int binaryIndex;
            var nw4F3Dif = Read(fileImage, xsdBasePath, out binaryIndex, out updated);

            // バイナリ部の読み込み
            if (binaryIndex != -1)
            {
                IfBinaryReadUtility.ReadStreamArray(streamArray, fileImage, binaryIndex);
            }

            return nw4F3Dif;
        }

        // ストリーム配列の読み込み
        public static void ReadStreamArray(
            List<IntermediateFileStream> streamArray, byte[] binaryImage, int binaryOffset = 0)
        {
            // ストリーム配列チャンクヘッダの読み込み
            int streamCount;
            uint[] streamOffsets;
            uint[] streamSizes;
            int binaryLength = binaryImage.Length - binaryOffset;
            using (var rd = new BinaryReader(new MemoryStream(binaryImage, binaryOffset, binaryLength)))
            {
                ulong chunkID = rd.ReadUInt64();
                Nintendo.Foundation.Contracts.Assertion.Operation.True(chunkID == G3dConstant.StreamArrayChunkID);

                streamCount = rd.ReadInt32();
                streamOffsets = new uint[streamCount];
                streamSizes = new uint[streamCount];
                for (int i = 0; i < streamCount; i++)
                {
                    streamOffsets[i] = rd.ReadUInt32() + (uint)binaryOffset;
                    streamSizes[i] = rd.ReadUInt32();
                }
            }

            // stream の読み込み
            var streams = new IntermediateFileStream[streamCount];
            Exception exception = null;
            G3dParallel.For(0, streamCount, delegate(int i)
            {
                try
                {
                    var stream = new IntermediateFileStream();
                    stream.Read(binaryImage, streamOffsets[i], streamSizes[i]);
                    streams[i] = stream;
                }
                catch (Exception exp)
                {
                    exception = exp;
                }
            });
            if (!object.ReferenceEquals(exception, null)) { throw exception; }
            streamArray.AddRange(streams);
        }

        // メモリ上のバイナリ中間ファイルイメージからテキスト部分を読み込む（ストリームバイナリ部分はファイルイメージのオフセットを返す）
        public static nw4f_3difType Read(byte[] fileImage, string xsdBasePath, out int binaryIndex)
        {
            bool updated;
            return IfBinaryReadUtility.Read(fileImage, xsdBasePath, out binaryIndex, out updated);
        }

        // メモリ上のバイナリ中間ファイルイメージからテキスト部分を読み込む（ストリームバイナリ部分はファイルイメージのオフセットを返す）
        public static nw4f_3difType Read(byte[] fileImage, string xsdBasePath, out int binaryIndex, out bool updated)
        {
            byte[] textImage;
            var nullIndex = Array.IndexOf<byte>(fileImage, 0);
            if (nullIndex == -1)
            {
                textImage = new byte[fileImage.Length];
                Buffer.BlockCopy(fileImage, 0, textImage, 0, fileImage.Length);
            }
            else
            {
                textImage = new byte[nullIndex];
                Buffer.BlockCopy(fileImage, 0, textImage, 0, nullIndex);
            }

            // テキスト部の読み込み
            var nw4F3Dif = IfTextReadUtility.Read(textImage, xsdBasePath, out updated);
            if (nullIndex == -1)
            {
                binaryIndex = -1;
            }
            else
            {
                binaryIndex = ((nullIndex + 1) +
                    (G3dConstant.BinaryAlignment - 1)) &
                    ~(G3dConstant.BinaryAlignment - 1);
            }
            return nw4F3Dif;
        }

        // バイナリ中間ファイルのテキスト/バイナリ分割
        public static void Separate(byte[] fileImage,
            out byte[] textImage, out byte[] binaryImage)
        {
            int nullIndex = Array.IndexOf<byte>(fileImage, 0);
            if (nullIndex == -1)
            {
                textImage = new byte[fileImage.Length];
                Buffer.BlockCopy(fileImage, 0, textImage, 0, fileImage.Length);
                binaryImage = null;
                return;
            }

            textImage = new byte[nullIndex];
            Buffer.BlockCopy(fileImage, 0, textImage, 0, nullIndex);

            // 終端文字 + 1 を RoundUp
            int binaryOffset = ((nullIndex + 1) +
                (G3dConstant.BinaryAlignment - 1)) &
                ~(G3dConstant.BinaryAlignment - 1);
            int binaryLength = fileImage.Length - binaryOffset;
            binaryImage = new byte[binaryLength];
            Buffer.BlockCopy(fileImage, binaryOffset, binaryImage, 0, binaryLength);
        }

        //=====================================================================

        /// <summary>
        /// バイナリー中間ファイルのテキスト部分だけを読みます。
        /// </summary>
        /// <param name="filePath">中間ファイルのパスです。</param>
        /// <param name="xsdBasePath">XSD のルートフォルダーパスです。</param>
        /// <returns>デシリアライズされた中間ファイルを返します。</returns>
        public static nw4f_3difType ReadTextPartOnly(string filePath, string xsdBasePath)
        {
            // 分割読み込みサイズはとりあえず 1024 にしておくが、必要があれば変更、
            // もしくは引数で与えられえるようにする
            int divideReadByteSize = 1024;

            byte[] fileImage = null;
            using (var fileStream = new System.IO.FileStream(filePath, System.IO.FileMode.Open, System.IO.FileAccess.Read))
            {
                int fileSize = (int)fileStream.Length;
                byte[] readBuffer = new byte[divideReadByteSize];
                fileImage = new byte[fileSize];
                int remainSize = fileSize;
                int offset = 0;
                while (remainSize > 0)
                {
                    int readSize = fileStream.Read(readBuffer, 0, Math.Min(divideReadByteSize, remainSize));
                    Buffer.BlockCopy(readBuffer, 0, fileImage, offset, readSize);
                    int nullIndex = Array.IndexOf<byte>(readBuffer, 0);
                    if (nullIndex != -1)
                    {
                        break;
                    }

                    offset += readSize;
                    remainSize -= readSize;
                }
            }

            byte[] textImage;
            {
                int textImageSize = (int)fileImage.Length;
                var nullIndex = Array.IndexOf<byte>(fileImage, 0);
                if (nullIndex != -1)
                {
                    textImageSize = nullIndex;
                }

                textImage = new byte[textImageSize];
                Buffer.BlockCopy(fileImage, 0, textImage, 0, textImageSize);
            }

            bool isUpdated;
            return IfTextReadUtility.Read(textImage, xsdBasePath, out isUpdated);
        }

        internal static byte[] StreamToByteArray(System.IO.Stream inputStream)
        {
            // バイト列に変換
            {
                var memoryStream = inputStream as MemoryStream;
                if (memoryStream != null)
                {
                    return memoryStream.ToArray();
                }
                else
                {
                    using (memoryStream = new MemoryStream())
                    {
                        inputStream.CopyTo(memoryStream);
                        return memoryStream.ToArray();
                    }
                }
            }
        }
    }
}
