﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Diagnostics;
using nw.g3d.nw4f_3dif;
using System.Text;

namespace nw.g3d.iflib
{
    // 中間ファイルデータフォーマッタ
    public static class IfDataFormatter
    {
        // データ列をフォーマットします
        public static void Format(nw4f_3difType nw4f_3dif)
        {
            Nintendo.Foundation.Contracts.Ensure.Argument.NotNull(nw4f_3dif.RootElement);

            // ユーザーデータのデータ列フォーマット
            G3dDataFormatter.FormatUserDataArrayData(nw4f_3dif.RootElement.user_data_array, string.Empty);

            // ストリームのデータ列フォーマット
            G3dDataFormatter.FormatStreamArrayData(nw4f_3dif.RootElement.stream_array);

            // モデル
            modelType model = nw4f_3dif.RootElement as modelType;
            if (model != null) { IfDataFormatter.FormatModelData(model); }

            // マテリアル
            materialType material = nw4f_3dif.RootElement as materialType;
            if (material != null)
            {
                IfDataFormatter.FormatMaterialData(material);
            }

            // コンバイナーシェーダー
            combiner_shaderType combiner = nw4f_3dif.RootElement as combiner_shaderType;
            if (combiner != null)
            {
                IfDataFormatter.FormatCombinerShaderData(combiner);
            }
        }

        //=====================================================================
        // コンバイナーシェーダーのデータ列をフォーマットします
        private static void FormatCombinerShaderData(combiner_shaderType combiner)
        {
            // マテリアルのデータ列フォーマット
            if (combiner.material != null)
            {
                FormatMaterialData(combiner.material);
            }
        }

        #region model
        //=====================================================================
        // モデルのデータ列をフォーマットします
        private static void FormatModelData(modelType model)
        {
            // マテリアルのデータ列フォーマット
            if (model.material_array != null)
            {
                foreach (materialType material in model.material_array.Enumerate())
                {
                    FormatMaterialData(material);
                }
            }

            // ボーンのデータ列フォーマット
            foreach (boneType bone in model.skeleton.bone_array.Enumerate())
            {
                FormatBoneData(bone);
            }
        }

        //=====================================================================
        // マテリアルのデータ列をフォーマットします
        public static void FormatMaterialData(materialType material)
        {
            shader_assignType shader_assign = material.shader_assign;
            if (shader_assign == null) { return; }

            render_info_arrayType render_info_array = shader_assign.render_info_array;
            if (render_info_array != null)
            {
                foreach (render_infoType render_info in render_info_array.Enumerate())
                {
                    FormatRenderInfoData(render_info);
                }
            }

            shader_param_arrayType shader_param_array =
                shader_assign.shader_param_array;
            if (shader_param_array == null) { return; }

            /// TODO: <shader_param> が count を持っていた名残りを無くす
            foreach (shader_paramType shader_param in shader_param_array.shader_param)
            {
                // shader_param_typeType の列挙順に依存
                if (shader_param.type <= shader_param_typeType.bool4)
                {
                    bool[] vals = G3dDataParser.ParseInt2BoolArray(shader_param.Value);
                    FormatShaderParamData(shader_param, vals);
                }
                else if (shader_param.type <= shader_param_typeType.int4)
                {
                    int[] vals = G3dDataParser.ParseIntArray(shader_param.Value);
                    FormatShaderParamData(shader_param, vals);
                }
                else if (shader_param.type <= shader_param_typeType.uint4)
                {
                    uint[] vals = G3dDataParser.ParseUIntArray(shader_param.Value);
                    FormatShaderParamData(shader_param, vals);
                }
                else if (shader_param.type <= shader_param_typeType.texsrt)
                {
                    float[] vals = G3dDataParser.ParseFloatArray(shader_param.Value);
                    FormatShaderParamData(shader_param, vals);
                }
            }

            G3dDataFormatter.FormatUserDataArrayData(material.user_data_array, "\t\t");
        }

        // render_info のデータ列をフォーマットします
        public static void FormatRenderInfoData(render_infoType render_info)
        {
            if (render_info.Value == null) { return; }
            string[] ref_names = G3dDataParser.Tokenize(render_info.Value);

            StringBuilder builder = new StringBuilder();
            builder.AppendLine();
            foreach (string ref_name in ref_names)
            {
                builder.AppendFormat("\t\t\t\t\t{0}", ref_name);
                builder.AppendLine();
            }
            builder.Append("\t\t\t\t");
            render_info.Value = builder.ToString();
        }

        //---------------------------------------------------------------------
        // シェーダーパラメーターの bool データ列をフォーマットします
        public static void FormatShaderParamData(
            shader_paramType shader_param, bool[] vals)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True((shader_param != null) &&
                (shader_param.type >= shader_param_typeType.@bool) &&
                (shader_param.type <= shader_param_typeType.bool4));

            shader_param.Value = FormatShaderParamData(
                shader_param.type - shader_param_typeType.@bool,
                vals.Length,
                delegate(StringBuilder builder, int index)
                {
                    return G3dDataFormatter.AppendInt(builder, vals[index] ? 1 : 0);
                });
        }

        // シェーダーパラメーターの int データ列をフォーマットします
        public static void FormatShaderParamData(
            shader_paramType shader_param, int[] vals)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True((shader_param != null) &&
                (shader_param.type >= shader_param_typeType.@int) &&
                (shader_param.type <= shader_param_typeType.int4));

            shader_param.Value = FormatShaderParamData(
                shader_param.type - shader_param_typeType.@int,
                vals.Length,
                delegate(StringBuilder builder, int index)
                {
                    return G3dDataFormatter.AppendInt(builder, vals[index]);
                });
        }

        // シェーダーパラメーターの uint データ列をフォーマットします
        public static void FormatShaderParamData(
            shader_paramType shader_param, uint[] vals)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True((shader_param != null) &&
                (shader_param.type >= shader_param_typeType.@uint) &&
                (shader_param.type <= shader_param_typeType.uint4));

            shader_param.Value = FormatShaderParamData(
                shader_param.type - shader_param_typeType.@uint,
                vals.Length,
                delegate(StringBuilder builder, int index)
                {
                    return G3dDataFormatter.AppendUInt(builder, vals[index]);
                });
        }

        // 値の追加デリゲート
        private delegate int AppendValue(StringBuilder builder, int index);

        // シェーダーパラメーターのデータ列をフォーマットします
        private static string FormatShaderParamData(
            int typeIndex, int count, AppendValue appendValue)
        {
            int column = GetColumn(typeIndex);
            int low = GetLow(typeIndex);

            StringBuilder builder = new StringBuilder();
            builder.AppendLine();
            int mtxCount = column * low;

            for (int i = 0; i < count; i++)
            {
                if ((i % column) == 0) { builder.Append("\t\t\t\t\t"); }

                int strlen = appendValue(builder, i);
                //builder.Append(vals[i] ? "true" : "false");

                if ((i % column) != (column - 1))
                {
                    // 末尾でなければタブ挿入
                    G3dDataFormatter.AppendTab4(builder, strlen);
                }
                else
                {
                    // 末尾なら改行
                    builder.AppendLine();

                    // 行列間の追加改行
                    if ((low > 1) &&
                        (i % mtxCount == (mtxCount - 1)) &&
                        (i != (count - 1)))
                    {
                        builder.AppendLine();
                    }
                }
            }

            builder.Append("\t\t\t\t");
            return builder.ToString();
        }

        // 行数の取得
        private static int GetLow(int typeIndex)
        {
            // 1
            if (typeIndex == 0) { return 1; }
            // 2 3 4 2x2 2x3 2x4 3x2 3x3 3x4 4x2 4x3 4x4
            return (typeIndex - 1) / 3 + 1;
        }

        // 列数の取得
        private static int GetColumn(int typeIndex)
        {
            // 1
            if (typeIndex == 0) { return 1; }
            // 2 3 4 2x2 2x3 2x4 3x2 3x3 3x4 4x2 4x3 4x4
            return (typeIndex - 1) % 3 + 2;
        }

        //---------------------------------------------------------------------
        // シェーダーパラメーターの float データ列をフォーマットします
        public static void FormatShaderParamData(
            shader_paramType shader_param, float[] vals)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True((shader_param != null) &&
                (shader_param.type >= shader_param_typeType.@float) &&
                (shader_param.type <= shader_param_typeType.texsrt));

            // 行列数の初期化
            int typeIndex = shader_param.type - shader_param_typeType.@float;
            int column = GetColumn(typeIndex);
            int low = GetLow(typeIndex);

            // srt2d、srt3d 対応、shader_param_typeType の列挙順に依存
            bool srt2d = false;
            bool texsrt_mode = false;
            if (shader_param.type == shader_param_typeType.srt3d)
            {
                column = 3;
                low = 3;
            }
            else if (shader_param.type >= shader_param_typeType.srt2d)
            {
                // srt2d、texsrt、texsrt_ex
                column = 2;
                low = 3;
                srt2d = true;
                texsrt_mode = shader_param.type >= shader_param_typeType.texsrt;
            }

            StringBuilder builder = new StringBuilder();
            builder.AppendLine();
            int mtxCount = column * low;

            if (texsrt_mode)
            {
                builder.Append("\t\t\t\t\t");
                G3dDataFormatter.AppendFloat(builder, vals[0]);
                builder.AppendLine();
                for (int i = 1; i < vals.Length; i++)
                {
                    vals[i - 1] = vals[i];
                }
                System.Array.Resize(ref vals, vals.Length - 1);
            }

            for (int i = 0; i < vals.Length; i++)
            {
                int layoutIndex = (i % mtxCount);

                if (srt2d)
                {
                    // srt2d は 5 値周期でレイアウトする
                    layoutIndex = (i % (mtxCount - 1));
                    // 4 つ目以降の値はインクリメントしてレイアウトする
                    if (layoutIndex > 2) { layoutIndex++; }
                }

                if ((layoutIndex % column) == 0) { builder.Append("\t\t\t\t\t"); }

                if (srt2d)
                {
                    // 3 つ目の値はタブを挿入した後に、4 つ目の値として改行する
                    if (layoutIndex == 2) { layoutIndex++; }
                }

                int strlen = G3dDataFormatter.AppendFloat(builder, vals[i]);

                if ((layoutIndex % column) != (column - 1))
                {
                    // 末尾でなければタブ挿入
                    G3dDataFormatter.AppendTab4(builder, strlen);
                }
                else
                {
                    // 末尾なら改行
                    builder.AppendLine();

                    // 行列間の追加改行
                    if ((low > 1) &&
                        (layoutIndex == (mtxCount - 1)) &&
                        (i != (vals.Length - 1)))
                    {
                        builder.AppendLine();
                    }
                }
            }

            builder.Append("\t\t\t\t");
            shader_param.Value = builder.ToString();
        }

        //=====================================================================
        // ボーンのデータ列をフォーマットします
        public static void FormatBoneData(boneType bone)
        {
            float[] vals = G3dDataParser.ParseFloatArray(bone.inv_model_matrix);
            if (vals != null) { FormatBoneData(bone, vals); }

            G3dDataFormatter.FormatUserDataArrayData(bone.user_data_array, "\t\t\t");
        }

        // ボーンのデータ列をフォーマットします
        public static void FormatBoneData(boneType bone, float[] vals)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(bone != null);
            Nintendo.Foundation.Contracts.Assertion.Argument.True((vals != null) && (vals.Length == 12));

            StringBuilder builder = new StringBuilder();
            builder.Append('\n');

            int column = 4;
            for (int i = 0; i < vals.Length; i++)
            {
                if ((i % column) == 0)
                {
                    builder.Append("\t\t\t\t");
                }

                int strlen = G3dDataFormatter.AppendFloat(builder, vals[i]);

                if ((i % column) != (column - 1))
                {
                    G3dDataFormatter.AppendTab4(builder, strlen);
                }
                else
                {
                    builder.Append('\n');
                }
            }

            builder.Append("\t\t\t");
            bone.inv_model_matrix = builder.ToString();
        }
        #endregion
    }
}
