﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // モデルの親子合体ボーン圧縮
    public class IfModelBoneUniteChildCompressor : IfModelCompressor
    {
        // コンストラクタ
        public IfModelBoneUniteChildCompressor() :
            base("IfModelBoneUniteChildCompressor_Log") { }

        // プロセス
        public override string Process
        {
            get { return "compress_bone_unite_child"; }
        }

        // カウントの取得
        public override int GetCount()
        {
            return this.Target.skeleton.bone_array.bone.Length;
        }

        // 圧縮
        protected override void Compress()
        {
            // TODO: Unite モデル対応
            // これより下は Unite されたモデルにまだ対応していない。
            if (this.Target.shape_array != null)
            {
                foreach (shapeType shape in this.Target.shape_array.shape)
                {
                    if (shape.mesh_array.mesh.Length > 1)
                    {
                        IfStrings.Throw("IfModelOptimizer_Error_UnitedModelUnsupported");
                    }
                }
            }

            modelType model = this.Target;
            boneType[] bones = this.Target.skeleton.bone_array.bone;
            int boneCount = bones.Length;
            BoneInfo[] boneInfos = new BoneInfo[boneCount];

            shapeType[] shapes;
            int shapeCount;
            ShapeInfo[] shapeInfos;
            if (this.Target.shape_array != null &&
                this.Target.shape_array.shape != null)
            {
                shapes = model.shape_array.shape;
                shapeCount = shapes.Length;
                shapeInfos = new ShapeInfo[shapeCount];
            }
            else
            {
                shapes = null;
                shapeCount = 0;
                shapeInfos = null;
            }

            // ボーンのソートチェック
            int isSorted = IfModelSortUtility.IsBoneSorted(this.Target);
            if (isSorted != -1)
            {
                IfStrings.Throw("IfModelBoneCompressor_Error_NotSorted",
                    bones[isSorted].name, bones[isSorted + 1].name);
            }

            // ボーン情報の初期化
            for (int i = 0; i < boneCount; i++)
            { boneInfos[i] = new BoneInfo(bones[i], this.Target.skeleton.skeleton_info); }
            BoneInfo.Setup(boneInfos);

            // シェープ情報の初期化
            for (int i = 0; i < shapeCount; i++)
            { shapeInfos[i] = new ShapeInfo(shapes[i]); }
            for (int i = 0; i < shapeCount; i++)
            { shapeInfos[i].Setup(boneInfos); }

            // 統合される部分木を調べる。
            List<BoneInfo> uniteRoot = new List<BoneInfo>();
            foreach (BoneInfo boneInfo in boneInfos)
            {
                if (IsUniteRoot(boneInfo))
                {
                    uniteRoot.Add(boneInfo);
                }
            }
            // 結合される部分木に削除フラグを設定する
            foreach (BoneInfo root in uniteRoot)
            {
                SetRemoveFlag(root);
            }

            // 新しいボーンに出力順のインデックスを設定する
            // UniteBelowBone() 内部で新しいインデックスを参照するので事前に設定しておく必要がある。
            ModelCompressUtility.SetNewIndex(boneInfos[0]);

            // 部分木毎に統合化（ただしノードの削除はあとで）
            if (shapeInfos != null)
            {
                foreach (BoneInfo uniteBone in uniteRoot)
                {
                    UniteBelowBone(uniteBone, shapeInfos);
                }

                // シェープが参照するボーンを変更する
                foreach (ShapeInfo shapeInfo in shapeInfos)
                {
                    shapeType shape = shapeInfo.Shape;
                    shape.shape_info.bone_name = shapeInfo.Bone.Bone.name;
                }
            }

            // ボーンの削除
            ModelCompressUtility.RemoveBones(this.Target, boneInfos);
            // シェイプをソートする
            ModelCompressUtility.SortShape(model);
            // 不要なストリームの削除とソートを行う。
            StreamUtility.SortStream(model, this.Streams);
        }

        /***
         * BoneInfo の RemoveFlag を設定します。
         * boneInfo に指定されたボーンより下の階層の RemoveFlag を true に設定します。
         * 引数として渡された boneInfo 自身は RemoveFlag を設定しません。
         */
        private static void SetRemoveFlag(BoneInfo boneInfo)
        {
            foreach (BoneInfo child in boneInfo.Children)
            {
                child.IsReservedToRemove = true;
                SetRemoveFlag(child);
            }
        }

        /**
         * ボーン以下をい統合します。
         * rootBone で指定されたボーン以下の階層を統合します。
         */
        private void UniteBelowBone(BoneInfo rootBone, ShapeInfo[] shapeInfos)
        {
            List<BoneInfo> children = new List<BoneInfo>();
            ModelCompressUtility.GetBoneDescendants(rootBone, children);

            bool needMatrixForRoot = false;

            // 削除されるボーンを参照しているシェープを列挙する。
            List<ShapeInfo> moveShapes = new List<ShapeInfo>();
            foreach (ShapeInfo shapeInfo in shapeInfos)
            {
                if (children.Contains(shapeInfo.Bone))
                {
                    moveShapes.Add(shapeInfo);
                }
            }

            // リクエストを保持するホルダーを作成する。
            TransformRequestHolder mReqHolder =
                new TransformRequestHolder(this.Target, this.Streams, null);

            // children で列挙されたボーンを参照しているシェープを
            // rootBone の座標系に変換するための要求を作成する。
            // リジッドスキン、スムーススキンのシェープは変換が不要なので、
            // 要求を作成しない。
            foreach (ShapeInfo shapeInfo in moveShapes)
            {
                if (shapeInfo.SkinningMode == ShapeInfo.ShapeSkinningMode.RigidBody)
                {
                    mReqHolder.InsertTransformRequest(rootBone, shapeInfo);
                    needMatrixForRoot = true;
                }
            }

            // 頂点座標を変換する
            mReqHolder.ApplyTransform();

            // シェープが参照するボーンを変更する
            foreach (ShapeInfo shapeInfo in moveShapes)
            {
                shapeInfo.Bone = rootBone;
            }

            // ルートボーンの rigid_body 属性を更新する
            if (needMatrixForRoot)
            {
                rootBone.Bone.rigid_body = true;
            }
        }

        /**
         * 子ノードを再帰的に辿って、部分木の枝ノードの統合条件をチェックします。
         */
        private bool IsUniteTree(BoneInfo boneInfo)
        {
            foreach (BoneInfo child in boneInfo.Children)
            {
                if (!child.Bone.compress_enable)
                {
                    return false;
                }
                if (child.Bone.matrix_index[0] != -1 || child.Bone.matrix_index[1] != -1)
                {
                    return false;
                }
                if (!IsUniteTree(child))
                {
                    return false;
                }
            }
            return true;
        }

        /**
         * ノードが統合される部分木のルートであれば真を返します。
         */
        private bool IsUniteRoot(BoneInfo boneInfo)
        {
            var compressEnable = boneInfo.Bone.compress_enable && boneInfo.Parent != null;
            if (compressEnable)
            {
                return false;
            }
            if (boneInfo.Children.Count == 0 || !IsUniteTree(boneInfo))
            {
                return false;
            }
            return true;
        }
    }
}
