﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // モデルの完全合体ボーン圧縮
    public class IfModelBoneUniteAllCompressor : IfModelCompressor
    {
        // コンストラクタ
        public IfModelBoneUniteAllCompressor() :
            base("IfModelBoneUniteAllCompressor_Log") { }

        // プロセス
        public override string Process
        {
            get { return "compress_bone_unite_all"; }
        }

        // カウントの取得
        public override int GetCount()
        {
            return this.Target.skeleton.bone_array.bone.Length;
        }

        // 圧縮
        protected override void Compress()
        {
            // TODO: Unite モデル対応
            // これより下は Unite されたモデルにまだ対応していない。
            if (this.Target.shape_array != null)
            {
                foreach (shapeType shape in this.Target.shape_array.shape)
                {
                    if (shape.mesh_array.mesh.Length > 1)
                    {
                        IfStrings.Throw("IfModelOptimizer_Error_UnitedModelUnsupported");
                    }
                }
            }

            modelType model = this.Target;
            List<G3dStream> streams = this.Streams;

            boneType[] bones = model.skeleton.bone_array.bone;
            int boneCount = bones.Length;
            BoneInfo[] boneInfos = new BoneInfo[boneCount];

            shapeType[] shapes;
            int shapeCount;
            ShapeInfo[] shapeInfos;
            if (this.Target.shape_array != null &&
                this.Target.shape_array.shape != null)
            {
                shapes = model.shape_array.shape;
                shapeCount = shapes.Length;
                shapeInfos = new ShapeInfo[shapeCount];
            }
            else
            {
                shapes = null;
                shapeCount = 0;
                shapeInfos = null;
            }

            // ボーンのソートチェック
            int isSorted = IfModelSortUtility.IsBoneSorted(this.Target);
            if (isSorted != -1)
            {
                IfStrings.Throw("IfModelBoneCompressor_Error_NotSorted",
                    bones[isSorted].name, bones[isSorted + 1].name);
            }

            // ボーン情報の初期化
            for (int i = 0; i < boneCount; i++)
            { boneInfos[i] = new BoneInfo(bones[i], model.skeleton.skeleton_info); }
            BoneInfo.Setup(boneInfos);

            // シェープ情報の初期化
            for (int i = 0; i < shapeCount; i++)
            { shapeInfos[i] = new ShapeInfo(shapes[i]); }
            for (int i = 0; i < shapeCount; i++)
            { shapeInfos[i].Setup(boneInfos); }

            // ルートノード(boneInfos[0])以外を全て削除する
            for (int i = 1; i < boneCount; i++)
            {
                boneInfos[i].IsReservedToRemove = true;
            }

            // 残されるルートボーンにのみ新しいインデクスを設定する
            BoneInfo rootBone = boneInfos[0];
            rootBone.NewIndex = 0;

            // 最後に残されたルートボーンは変換が単位行列に設定される。
            // そのため、全てのシェープはワールド座標系に変換しなければならない。
            // ModelCompressUtility.GetBoneTransform() では指定された移動先の
            // ボーンの座標系に変換されてしまうので、ルートボーンの変換が単位行列でない場合に
            // toBone 引数へルートボーンを指定してもワールド座標系に変換されない。
            // この問題を回避するためにルートボーンの親にダミーのボーンを作成し、
            // そのダミーボーンへの変換を行う。
            BoneInfo dummyRoot = CreateDummyBone(model.skeleton.skeleton_info);
            rootBone.Parent = dummyRoot;
            dummyRoot.Children.Add(rootBone);

            // 行列パレットを作成するために、パレットとして使用されるボーンを収集する。
            BonePalette bonePltt = ModelCompressUtility.CreateBonePalette(boneInfos, model);

            // リクエストホルダーを作成する
            TransformRequestHolder reqHolder =
                new TransformRequestHolder(model, streams, bonePltt);

            // Shape の移動、およびエンベロープのプリミティブ化によって必要な頂点座標と
            // 法線ベクトルの変換を収集する。
            if (shapeInfos != null)
            {
                foreach (ShapeInfo shapeInfo in shapeInfos)
                {
                    // スキンはルートノードへ移動
                    reqHolder.InsertTransformRequest(dummyRoot, shapeInfo);
                }
            }

            // 頂点座標を変換する
            reqHolder.ApplyTransform();

            // <shape>を変換：
            // ・エンベロープをリジッドボディ化。
            if (shapeInfos != null)
            {
                foreach (ShapeInfo shapeInfo in shapeInfos)
                {
                    ModelCompressUtility.ConvertShape(model, streams, rootBone, shapeInfo);
                }

                // <shape>を上位階層の非圧縮ボーンに移動させる。
                foreach (ShapeInfo shapeInfo in shapeInfos)
                {
                    shapeType shape = shapeInfo.Shape;
                    shapeInfo.Bone = rootBone;
                    shape.shape_info.bone_name = rootBone.Bone.name;
                }
            }

            // キーシェイプを削除する
            ModelCompressUtility.RemoveKeyShape(model);

            // rigid_body フラグとスキニング情報の更新
            if (shapeInfos != null && shapeInfos.Length > 0)
            {
                rootBone.Bone.rigid_body = true;
                // Unite All を実行するとすべてリジッドボディに変換されるので、
                // スキニング関連の情報を初期化しておく。
                rootBone.Bone.matrix_index[0] = -1;
                rootBone.Bone.matrix_index[1] = -1;
                rootBone.Bone.inv_model_matrix = null;
            }

            // ルートボーンの変換を単位行列に設定する
            rootBone.Bone.scale = new float[] { 1, 1, 1 };
            rootBone.Bone.rotate = new float[] { 0, 0, 0, 1 };
            rootBone.Bone.translate = new float[] { 0, 0, 0 };

            // ボーンの削除
            ModelCompressUtility.RemoveBones(model, boneInfos);

            // unite all では出力するボーンが１つだけなのでボーンをソートする必要がない

            // シェイプをソートする
            ModelCompressUtility.SortShape(model);
            // 不要なストリームの削除とソートを行う。
            StreamUtility.SortStream(model, streams);
        }

        private BoneInfo CreateDummyBone(skeletonInfoType skletonInfo)
        {
            boneType dummyBone = new boneType();
            dummyBone.index = -1;
            dummyBone.name = "dummy_root";
            dummyBone.matrix_index = new int[] { -1, -1 };
            dummyBone.scale = new float[] { 1, 1, 1 };
            dummyBone.rotate = new float[] { 0, 0, 0, 1 };
            dummyBone.translate = new float[] { 0, 0, 0 };
            dummyBone.scale_compensate = false;

            BoneInfo dummyBoneInfo = new BoneInfo(dummyBone, skletonInfo);
            return dummyBoneInfo;
        }
    }
}
