﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using nw.g3d.nw4f_3dif;

namespace nw.g3d.iflib
{
    // ボーンビジビリティアニメーションの親子合体ボーン圧縮
    public class IfBoneVisibilityAnimBoneUniteChildCompressor
        : IfBoneVisibilityAnimCompressor
    {
        // コンストラクタ
        public IfBoneVisibilityAnimBoneUniteChildCompressor() :
            base("IfBoneVisibilityAnimBoneUniteChildCompressor_Log") { }

        // プロセス
        public override string Process
        {
            get { return "compress_bone_unite_child"; }
        }

        // 圧縮
        protected override void Compress()
        {
            bone_vis_bone_animType[] bone_anims =
                this.Target.bone_vis_bone_anim_array.bone_vis_bone_anim;
            int boneAnimCount = bone_anims.Length;
            BoneVisAnimInfo[] boneAnimInfos = new BoneVisAnimInfo[boneAnimCount];

            // ボーンアニメーション情報の初期化
            for (int i = 0; i < boneAnimCount; i++)
            { boneAnimInfos[i] = new BoneVisAnimInfo(bone_anims[i]); }
            for (int i = 0; i < boneAnimCount; i++)
            { boneAnimInfos[i].Setup(boneAnimInfos); }

            // 統合される部分木を調べる。
            List<BoneVisAnimInfo> uniteRoot = new List<BoneVisAnimInfo>();
            foreach (BoneVisAnimInfo boneInfo in boneAnimInfos)
            {
                if (IsUniteRoot(boneInfo))
                {
                    uniteRoot.Add(boneInfo);
                }
            }
            // 結合される部分木に削除フラグを設定する
            foreach (BoneVisAnimInfo root in uniteRoot)
            {
                SetRemoveFlag(root);

                // bone_vis_bone_animType の rigid_body フラグを更新する。
                // rigid_body 属性が false の場合は、部分木にリジッドボディが存在しているか確認し、
                // 存在していれば rigid_body 属性を true に設定する。
                if (!root.BoneAnim.rigid_body)
                {
                    List<BoneVisAnimInfo> uniteTree = new List<BoneVisAnimInfo>();
                    BoneVisibilityAnimCompressUtility.GetBoneDescendants(root, uniteTree);
                    int idx = uniteTree.FindIndex(
                        delegate(BoneVisAnimInfo bone) { return bone.BoneAnim.rigid_body; });
                    if (idx != -1)
                    {
                        root.BoneAnim.rigid_body = true;
                    }
                }
            }

            // アニメーションが付いているボーンが削除されるか確認する
            BoneVisibilityAnimCompressUtility.CheckAnimationRemoved(boneAnimInfos);
            // 新しいアニメーションのインデックスを設定する
            BoneVisibilityAnimCompressUtility.SetNewIndex(boneAnimInfos[0]);
            // アニメーションの削除
            BoneVisibilityAnimCompressUtility.RemoveBoneAnims(this.Target, boneAnimInfos);
            // 不要になったストリームを削除する
            BoneVisibilityAnimCompressUtility.RemoveUselessStream(this.Target, this.Streams);
        }

        /***
         * BoneVisAnimInfo の RemoveFlag を設定します。
         * boneInfo に指定されたボーンより下の階層の RemoveFlag を true に設定します。
         * 引数として渡された boneInfo 自身は RemoveFlag を設定しません。
         */
        private static void SetRemoveFlag(BoneVisAnimInfo boneInfo)
        {
            foreach (BoneVisAnimInfo child in boneInfo.Children)
            {
                child.RemoveFlag = true;
                SetRemoveFlag(child);
            }
        }

        /**
         * 子ノードを再帰的に辿って、部分木の枝ノードの統合条件をチェックします。
         */
        private bool IsUniteTree(BoneVisAnimInfo boneInfo)
        {
            foreach (BoneVisAnimInfo child in boneInfo.Children)
            {
                if (!child.BoneAnim.compress_enable)
                {
                    return false;
                }
                if (child.BoneAnim.matrix_index[0] != -1 || child.BoneAnim.matrix_index[1] != -1)
                {
                    return false;
                }
                if (!IsUniteTree(child))
                {
                    return false;
                }
            }
            return true;
        }

        /**
         * ノードが統合される部分木のルートであれば真を返します。
         */
        private bool IsUniteRoot(BoneVisAnimInfo boneInfo)
        {
            var compressEnable = boneInfo.BoneAnim.compress_enable && boneInfo.Parent != null;
            if (compressEnable)
            {
                return false;
            }
            if (boneInfo.Children.Count == 0 || !IsUniteTree(boneInfo))
            {
                return false;
            }
            return true;
        }
    }
}
