﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;
using Nintendo.G3dTool.Entities;

namespace nw.g3d.ifcvtr
{
    internal class IfSkeletalAnimationConverter : Converter
    {
        private string xsdBasePath = null;
        private CommandLineOptions options = null;

        /// <summary>
        /// コンストラクタです。
        /// 入出力のパスの設定、パラメータの取得を行います。
        /// </summary>
        /// <param name="sourcePath">入力ファイルのパス</param>
        /// <param name="param">コンバートを行うパラメータ</param>
        /// <param name="inputXsdBasePath">xsd のパス</param>
        internal IfSkeletalAnimationConverter(string sourcePath, CommandLineOptions param, string inputXsdBasePath)
            : base(sourcePath)
        {
            Nintendo.Foundation.Contracts.Ensure.Argument.True(!string.IsNullOrEmpty(inputXsdBasePath));
            this.xsdBasePath = inputXsdBasePath;

            if (!string.IsNullOrEmpty(param.Output))
            {
                if (G3dPath.IsSkeletalAnimPath(param.Output))
                {
                    this.OutputFilePath = param.Output;
                }
                else
                {
                    this.OutputFilePath = Path.ChangeExtension(param.Output,
                        G3dPath.SkeletalAnimBinaryExtension);
                }
            }
            else
            {
                this.OutputFilePath = Path.ChangeExtension(
                    this.SourceFilePath, G3dPath.SkeletalAnimBinaryExtension);
            }

            this.options = param;
        }

        /// <summary>
        /// 変換処理を実行します。
        /// </summary>
        internal override void Convert()
        {
            string outputPath = Path.GetFullPath(this.OutputFilePath);

            // スケルタルアニメーション中間ファイルのロード
            string srcPath = Path.GetFullPath(this.SourceFilePath);
            var file = IfReadUtility.ReadIntermediateFile(srcPath, xsdBasePath);

            SkeletalAnim skeletalAnim = file.GetRootEntity<SkeletalAnim>();

            // Magnify を適用する
            ApplyMagnify(skeletalAnim);

            // 書き出し
            IfWriteUtility.WriteIntermediateFile(file, outputPath, xsdBasePath);
        }

        /// <summary>
        /// コンバートオプションで受け取った magnify をスケルタルアニメーション中間ファイルに適用します。
        /// </summary>
        /// <param name="skeletalAnim">スケルタルアニメーション中間ファイル</param>
        private void ApplyMagnify(SkeletalAnim skeletalAnim)
        {
            foreach (var boneAnim in skeletalAnim.BoneAnims)
            {
                foreach (var boneAnimTarget in boneAnim.BoneAnimTargets)
                {
                    if ((boneAnimTarget.Key == bone_anim_target_targetType.translate_x) ||
                        (boneAnimTarget.Key == bone_anim_target_targetType.translate_y) ||
                        (boneAnimTarget.Key == bone_anim_target_targetType.translate_z))
                    {
                        ApplyMagnifyImpl(boneAnimTarget.Value);
                    }
                }
            }

            // アニメーションの値を変更しているため、量子化精度を無効にする。
            skeletalAnim.SkeletalAnimInfo.QuantizeToleranceTranslate = .0f;
        }

        /// <summary>
        /// ボーンアニメーションに magnify を適用します。
        /// </summary>
        /// <param name="boneAnimTarget">ボーンアニメーションターゲット</param>
        private void ApplyMagnifyImpl(BoneAnimTarget boneAnimTarget)
        {
            // 初期値に適用する
            boneAnimTarget.BaseValue *= this.options.Magnify;

            // カーブの値に適用する
            ICurve curve = boneAnimTarget.Curve;
            if (curve == null)
            {
                return;
            }

            StreamFloat stream = curve.Stream as StreamFloat;
            Nintendo.Foundation.Contracts.Ensure.Operation.NotNull(stream);

            const int FlameIndex = 0;
            for (int streamValueIndex = 0; streamValueIndex < stream.Count; ++streamValueIndex)
            {
                // アニメーションのストリームは、先頭にキーフレームのインデックスが入っているためスキップします。
                if ((streamValueIndex % stream.Column) == FlameIndex)
                {
                    continue;
                }

                stream.Values[streamValueIndex] = stream.Values[streamValueIndex] * this.options.Magnify;
            }

            // アニメーションの値を変更しているため、量子化精度を無効にする。
            curve.KeyType = curve_key_typeType.none;
            curve.Scale = 1.0f;
            curve.Offset = .0f;
        }
    }
}
