﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.Foundation.IO;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Resources;
using System.Text;
using System.Threading.Tasks;
using nw.g3d.iflib.nw3de;
using nw.g3d.toollib.Ext;
using nw.g3d.ifassign.Resources;

namespace nw.g3d.ifassign
{
    public class g3difassignGlobalParams
    {
        public static Func<string, string, string> CustomDescriptionConverterFunc { get; set; }
        public static string DescriptionConverter(string description, string name)
        {
            string value;
            if (CustomDescriptionConverterFunc != null)
            {
                value = CustomDescriptionConverterFunc(description.Replace("-", "_"), name);
                if (!string.IsNullOrEmpty(value))
                {
                    return value;
                }
            }

            value = g3difassignParams.ResourceManager.GetString(description.Replace("-", "_"));
            Nintendo.Foundation.Contracts.Assertion.Operation.True(!string.IsNullOrEmpty(value));
            return value;
        }

        [CommandLineOption('s', "silent", Description = "silent", DescriptionConverterName = "DescriptionConverter")]
        public bool Silent { get; set; }

        [CommandLineOption('j', "jobs", Description = "jobs", DescriptionConverterName = "DescriptionConverter", ValueName = "number")]
        public int Job { get; set; }
    }

    public class g3difassignParams
    {
        static g3difassignParams()
        {
            ResourceManager = new ResourceManager("nw.g3d.ifassign.Resources.CommandLineHelp", typeof(g3difassignParams).Assembly);
        }

        internal static ResourceManager ResourceManager { get; set; }

        public static string DescriptionConverter(string description, string name)
        {
            var value = ResourceManager.GetString(description.Replace("-", "_"));
            Nintendo.Foundation.Contracts.Assertion.Operation.True(!string.IsNullOrEmpty(value));
            return value;
        }

        [CommandLineSubCommand("assign", Description = "assign", DescriptionConverterName = "DescriptionConverter")]
        public AssignSubCommand Assign { get; set; }

        [CommandLineSubCommand("update-model", Description = "update-model", DescriptionConverterName = "DescriptionConverter")]
        public UpdateModelSubCommand UpdateModel { get; set; }

        [CommandLineSubCommand("update-animation", Description = "update-animation", DescriptionConverterName = "DescriptionConverter")]
        public UpdateAnimationSubCommand UpdateAnimation { get; set; }

        [CommandLineSubCommand("assign-search-path", Description = "assign-search-path", DescriptionConverterName = "DescriptionConverter")]
        public AssignSearchPathSubCommand AssignSearchPath { get; set; }

        [CommandLineSubCommand("assign-parent-material", Description = "assign-parent-material", DescriptionConverterName = "DescriptionConverter")]
        public AssignParentMaterialSubCommand AssignParentMaterial { get; set; }

        [CommandLineSubCommand("apply-parent-material", Description = "apply-parent-material", DescriptionConverterName = "DescriptionConverter")]
        public ApplyParentMaterialSubCommand ApplyParentMaterial { get; set; }

        [CommandLineSubCommand("assign-material-reference-behavior", Description = "assign-material-reference-behavior", DescriptionConverterName = "DescriptionConverter")]
        public MaterialReferenceBehaviorAssignSubCommand AssignMaterialReferenceBehavior { get; set; }

        [CommandLineArgFileOption("args-file", Description = "args-file", DescriptionConverterName = "DescriptionConverter")]
        public string ArgsFile { get; set; }

        [CommandLineOption("disable-file-info", Description = "disable-file-info", DescriptionConverterName = "DescriptionConverter")]
        public bool DisableFileInfo { get; set; }

        public class SubCommandBase
        {
            public static Func<string, string, string> CustomDescriptionConverterFunc { get; set; }
            public static string DescriptionConverter(string description, string name)
            {
                string value;
                if (CustomDescriptionConverterFunc != null)
                {
                    value = CustomDescriptionConverterFunc(description.Replace("-", "_"), name);
                    if (!string.IsNullOrEmpty(value))
                    {
                        return value;
                    }
                }

                value = g3difassignParams.ResourceManager.GetString(description.Replace("-", "_"));
                Nintendo.Foundation.Contracts.Assertion.Operation.True(!string.IsNullOrEmpty(value));
                return value;
            }

            [CommandLineOption("disable-file-info", Description = "disable-file-info", DescriptionConverterName = "DescriptionConverter")]
            public bool DisableFileInfo { get; set; }
        }

        public class AssignSubCommand : SubCommandBase
        {
            public static string FilePathConverter(string path, string option)
            {
                Nintendo.Foundation.Contracts.Ensure.Operation.True(
                    System.IO.File.Exists(path),
                    string.Format(StringResource.Error_InputFileNotFound, path));

                return path;
            }

            [CommandLineOption("shader", Description = "shader", DescriptionConverterName = "DescriptionConverter", ValueName = "file-path", IsRequired = true, ConverterName = "FilePathConverter")]
            public string Shader { get; set; }

            [CommandLineOption("shading-model", Description = "shading-model", DescriptionConverterName = "DescriptionConverter", ValueName = "name")]
            public string ShadingModel { get; set; }

            [CommandLineOption("shading-model-index", Description = "shading-model-index", DescriptionConverterName = "DescriptionConverter", ValueName = "index")]
            public int? ShadingModelIndex { get; set; }

            [CommandLineOption("material", Description = "assign_target_material", DescriptionConverterName = "DescriptionConverter", ValueName = "name")]
            public string[] TargetMaterials { get; set; }

            [CommandLineOption('f', "force", Description = "force", DescriptionConverterName = "DescriptionConverter")]
            public bool Force { get; set; }

            [CommandLineOption("keep-existing-values", Description = "keep_existing_values", DescriptionConverterName = "DescriptionConverter")]
            public bool KeepExistingValues { get; set; }

            [CommandLineValue(0, Description = "intermediate-file-path", ValueName = "file-path", DescriptionConverterName = "DescriptionConverter", IsRequired = true)]
            public string Path { get; set; }

            [CommandLineOption('o', "output", Description = "output", DescriptionConverterName = "DescriptionConverter")]
            public string Output { get; set; }
        }

        public class UpdateModelSubCommand : SubCommandBase
        {
            [CommandLineOption("shader-path", Description = "shader-path", DescriptionConverterName = "DescriptionConverter", ValueName = "folder-path", IsRequired = true)]
            public string ShaderPath { get; set; }

            [CommandLineOption('t', "test", Description = "test", DescriptionConverterName = "DescriptionConverter")]
            public bool Test { get; set; }

            [CommandLineValue(0, Description = "model-file-path", ValueName = "file-path", DescriptionConverterName = "DescriptionConverter", IsRequired = true)]
            public string Path { get; set; }

            [CommandLineOption('o', "output", Description = "output", DescriptionConverterName = "DescriptionConverter")]
            public string Output { get; set; }
        }

        public class UpdateAnimationSubCommand : SubCommandBase
        {
            [CommandLineOption("shader", Description = "shader", DescriptionConverterName = "DescriptionConverter", ValueName = "file-path", IsRequired = true)]
            public string Shader { get; set; }

            [CommandLineOption("shading-model", Description = "shading-model", DescriptionConverterName = "DescriptionConverter", ValueName = "name", IsRequired = true)]
            public string ShadingModel { get; set; }

            [CommandLineOption('t', "test", Description = "test", DescriptionConverterName = "DescriptionConverter")]
            public bool Test { get; set; }

            [CommandLineValue(0, Description = "animation-file-path", ValueName = "file-path", DescriptionConverterName = "DescriptionConverter", IsRequired = true)]
            public string Path { get; set; }

            [CommandLineOption('o', "output", Description = "output", DescriptionConverterName = "DescriptionConverter")]
            public string Output { get; set; }
        }

        public class AssignSearchPathSubCommand : SubCommandBase
        {
            [CommandLineOption("search-path", Description = "search-path", DescriptionConverterName = "DescriptionConverter", ValueName = "folder-path")]
            public string[] SearchPaths { get; set; }

            [CommandLineOption("search-paths", Description = "search-paths", DescriptionConverterName = "DescriptionConverter", ValueName = "folder-path")]
            public string[] SearchPathsOld { get; set; }

            [CommandLineOption('r', "recursive", Description = "recursive", DescriptionConverterName = "DescriptionConverter")]
            public bool Recursive { get; set; }

            [CommandLineValue(0, Description = "model-ftp-fma-file-path", ValueName = "file-path", DescriptionConverterName = "DescriptionConverter", IsRequired = true)]
            public string Path { get; set; }

            [CommandLineOption('o', "output", Description = "output", DescriptionConverterName = "DescriptionConverter")]
            public string Output { get; set; }
        }

        public class AssignParentMaterialSubCommand : SubCommandBase
        {
            [CommandLineOption("parent-material-assign", Description = "parent-material-assign", DescriptionConverterName = "DescriptionConverter", ConverterName = "ParentMaterialAssignConverter")]
            public ParentMaterialAssignOption[] ParentMaterialAssigns { get; set; }

            [CommandLineValue(0, Description = "model-file-path", ValueName = "file-path", DescriptionConverterName = "DescriptionConverter", IsRequired = true)]
            public string Path { get; set; }

            [CommandLineOption('o', "output", Description = "output", DescriptionConverterName = "DescriptionConverter")]
            public string Output { get; set; }

            [CommandLineOption("parent-material-path", Description = "parent-material-path", DescriptionConverterName = "DescriptionConverter")]
            public string[] ParentMaterialPaths { get; set; }

            public static object ParentMaterialAssignConverter(string value, string name)
            {
                return CommandLineUtility.SubOptionConverter<ParentMaterialAssignOption>(value);
            }
        }

        public class ApplyParentMaterialSubCommand : SubCommandBase
        {
            [CommandLineOption("shader-path", Description = "shader-path", DescriptionConverterName = "DescriptionConverter", ValueName = "folder-path", IsRequired = true)]
            public string ShaderPath { get; set; }

            [CommandLineValue(0, Description = "model-file-path", ValueName = "file-path", DescriptionConverterName = "DescriptionConverter", IsRequired = true)]
            public string Path { get; set; }

            [CommandLineOption('o', "output", Description = "output", DescriptionConverterName = "DescriptionConverter")]
            public string Output { get; set; }

            [CommandLineOption("parent-material-path", Description = "parent-material-path", DescriptionConverterName = "DescriptionConverter")]
            public string[] ParentMaterialPaths { get; set; }
        }

        public class ParentMaterialAssignOption
        {
            [CommandLineOption("material", Description = "d")]
            public string Material { get; set; }

            [CommandLineOption("parent-material", Description = "d", IsRequired = true)]
            public string ParentMaterial { get; set; }

            [CommandLineOption("parent-model", Description = "d")]
            public string ParentModel { get; set; }
        }

        public class MaterialReferenceBehaviorAssignSubCommand : SubCommandBase
        {
            [CommandLineOption("material-reference-behavior-assign", Description = "material-reference-behavior-assign", DescriptionConverterName = "DescriptionConverter", ConverterName = "MaterialReferenceBehaviorAssignConverter")]
            public MaterialReferenceBehaviorAssignOption[] MaterialReferenceBehaviorAssigns { get; set; }

            [CommandLineOption("material-reference-sampler-behavior-assign", Description = "material-reference-sampler-behavior-assign", DescriptionConverterName = "DescriptionConverter", ConverterName = "MaterialReferenceSamplerBehaviorAssignConverter")]
            public MaterialReferenceSamplerBehaviorAssignOption[] MaterialReferenceSamplerBehaviorAssigns { get; set; }

            [CommandLineValue(0, Description = "model-file-path", ValueName = "file-path", DescriptionConverterName = "DescriptionConverter", IsRequired = true)]
            public string Path { get; set; }

            [CommandLineOption("shader-path", Description = "shader-path", ValueName = "folder-path", DescriptionConverterName = "DescriptionConverter", IsRequired = false)]
            public string ShaderPath { get; set; }

            [CommandLineOption('o', "output", Description = "output", DescriptionConverterName = "DescriptionConverter")]
            public string Output { get; set; }

            public static object MaterialReferenceBehaviorAssignConverter(string value, string name)
            {
                return CommandLineUtility.SubOptionConverter<MaterialReferenceBehaviorAssignOption>(value);
            }
            public static object MaterialReferenceSamplerBehaviorAssignConverter(string value, string name)
            {
                return CommandLineUtility.SubOptionConverter<MaterialReferenceSamplerBehaviorAssignOption>(value);
            }
        }

        public enum ShaderAssignType
        {
            AttribAssign,
            RenderInfo,
            SamplerAssign,
            ShaderOption,
            ShaderParam,
            Sampler
        }

        public class MaterialReferenceSamplerBehaviorAssignOption
        {
            [CommandLineOption("material", Description = "d")]
            public string Material { get; set; }

            [CommandLineOption("sampler", Description = "d")]
            public string Sampler { get; set; }

            [CommandLineOption("sampler-structure-restriction", Description = "d", ConverterName = "ChildSamplerStructureRestrictionStateConverter")]
            public ChildSamplerStructureRestrictionState? SamplerStructureRestriction { get; set; }

            [CommandLineOption("is-sampler-required", Description = "d")]
            public bool? IsSamplerRequiredForChild { get; set; }

            public static object ChildSamplerStructureRestrictionStateConverter(string value, string name)
            {
                ChildSamplerStructureRestrictionState restriction;
                if (Enum.TryParse(value, true, out restriction))
                {
                    return restriction;
                }

                throw new Exception(MaterialReferenceBehaviorAssignOption.InvallidValueMessage(name, value, typeof(ChildSamplerStructureRestrictionState)));
            }
        }

        public class MaterialReferenceBehaviorAssignOption
        {
            [CommandLineOption("material", Description = "d")]
            public string Material { get; set; }

            [CommandLineOption("sampler", Description = "d")]
            public string Sampler { get; set; }

            [CommandLineOption("type", Description = "d", ConverterName = "ShaderAssignTypeConverter")]
            public ShaderAssignType? Type { get; set; }

            [CommandLineOption("id", Description = "d")]
            public string Id { get; set; }

            [CommandLineOption("child-restriction", Description = "d", ConverterName = "ShaderItemRestrictionStateConverter")]
            public ShaderItemRestrictionState? ChildRestriction { get; set; }

            [CommandLineOption("value", Description = "d", ConverterName = "ShaderItemValueStateConverter")]
            public ShaderItemValueState? Value { get; set; }

            [CommandLineOption("unassigned-param-only", Description = "d", IsRequired = false)]
            public bool IsUnassignedParamOnly { get; set; }

            internal static string InvallidValueMessage(string name, string value, Type enumType)
            {
                return string.Format(Resources.StringResource.Error_InvalidValue, name, value, string.Join(", ", Enum.GetNames(enumType)));
            }

            public static object ShaderAssignTypeConverter(string value, string name)
            {
                ShaderAssignType assignType;
                if (Enum.TryParse(value, true, out assignType))
                {
                    return assignType;
                }

                throw new Exception(InvallidValueMessage(name, value, typeof(ShaderAssignType)));
            }


            public static object ShaderItemValueStateConverter(string value, string name)
            {
                ShaderItemValueState valueState;
                if (Enum.TryParse(value, true, out valueState))
                {
                    return valueState;
                }

                throw new Exception(InvallidValueMessage(name, value, typeof(ShaderItemValueState)));
            }
            public static object ShaderItemRestrictionStateConverter(string value, string name)
            {
                ShaderItemRestrictionState restriction;
                if (Enum.TryParse(value, true, out restriction))
                {
                    return restriction;
                }

                throw new Exception(InvallidValueMessage(name, value, typeof(ShaderItemRestrictionState)));
            }
        }
    }
}
