﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using nw.g3d.toollib;
using nw.g3d.nw4f_3dif;
using nw.g3d.iflib;
using System.IO;
using nw.g3d.ifassign.Resources;

namespace nw.g3d.ifassign
{
    internal class AssignShaderUtility : AssignUtility
    {
        // コンストラクタ
        internal AssignShaderUtility(g3difassignParams.AssignSubCommand programOption)
            : base(programOption)
        {
            this.ShaderPath = programOption.Shader;
            this.ShadingModel = programOption.ShadingModel;
            this.ShadingModelIndex = programOption.ShadingModelIndex;
            this.TargetMaterials = programOption.TargetMaterials;
            this.Force = programOption.Force;
            this.KeepExistingValues = programOption.KeepExistingValues;
            Input = programOption.Path;
            Output = programOption.Output;
        }

        //=====================================================================
        // 処理
        internal override void Process()
        {
            // ShadingModel、ShadingModelIndex ともに未指定の場合はエラー
            if ((this.ShadingModel == null && !this.ShadingModelIndex.HasValue)
                || (this.ShadingModel != null && this.ShadingModelIndex.HasValue))
            {
                throw new Exception(StringResource.Error_NeedOptionShadingModelOrShadingModelIndex);
            }

            // シェーダ定義ファイルの読み込み
            var shaderArchive = Path.GetFileNameWithoutExtension(this.ShaderPath);
            var shaderDefinition = g3difassign.GetShaderDefinitionFromPath(this.ShaderPath, this.ShaderDefinitionStreams);

            // シェーディングモデル
            shading_modelType shadingModel = null;
            var shadingModelArray = shaderDefinition.shading_model_array;
            if (shadingModelArray == null)
            {
                throw new Exception(string.Format(StringResource.Error_AnyShadingModelNotFoundInShaderDefinition, this.ShaderPath));
            }

            if (this.ShadingModel != null)
            {
                shadingModel = shadingModelArray.shading_model.FirstOrDefault(x => x.name == this.ShadingModel);
                if (shadingModel == null)
                {
                    throw new Exception(string.Format(Resources.StringResource.Error_ShadingModelNotFoundInShaderDefinition, this.ShadingModel, this.ShaderPath));
                }
                if (!shadingModel.material_shader)
                {
                    throw new Exception(string.Format(Resources.StringResource.Error_ShadingModelIsNotForMaterial, this.ShadingModel));
                }
            }
            else
            {
                Nintendo.Foundation.Contracts.Assertion.Operation.True(this.ShadingModelIndex.HasValue);
                if (this.ShadingModelIndex < 0 || this.ShadingModelIndex >= shadingModelArray.shading_model.Length)
                {
                    throw new Exception(string.Format(StringResource.Error_ShadingModelIndexIsOutOfRange, this.ShadingModelIndex));
                }
                shadingModel = shadingModelArray.shading_model[this.ShadingModelIndex.Value];
                if (shadingModel == null)
                {
                    throw new Exception(string.Format(StringResource.Error_ShadingModelIndexIsOutOfRange, this.ShadingModelIndex));
                }
                if (!shadingModel.material_shader)
                {
                    throw new Exception(string.Format(Resources.StringResource.Error_ShadingModelIsNotForMaterial, shadingModel.name));
                }
            }

            MessageBuilder.Append(DeprecatedMessage(shadingModel));

            // 引数ファイル毎
            {
                // 対象がシェーダパラメータアニメかどうか
                bool shaderParameterAnimation = false;
                bool materialAnimation = false;
                string sourcePath = Input;

                // パスのチェック
                if (!File.Exists(sourcePath))
                {
                    throw new Exception(string.Format(Resources.StringResource.Error_InputFileNotFound, sourcePath));
                }
                if (G3dPath.IsModelPath(sourcePath))
                {
                }
                else if (G3dPath.IsShaderParamAnimGroupPath(sourcePath))
                {
                    shaderParameterAnimation = true;
                }
                else if (G3dPath.IsMaterialAnimPath(sourcePath))
                {
                    materialAnimation = true;
                }
                else
                {
                    throw new Exception(string.Format(Resources.StringResource.Error_InputFileIsNotModelOrAnimation, sourcePath));
                }

                // 出力パスの設定
                string outputPath = sourcePath;
                if (Output != null)
                {
                    outputPath = Output;

                    // もっと厳密にチェックもできけるけど複雑なケースもあるので簡易チェック
                    string inExt = Path.GetExtension(sourcePath).ToLower();
                    string outExt = Path.GetExtension(outputPath).ToLower();
                    if (outExt.Length == 0 ||
                        !"ab".Contains(outExt[outExt.Length - 1]) ||
                        inExt.Substring(0, inExt.Length - 1) != outExt.Substring(0, outExt.Length - 1))
                    {
                        throw new Exception(string.Format(Resources.StringResource.Error_OutputFileExt, outputPath));
                    }
                }

                // ファイル読み込み
                List<G3dStream> streams = new List<G3dStream>();
                nw4f_3difType nw4f_3dif = IfReadUtility.Read(streams, sourcePath, g3difassign.XsdBasePath);

                // シェーダ割り当て
                if (shaderParameterAnimation)
                {
                    IfAnimationAssignUtility.ApplyOriginalAnimation((shader_param_animType)nw4f_3dif.Item, shadingModel, streams, this.Force);
                }
                else if (materialAnimation)
                {
                    IfAnimationAssignUtility.ApplyOriginalAnimation((material_animType)nw4f_3dif.Item, shadingModel, streams, this.Force);
                }
                else
                {
                    // モデルへのシェーダーアサイン
                    if (this.TargetMaterials != null)
                    {
                        IfShaderAssignUtility.AssignShaderAssign(
                            (modelType)nw4f_3dif.Item, this.TargetMaterials, shaderArchive, shadingModel, this.Force, this.KeepExistingValues);
                    }
                    else
                    {
                        IfShaderAssignUtility.AssignShaderAssign(
                            (modelType)nw4f_3dif.Item, shaderArchive, shadingModel, this.Force, this.KeepExistingValues);
                    }
                }

                // 中間ファイル出力
                if (!this.DisableFileInfo) { IfFileLogUtility.SetModify(nw4f_3dif); }
                IfWriteUtility.Write(nw4f_3dif, streams, outputPath);
            }
        }

        private readonly string[] TargetMaterials;
        private readonly string ShaderPath;
        private readonly string ShadingModel;
        private readonly bool Force;
        private readonly bool KeepExistingValues;
        private readonly string Input;
        private readonly string Output;

        private List<G3dStream> ShaderDefinitionStreams = new List<G3dStream>();
        private int? ShadingModelIndex;
    }
}
