﻿using _3dExcelToShaderAnnotation.Properties;
using Nintendo.ToolFoundation.CommandLine;
using ShaderAssistExAddons.Modules.ParamCode;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;

namespace _3dExcelToShaderAnnotation
{
    public class Context
    {
        private bool isSilent = false;

        /// <summary>
        /// 処理を実行します。
        /// </summary>
        /// <param name="args">引数です。</param>
        /// <returns>結果を返します。</returns>
        public void Execute(params string[] args)
        {
            List<string> inputExcelPaths = new List<string>();
            {
                // 引数パース
                ActionCommand rootCommand = ActionCommand.CreateRootCommand(true);

                rootCommand.AddFlagOption('s', "silent", () => this.isSilent = true)
                    .GetBuilder()
                    .SetDescription(_3dExcelToShaderAnnotation.Properties.Resources.HelpSilent);

                rootCommand.AddValue(-1, path => inputExcelPaths.Add(path))
                    .GetBuilder()
                    .Require()
                    .SetDescription(_3dExcelToShaderAnnotation.Properties.Resources.HelpInput)
                    .SetValueName("path");

                CommandLine.ParseArgs(args, rootCommand, new ParseSettings()
                {
                    HelpWriter = Console.WriteLine
                });
            }

            foreach (string filePath in inputExcelPaths)
            {
                string fileFullPath = Path.GetFullPath(filePath);
                string extension = Path.GetExtension(filePath);
                switch (extension)
                {
                case ".xlsx":
                    {
                        this.MakeParamCode(fileFullPath);
                    }
                    break;
                case ".fsdb":
                case ".fsda":
                    {
                        ShaderDefinitionToExcelConverter.CreateExcelsFromShaderDefinition(
                            fileFullPath, this.WriteMessage, this.WriteErrorMessage);
                    }
                    break;
                default:
                    throw new Exception($"Unsupported file extension {extension}");
                }
            }
        }

        public void WriteMessage(string message)
        {
            if (this.isSilent)
            {
                return;
            }

            Console.WriteLine(message);
        }

        public void WriteErrorMessage(string message)
        {
            Console.Error.WriteLine($"Error: {message}");
        }

        private void MakeParamCode(string inputFilePath)
        {
            if (!File.Exists(inputFilePath))
            {
                throw new Exception(string.Format(_3dExcelToShaderAnnotation.Properties.Resources.ErrorNotExists, inputFilePath));
            }

            if (Path.GetExtension(inputFilePath) != ".xlsx")
            {
                throw new Exception(string.Format(_3dExcelToShaderAnnotation.Properties.Resources.ErrorUnkownFile, inputFilePath));
            }

            var makeParamCode = new MakeParamCodeExecutor();
            var executorResult = makeParamCode.Execute(inputFilePath);
            if (!executorResult.IsSuccessful)
            {
                throw new Exception(executorResult.ErrorMessage);
            }

            string expandedFileDir = PathUtility.GetExpandEnvironmentVariable(makeParamCode.AutogenFileDir);
            if (makeParamCode.AutogenFileDir == string.Empty || expandedFileDir == null)
            {
                throw new Exception(_3dExcelToShaderAnnotation.Properties.Resources.ErrorAutoGenNeedFileDir);
            }
            else if (makeParamCode.AutogenFileName == string.Empty)
            {
                throw new Exception(_3dExcelToShaderAnnotation.Properties.Resources.ErrorAutoGenNeedFileName);
            }

            string outputFolder = expandedFileDir;
            if (!System.IO.Path.IsPathRooted(expandedFileDir))
            {
                outputFolder = System.IO.Path.Combine(
                    System.IO.Path.GetDirectoryName(inputFilePath),
                    expandedFileDir);
            }

            if (!Directory.Exists(outputFolder))
            {
                throw new Exception(string.Format(_3dExcelToShaderAnnotation.Properties.Resources.ErrorNotExistsDir, outputFolder));
            }

            string outputPath = Path.Combine(outputFolder, makeParamCode.AutogenFileName);
            try
            {
                makeParamCode.Write(outputPath);
            }
            catch (Exception e)
            {
                string message = string.Format(_3dExcelToShaderAnnotation.Properties.Resources.ErrorSaveFileError, outputPath);

                Exception ex = e;
                while (ex != null)
                {
#if DEBUG
                    message += ex.ToString();
#else
                    if (!string.IsNullOrEmpty(ex.Message))
                    {
                        message += ex.Message;
                    }
#endif
                    ex = ex.InnerException;
                }

                throw new Exception(message);
            }

            this.WriteMessage(string.Format(_3dExcelToShaderAnnotation.Properties.Resources.Saved, outputPath));
        }
    }

    public class Program
    {
        private static Context context;
        public static void Main(string[] args)
        {
#if !DEBUG
            try
#endif
            {
                context = new Context();
                context.Execute(args);
            }
#if !DEBUG
            catch (Exception exception)
            {
                context.WriteErrorMessage(exception.Message);
                var innerException = exception.InnerException;
                while (innerException != null)
                {
                    context.WriteErrorMessage(innerException.Message);
                    innerException = innerException.InnerException;
                }

                Environment.Exit(-1);
            }
#endif
        }
    }
}
