﻿using Nintendo.G3dTool.Entities;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace G3dCombinerShaderConverter
{
    public class GlslUtility
    {
        /// <summary>
        /// 指定された glsl ファイルの main() ブロックだけを読みます。
        /// </summary>
        /// <param name="textFilePath"></param>
        /// <returns></returns>
        public static string ReadMainBlockText(string textFilePath)
        {
            string sourceCode;
            using (var reader = new System.IO.StreamReader(textFilePath, System.Text.Encoding.UTF8))
            {
                sourceCode = reader.ReadToEnd();
            }

            int mainFuncIndex = sourceCode.IndexOf("void main()");
            sourceCode = sourceCode.Substring(mainFuncIndex);
            int mainStartIndex = sourceCode.IndexOf('{');
            int mainEndIndex = sourceCode.LastIndexOf('}');
            sourceCode = sourceCode.Substring(mainStartIndex, mainEndIndex - mainStartIndex + 1);
            return sourceCode;
        }

        public static IEnumerable<string> ExtractIncludeFilePaths(string sourceCode)
        {
            sourceCode = sourceCode.Replace("#", " # ");
            sourceCode = sourceCode.Replace("<", " < ");
            sourceCode = sourceCode.Replace(">", " > ");
            sourceCode = sourceCode.Replace("\"", " \" ");
            string[] lines = sourceCode.Split('\n');
            foreach (string line in lines)
            {
                string commentRemoved = RemoveCommentOutFromLine(line);
                string[] tokens = commentRemoved.Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
                for (int tokenIndex = 0; tokenIndex < tokens.Length - 2; ++tokenIndex)
                {
                    if (tokens[tokenIndex] == "#" && tokens[tokenIndex + 1] == "include")
                    {
                        string includeFilePath = tokens[tokenIndex + 3];
                        yield return includeFilePath;
                    }
                }
            }
        }

        public static string RemoveCommentOutFromLine(string line)
        {
            // とりあえず複数行に渡る C スタイルコメントは考慮しない
            int lineEndIndex = line.Length - 1;
            int cppStyleCommentBeginIndex = line.IndexOf("//");
            if (cppStyleCommentBeginIndex >= 0)
            {
                lineEndIndex = cppStyleCommentBeginIndex;
            }

            int cStyleCommentBeginIndex = line.IndexOf("/*");
            if (cStyleCommentBeginIndex >= 0)
            {
                lineEndIndex = cStyleCommentBeginIndex < lineEndIndex ? cStyleCommentBeginIndex : lineEndIndex;
            }

            return line.Substring(0, lineEndIndex + 1);
        }

        public static string EstimateReturnCode(string inputText)
        {
            // TODO: 改行コードの適当な推測を直す
            if (inputText.Contains("\r\n"))
            {
                return "\r\n";
            }
            else if (inputText.Contains("\r"))
            {
                // CR はシェーダーコンバーターが対応していないが一応
                return "\r";
            }

            return "\n";
        }

        public void ExtractIncludeFilePathsByFilePathRecursive(
            List<string> inOutIncludeFileList, string inputSourceCodePath, IEnumerable<string> includeSearchPaths)
        {
            string sourceCode = ReadSourceCode(inputSourceCodePath);
            foreach (string filePath in GlslUtility.ExtractIncludeFilePaths(sourceCode))
            {
                string includeFileSourceCodePath = filePath;
                if (!System.IO.Path.IsPathRooted(includeFileSourceCodePath))
                {
                    List<string> searchPaths = new List<string>(includeSearchPaths);
                    searchPaths.Add(System.IO.Path.GetDirectoryName(inputSourceCodePath));
                    IEnumerable<string> existingPaths = CreateExistingFilePathList(
                        searchPaths,
                        new string[] { includeFileSourceCodePath });
                    if (existingPaths.Count() == 0)
                    {
                        continue;
                    }

                    includeFileSourceCodePath = existingPaths.First();
                }

                inOutIncludeFileList.Add(includeFileSourceCodePath);
                ExtractIncludeFilePathsByFilePathRecursive(inOutIncludeFileList, includeFileSourceCodePath, includeSearchPaths);
            }
        }

        public IEnumerable<string> ExtractIncludeFilePathsByFilePath(string sourceCodePath, IEnumerable<string> includeSearchPaths)
        {
            List<string> fileList = new List<string>();
            ExtractIncludeFilePathsByFilePathRecursive(fileList, sourceCodePath, includeSearchPaths);

            // インクルードガードを無視しているので、同じファイルが複数含まれることがあるので、ここで削除する
            List<string> resultList = new List<string>();
            foreach (var filePath in fileList)
            {
                if (resultList.FirstOrDefault(path => Utility.ArePathEqual(path, filePath)) == null)
                {
                    resultList.Add(filePath);
                }
            }

            return resultList;
        }

        private string ReadSourceCode(string targetGlslPath)
        {
            string sourceCode;
            using (var reader = new System.IO.StreamReader(targetGlslPath, System.Text.Encoding.UTF8))
            {
                sourceCode = reader.ReadToEnd();
            }

            sourceCode = sourceCode.Replace("\r\n", "\n");
            sourceCode = sourceCode.Replace("\\\n", string.Empty);
            return sourceCode;
        }

        /// <summary>
        /// 探索パスと相対ファイルパスを組み合わせに対して、存在するパスを列挙します。
        /// </summary>
        /// <param name="searchFolders">探索パス</param>
        /// <param name="relativeFilePaths">探索パスからの相対ファイルパス</param>
        /// <returns></returns>
        private IEnumerable<string> CreateExistingFilePathList(IEnumerable<string> searchFolders, IEnumerable<string> relativeFilePaths)
        {
            foreach (string relativePath in relativeFilePaths)
            {
                foreach (string includeFolderPath in searchFolders)
                {
                    string path = System.IO.Path.Combine(includeFolderPath, relativePath);
                    if (System.IO.File.Exists(path))
                    {
                        yield return path;
                        break;
                    }
                }
            }
        }
    }
}
