﻿using Nintendo.G3dTool.Entities;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;

namespace G3dCombinerShaderConverter
{
    /// <summary>
    /// コンバイナー関連のユーティリティークラスです。
    /// </summary>
    public class CombinerUtility
    {
        private static readonly string CombinerProcessAnnotation = "combiner_process";
        private static readonly string BeginMark = "BEGIN generated code by";
        private static readonly string EndMark = "END generated code by";

        /// <summary>
        /// ファイル未割り当て用バリエーションの値
        /// </summary>
        private const int CombinerDefaultVariationValue = 0;
        private const int CombinerActiveVariationStartValue = CombinerDefaultVariationValue + 1;

        public static string CombinerEditorPath { get; set; }

        /// <summary>
        /// CombinerEditor を使って fcmb から glsl ソースコードを生成します。
        /// </summary>
        /// <param name="inputFcmbFilePath"></param>
        /// <param name="outputGlslFilePath"></param>
        /// <param name="nodeDefFolderPaths"></param>
        /// <param name="writeMessageAction"></param>
        /// <param name="errorMessageAction"></param>
        public static void GenerateGlslSource(
            string inputFcmbFilePath, string outputGlslFilePath, IEnumerable<string> nodeDefFolderPaths,
            Action<string> writeMessageAction,
            Action<string> errorMessageAction)
        {
            string combinerEditorPath = CombinerEditorPath;
            if (string.IsNullOrEmpty(combinerEditorPath))
            {
                combinerEditorPath = Utility.FindCombinerEditorPath();
            }

            StringBuilder shaderEnvPathArgs = new StringBuilder();
            foreach (string path in nodeDefFolderPaths)
            {
                shaderEnvPathArgs.Append($"{path};");
            }

            string args = $"-i {inputFcmbFilePath} -o {outputGlslFilePath} --shader-env-path \"{shaderEnvPathArgs.ToString()}\" --disable-alloc-console";
            Utility.ExecuteProcess(combinerEditorPath, args, false, writeMessageAction, errorMessageAction);
        }

        public static void EmbedCombinerOptionInfoIntoShaderDefition(
            ref ShaderDefinition shaderDef,
            IEnumerable<string> presetFilePaths)
        {
            // コンバイナーオプションの抽出
            IEnumerable<CombinerOptionInfo> combinerProcessOptionInfos = CreateOptionInfo(shaderDef);

            List<ShaderSrc> combinerTargetSources = new List<ShaderSrc>();
            foreach (var source in shaderDef.ShaderSrcs)
            {
                if (source.Stream.Value.Contains(CombinerProcessAnnotation))
                {
                    combinerTargetSources.Add(source);
                }
            }

            // choice 値用の #define プリプロセッサー埋め込み
            foreach (var shadingModel in shaderDef.ShadingModels)
            {
                List<ShaderSrc> shaderSources = new List<ShaderSrc>();
                shaderSources.Add(shadingModel.FragmentStage.ShaderSrc);
                shaderSources.Add(shadingModel.VertexStage.ShaderSrc);
                foreach (var shaderSource in shaderSources.Distinct())
                {
                    string sourceCode = shaderSource.Stream.Value;
                    if (!sourceCode.Contains(CombinerProcessAnnotation))
                    {
                        continue;
                    }

                    string embeddedSourceCode = EmbedOptionValueDefinitions(
                        sourceCode, combinerProcessOptionInfos, presetFilePaths);
                    shaderSource.Stream.Value = embeddedSourceCode;
                }
            }

            // choice の書き換え
            if (combinerProcessOptionInfos.Count() > 0)
            {
                ShaderOptionChoice choice = new ShaderOptionChoice();
                foreach (string presetFilePath in presetFilePaths)
                {
                    string presetFileName = System.IO.Path.GetFileNameWithoutExtension(presetFilePath);
                    choice.Values.Add(new ChoiceValueString(presetFileName, null));
                }

                {
                    // choice に未割り当て扱いの CombinerDefaultVariationValue が含まれていなければ追加する。
                    var defaultVariationValue = CombinerDefaultVariationValue.ToString();
                    if (!choice.Values.Any(x => x.Value == defaultVariationValue))
                    {
                        choice.Values.Insert(0, new ChoiceValueString(defaultVariationValue, null));
                    }
                }

                string defaultChoice = choice.Values.First().Value;

                // ソースコードの choice の書き換え
                foreach (var src in shaderDef.ShaderSrcs)
                {
                    src.Stream.Value = EmbedOptionChoice(
                        src.Stream.Value, combinerProcessOptionInfos, choice, defaultChoice);
                }

                // 定義の書き換え
                foreach (var shadingModel in shaderDef.ShadingModels)
                {
                    if (shadingModel.VertexStage != null
                        && (combinerTargetSources.Contains(shadingModel.VertexStage.ShaderSrc)
                            || ContainsTargetFileInIncludeFiles(shadingModel.VertexStage.ShaderSrc, combinerTargetSources)))
                    {
                        EmbedOptionVar(shadingModel, combinerProcessOptionInfos, choice, defaultChoice, ShaderStage.VertexStage);
                    }
                    if (shadingModel.FragmentStage != null
                        && (combinerTargetSources.Contains(shadingModel.FragmentStage.ShaderSrc)
                            || ContainsTargetFileInIncludeFiles(shadingModel.FragmentStage.ShaderSrc, combinerTargetSources)))
                    {
                        EmbedOptionVar(shadingModel, combinerProcessOptionInfos, choice, defaultChoice, ShaderStage.FragmentStage);
                    }
                    if (shadingModel.GeometryStage != null
                        && (combinerTargetSources.Contains(shadingModel.GeometryStage.ShaderSrc)
                            || ContainsTargetFileInIncludeFiles(shadingModel.GeometryStage.ShaderSrc, combinerTargetSources)))
                    {
                        EmbedOptionVar(shadingModel, combinerProcessOptionInfos, choice, defaultChoice, ShaderStage.GeometryStage);
                    }
                    if (shadingModel.ComputeStage != null
                        && (combinerTargetSources.Contains(shadingModel.ComputeStage.ShaderSrc)
                            || ContainsTargetFileInIncludeFiles(shadingModel.ComputeStage.ShaderSrc, combinerTargetSources)))
                    {
                        EmbedOptionVar(shadingModel, combinerProcessOptionInfos, choice, defaultChoice, ShaderStage.ComputeStage);
                    }
                }
            }
        }

        private static bool ContainsTargetFileInIncludeFiles(ShaderSrc mainSource, IEnumerable<ShaderSrc> targetSources)
        {
            if (targetSources.Contains(mainSource))
            {
                return true;
            }
            foreach (string filePath in GlslUtility.ExtractIncludeFilePaths(mainSource.Stream.Value))
            {
                if (targetSources.Any(x => x.Path == filePath))
                {
                    return true;
                }
            }

            return false;
        }

        private enum ShaderStage
        {
            VertexStage,
            GeometryStage,
            FragmentStage,
            ComputeStage,
        };

        private static void EmbedOptionVar(
            ShadingModel shadingModel,
            IEnumerable<CombinerOptionInfo> combinerProcessOptionInfos,
            ShaderOptionChoice choice,
            string defaultChoice,
            ShaderStage stage)
        {
            foreach (var targetOptionInfo in combinerProcessOptionInfos)
            {
                var optionVar = shadingModel.OptionVars.FirstOrDefault(x => x.Id == targetOptionInfo.OptionId);
                if (optionVar == null)
                {
                    optionVar = new OptionVar()
                    {
                        Id = targetOptionInfo.OptionId,
                        Type = option_var_typeType.@static,
                    };
                    // コンバイナー用オプション変数がなければ作る
                    shadingModel.OptionVars.Add(optionVar);
                }

                switch (stage)
                {
                    case ShaderStage.VertexStage:
                        optionVar.VertexSymbol.Name = targetOptionInfo.OptionMacroName;
                        break;
                    case ShaderStage.GeometryStage:
                        optionVar.GeometrySymbol.Name = targetOptionInfo.OptionMacroName;
                        break;
                    case ShaderStage.FragmentStage:
                        optionVar.FragmentSymbol.Name = targetOptionInfo.OptionMacroName;
                        break;
                    case ShaderStage.ComputeStage:
                        optionVar.ComputeSymbol.Name = targetOptionInfo.OptionMacroName;
                        break;
                    default:
                        throw new Exception("Unexpected default");
                }

                optionVar.Branch = IsDefinedInOptionBlock(targetOptionInfo.OptionId, shadingModel);
                optionVar.Choice.DeepCopyFrom(choice);
                optionVar.Default = defaultChoice;
                optionVar.UiItem = new UiItem() { Value = ui_item_valueType.combiner };
            }
        }

        /// <summary>
        /// シェーダーのソースコードにオプションによる分岐コードを挿入します。
        /// </summary>
        /// <param name="shaderDef">対象のシェーダー定義を渡します。</param>
        /// <param name="presetFilePaths">分岐コードの元となる fcmb ファイルパスを指定します。</param>
        /// <param name="messageAction"></param>
        /// <param name="warningMessageAction"></param>
        /// <param name="errorMessageAction"></param>
        public static void EmbedCombinerSourceIntoShaderDefinition(
            ref ShaderDefinition shaderDef,
            IEnumerable<string> presetFilePaths,
            Action<string> messageAction,
            Action<string> warningMessageAction,
            Action<string> errorMessageAction)
        {
            using (var scopedDirCreator = new ScopedWorkDirectoryCreator(Utility.GetTempWorkFolderPath()))
            {
                List<string> includeSources = new List<string>();
                List<string> includePaths = new List<string>();
                includePaths.Add(scopedDirCreator.WorkDirectoryPath);

                // コンバイナーエディターにソースコードを渡すために一時ファイルに書き出し
                foreach (var src in shaderDef.ShaderSrcs)
                {
                    StreamWstring sourceStream = src.Stream;
                    string sourceCode = sourceStream.Value;
                    includeSources.Add(sourceCode);

                    string tempFilePath = System.IO.Path.Combine(
                        scopedDirCreator.WorkDirectoryPath, src.Path);
                    string tempFolder = System.IO.Path.GetDirectoryName(tempFilePath);
                    System.IO.Directory.CreateDirectory(tempFolder);
                    using (var stream = System.IO.File.OpenWrite(tempFilePath))
                    {
                        byte[] bytes = System.Text.Encoding.UTF8.GetBytes(sourceCode);
                        stream.Write(bytes, 0, bytes.Length);
                    }

                    if (includePaths.FirstOrDefault(x => Utility.ArePathEqual(tempFolder, x)) == null)
                    {
                        includePaths.Add(tempFolder);
                    }
                }

                foreach (var shadingModel in shaderDef.ShadingModels)
                {
                    List<ShaderSrc> shaderSources = new List<ShaderSrc>();
                    shaderSources.Add(shadingModel.FragmentStage.ShaderSrc);
                    shaderSources.Add(shadingModel.VertexStage.ShaderSrc);
                    foreach (var mainSource in shaderSources)
                    {
                        string mainSourceCode = mainSource.Stream.Value;
                        EmbedCombinerInfoWithIncludeFiles(mainSource,
                            shaderDef,
                            presetFilePaths,
                            includeSources,
                            includePaths,
                            messageAction,
                            warningMessageAction,
                            errorMessageAction);
                    }
                }
            }
        }

        private static void EmbedCombinerInfoWithIncludeFiles(
            ShaderSrc source,
            ShaderDefinition shaderDef,
            IEnumerable<string> presetFilePaths,
            IEnumerable<string> includeSources,
            IEnumerable<string> includePaths,
            Action<string> messageAction,
            Action<string> warningMessageAction,
            Action<string> errorMessageAction)
        {
            string sourceCode = source.Stream.Value;
            if (sourceCode.Contains(CombinerProcessAnnotation))
            {
                // とりあえずすべてのソースコードをコンバイナーオプション定義が書いてある候補ソースコードとして入力する
                try
                {
                    EmbedCombinerInfo(
                        ref sourceCode,
                        shaderDef,
                        includeSources,
                        includePaths,
                        presetFilePaths,
                        messageAction,
                        errorMessageAction);
                }
                catch (WarningException exception)
                {
                    warningMessageAction(exception.Message + $": {source.Path}");
                }

                source.Stream.Value = sourceCode;
            }

            foreach (var includeFilePath in GlslUtility.ExtractIncludeFilePaths(sourceCode))
            {
                ShaderSrc includeSource = shaderDef.ShaderSrcs.First(x => x.Path == includeFilePath);
                EmbedCombinerInfoWithIncludeFiles(
                    includeSource,
                    shaderDef,
                    presetFilePaths,
                    includeSources,
                    includePaths,
                    messageAction,
                    warningMessageAction,
                    errorMessageAction);
            }
        }

        /// <summary>
        /// ソースコードにコンバイナー情報から生成されたソースコードを埋め込みます。
        /// </summary>
        /// <param name="embedTargetGlslSourceCode"></param>
        /// <param name="optionDefGlslSourceCodes"></param>
        /// <param name="nodeDefFolderPaths"></param>
        /// <param name="presetFolderPath"></param>
        /// <param name="messageAction"></param>
        /// <param name="errorMessageAction"></param>
        /// <returns></returns>
        public static void EmbedCombinerInfo(
            ref string embedTargetGlslSourceCode,
            ShaderDefinition shaderDef,
            IEnumerable<string> optionDefGlslSourceCodes,
            IEnumerable<string> nodeDefFolderPaths,
            IEnumerable<string> presetFilePaths,
            Action<string> messageAction,
            Action<string> errorMessageAction)
        {
            IoUtility.WriteDebugMessageLine("アノテーションとオプション変数の解析を行います。");
            IEnumerable<CombinerOptionInfo> combinerOptions = CombinerUtility.CreateCombinerInfo(
                embedTargetGlslSourceCode, shaderDef, optionDefGlslSourceCodes, presetFilePaths);
            List<InsertTextInfo> insertTextBlockList = new List<InsertTextInfo>();
            if (combinerOptions.Count() == 0)
            {
                throw new WarningException($"{CombinerUtility.CombinerProcessAnnotation} アノテーションがソースコード内にひとつも見つかりませんでした。コンバイナー処理の埋め込みは行われません。");
            }

            foreach (CombinerOptionInfo combinerOption in combinerOptions)
            {
                List<string> presetFilePathsForChoice = combinerOption.PresetFilePathList;

                if (presetFilePathsForChoice.Count == 0)
                {
                    IoUtility.WriteDebugMessageLine($"オプション ID \"{combinerOption.OptionId}\" に対する choice が見つからなかったか、解析に失敗しました。");
                    continue;
                }

                StringBuilder combinerProcessText = new StringBuilder();
                for (int variationIndex = 0; variationIndex < presetFilePathsForChoice.Count; ++variationIndex)
                {
                    var presetFilePath = presetFilePathsForChoice[variationIndex];
                    if (!System.IO.File.Exists(presetFilePath))
                    {
                        throw new Exception($"\"{presetFilePath}\" が見つかりません。");
                    }

                    string presetName = System.IO.Path.GetFileNameWithoutExtension(presetFilePath);

                    if (variationIndex == 0)
                    {
                        if (combinerOption.IsDefinedInOptionBlock)
                        {
                            combinerProcessText.AppendLine($"if ({combinerOption.OptionMacroName} == ({presetName}))");
                        }
                        else
                        {
                            combinerProcessText.AppendLine($"#if {combinerOption.OptionMacroName} == ({presetName})");
                        }
                    }
                    else
                    {
                        if (combinerOption.IsDefinedInOptionBlock)
                        {
                            combinerProcessText.AppendLine($"else if ({combinerOption.OptionMacroName} == ({presetName}))");
                        }
                        else
                        {
                            combinerProcessText.AppendLine($"#elif {combinerOption.OptionMacroName} == ({presetName})");
                        }
                    }

                    // シェーダーソースコードの生成
                    string outputGlslFilePath = Utility.GetTempFilePath($"{combinerOption.OptionId}-{presetName}.autogen.glsl");
                    using (new ScopedFileCreator(outputGlslFilePath, true))
                    {
                        CombinerUtility.GenerateGlslSource(presetFilePath, outputGlslFilePath, nodeDefFolderPaths,
                            messageAction, errorMessageAction);

                        // 生成されたコードのmain()の中身だけを取得
                        string blockText = GlslUtility.ReadMainBlockText(outputGlslFilePath);
                        combinerProcessText.AppendLine(blockText);
                    }
                }

                if (0 < presetFilePathsForChoice.Count)
                {
                    if (!combinerOption.IsDefinedInOptionBlock)
                    {
                        combinerProcessText.AppendLine($"#endif");
                    }
                    insertTextBlockList.Add(new InsertTextInfo() { Id = combinerOption.OptionId, Text = combinerProcessText.ToString() });
                }
            }

            if (insertTextBlockList.Count == 0)
            {
                return;
            }

            string outputSourceCode = CombinerUtility.InsertCombinerProcessBlock(embedTargetGlslSourceCode, insertTextBlockList);
            embedTargetGlslSourceCode = outputSourceCode;
        }

        public static void ConvertCombinerEnumChoiceToHashValue(ref ShaderDefinition shaderDef)
        {
            // 今のところ、すべてのコンバイナーオプションの choice は一致するため、最初に見つかったひとつを置き換えに使用
            // TODO: オプション毎に choice が一致しなくなったら直す
            Dictionary<string, string> choiceValueConvertTable = new Dictionary<string, string>();

            {
                CRC32 crc = new CRC32();
                foreach (var shadingModel in shaderDef.ShadingModels)
                {
                    foreach (var optionVar in shadingModel.OptionVars)
                    {
                        if (optionVar.UiItem == null || optionVar.UiItem.Value != ui_item_valueType.combiner)
                        {
                            continue;
                        }

                        foreach (var choiceValue in optionVar.Choice.Values)
                        {
                            if (choiceValueConvertTable.ContainsKey(choiceValue.Value))
                            {
                                continue;
                            }

                            if (choiceValue.Value == CombinerDefaultVariationValue.ToString())
                            {
                                choiceValueConvertTable.Add(choiceValue.Value, choiceValue.Value);
                            }
                            else
                            {
                                uint hash = crc.ComputeHashUInt32(choiceValue.Value);
                                choiceValueConvertTable.Add(choiceValue.Value, hash.ToString());
                            }
                        }
                    }
                }
            }

            // 定義の書き換え
            {
                foreach (var shadingModel in shaderDef.ShadingModels)
                {
                    foreach (var optionVar in shadingModel.OptionVars)
                    {
                        if (optionVar.UiItem == null || optionVar.UiItem.Value != ui_item_valueType.combiner)
                        {
                            continue;
                        }

                        foreach (var choiceValue in optionVar.Choice.Values)
                        {
                            string newValue = choiceValueConvertTable[choiceValue.Value];
                            choiceValue.Value = newValue;
                        }
                    }
                }
            }

            // choice 値用の #define プリプロセッサー書き換え
            {
                foreach (var shadingModel in shaderDef.ShadingModels)
                {
                    List<ShaderSrc> shaderSources = new List<ShaderSrc>();
                    shaderSources.Add(shadingModel.FragmentStage.ShaderSrc);
                    shaderSources.Add(shadingModel.VertexStage.ShaderSrc);
                    foreach (var shaderSource in shaderSources.Distinct())
                    {
                        string sourceCode = shaderSource.Stream.Value;
                        if (!sourceCode.Contains(BeginMark))
                        {
                            continue;
                        }

                        foreach (var value in choiceValueConvertTable)
                        {
                            if (value.Key == CombinerDefaultVariationValue.ToString())
                            {
                                continue;
                            }
                            sourceCode = ReplaceChoiceMacroValue(sourceCode, value.Key, value.Value);
                        }

                        shaderSource.Stream.Value = sourceCode;
                    }
                }
            }

            // ソースコードの書き換え
            {
                string oldChoiceAnnotation = $"choice=\"{string.Join(",", choiceValueConvertTable.Select(x => x.Key))}\"";
                string newChoiceAnnotation = $"choice=\"{string.Join(",", choiceValueConvertTable.Select(x => x.Value))}\"";
                foreach (var src in shaderDef.ShaderSrcs)
                {
                    src.Stream.Value = src.Stream.Value.Replace(oldChoiceAnnotation, newChoiceAnnotation);
                }
            }
        }

        private static IEnumerable<CombinerOptionInfo> CreateOptionInfo(ShaderDefinition shaderDef)
        {
            // コンバイナーオプションの抽出
            List<string> sourceCodeList = new List<string>();
            List<ShaderSrc> combinerTargetSources = new List<ShaderSrc>();
            foreach (var source in shaderDef.ShaderSrcs)
            {
                StreamWstring sourceStream = source.Stream;
                string sourceCode = sourceStream.Value;
                sourceCodeList.Add(sourceCode);
                if (sourceCode.Contains(CombinerProcessAnnotation))
                {
                    combinerTargetSources.Add(source);
                }
            }

            List<CombinerOptionInfo> combinerProcessOptionInfos = new List<CombinerOptionInfo>();
            foreach (var sourceCode in sourceCodeList)
            {
                var optionInfos = CreateCombinerInfo(sourceCode, shaderDef, sourceCodeList, new string[] { });
                if (optionInfos.Count() != 0)
                {
                    combinerProcessOptionInfos.AddRange(optionInfos);
                }
            }

            return combinerProcessOptionInfos;
        }

        private static bool IsDefinedInOptionBlock(string optionId, ShaderDefinition shaderDef)
        {
            // すべてのシェーディングモデルでオプションブロックに定義されているかを調べる
            // (シェーディングモデル毎にオプションブロックが変化すると破綻するが、まずなさそうなのでとりあえず無視)
            foreach (var shadingModel in shaderDef.ShadingModels)
            {
                if (IsDefinedInOptionBlock(optionId, shadingModel))
                {
                    return true;
                }
            }

            return false;
        }

        private static bool IsDefinedInOptionBlock(string optionId, ShadingModel shadingModel)
        {
            foreach (var optionBlockVar in shadingModel.BlockVars.Where(x => x.Type == block_var_typeType.option))
            {
                foreach (var optionUniformVar in optionBlockVar.UniformVars)
                {
                    if (optionUniformVar.Id == optionId)
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        private static IEnumerable<CombinerOptionInfo> CreateCombinerInfo(
            string embedTargetGlslSourceCode,
            ShaderDefinition shaderDef,
            IEnumerable<string> optionDefGlslSources,
            IEnumerable<string> presetFilePaths)
        {
            IEnumerable<CombinerOptionInfo> combinerOptionInfos = ExtractCombinerProcessAnnotation(embedTargetGlslSourceCode);
            foreach (var optionInfo in combinerOptionInfos)
            {
                optionInfo.IsDefinedInOptionBlock = IsDefinedInOptionBlock(optionInfo.OptionId, shaderDef);

                foreach (string optionDefGlslSourceCode in optionDefGlslSources)
                {
                    string[] lines = optionDefGlslSourceCode.Split('\n');
                    // オプションのマクロ名とプリセット名を探す
                    foreach (string line in lines)
                    {
                        string macroName = FindMacroName(line);
                        if (string.IsNullOrEmpty(macroName))
                        {
                            continue;
                        }

                        int commentStartIndex = line.IndexOf("//");
                        if (commentStartIndex == -1)
                        {
                            continue;
                        }

                        string comment = line.Substring(commentStartIndex);
                        string id = CombinerUtility.FindAnnotationValue("id", comment);
                        if (optionInfo.OptionId != id)
                        {
                            continue;
                        }

                        IoUtility.WriteDebugMessageLine($"オプション変数 \"{id}\" を検知しました。");
                        optionInfo.OptionMacroName = macroName;

                        string choice = CombinerUtility.FindAnnotationValue("choice", comment);
                        if (!string.IsNullOrEmpty(choice))
                        {
                            string[] commaSplitedList = choice.Split(new char[] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                            foreach (string commaSplited in commaSplitedList)
                            {
                                string[] colonSplitedList = commaSplited.Split(new char[] { ':' }, StringSplitOptions.RemoveEmptyEntries);
                                string presetName = colonSplitedList.First().Trim(new char[] { ' ', '\t' });
                                if (presetName == CombinerDefaultVariationValue.ToString())
                                {
                                    // 何もしない
                                }
                                else
                                {
                                    string presetPath = presetFilePaths.FirstOrDefault(x => System.IO.Path.GetFileNameWithoutExtension(x).ToLower() == presetName.ToLower());
                                    if (presetPath != null)
                                    {
                                        optionInfo.PresetFilePathList.Add(presetPath);
                                        IoUtility.WriteDebugMessageLine($"  \"{id}\" の choice に定義されたプリセット \"{presetName}\" を検知しました。");
                                    }
                                    else
                                    {
                                        throw new Exception($"\"{id}\" の choice に定義されたプリセット \"{presetName}\" に該当するファイルが見つかりません");
                                    }
                                }
                            }
                        }
                    }
                }
            }

            return combinerOptionInfos;
        }



        private static string InsertCombinerProcessBlock(string inputText, IEnumerable<InsertTextInfo> idAndTextBlockList)
        {
            string returnCode = GlslUtility.EstimateReturnCode(inputText);

            StringBuilder outputText = new StringBuilder();
            foreach (string line in inputText.Split(new string[] { returnCode }, StringSplitOptions.None))
            {
                int commentStartIndex = line.IndexOf("//");
                if ((commentStartIndex != -1) && line.Contains("@@") && line.Contains(CombinerProcessAnnotation))
                {
                    string annotationComment = line.Substring(commentStartIndex);
                    string id = FindAnnotationValue(CombinerProcessAnnotation, annotationComment);
                    var textBlock = idAndTextBlockList.FirstOrDefault(x => x.Id == id);
                    if (textBlock != null)
                    {
                        outputText.Append(textBlock.Text + returnCode);
                        continue;
                    }
                }

                outputText.Append(line + returnCode);
            }

            return outputText.ToString();
        }

        public static string FindAnnotationValue(string annotationName, string comment)
        {
            string[] spaceSplitedTokens = comment.Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);

            // ダブルクォーテーションで囲まれたトークンが切り離されているので結合する
            List<string> annotationTokens = new List<string>();
            for (int tokenIndex = 0; tokenIndex < spaceSplitedTokens.Length; ++tokenIndex)
            {
                string token = spaceSplitedTokens[tokenIndex];
                if ((token.Count(x => x == '"') == 1))
                {
                    StringBuilder combinedToken = new StringBuilder(token);
                    ++tokenIndex;
                    for (; tokenIndex < spaceSplitedTokens.Length; ++tokenIndex)
                    {
                        string subToken = spaceSplitedTokens[tokenIndex];
                        combinedToken.Append(' ' + subToken);
                        if ((subToken.Count(x => x == '"') == 1))
                        {
                            break;
                        }
                    }

                    annotationTokens.Add(combinedToken.ToString());
                }
                else
                {
                    annotationTokens.Add(token);
                }
            }

            // アノテーションの値を探す
            foreach (string annotationToken in annotationTokens)
            {
                if (annotationToken.StartsWith($"{annotationName}="))
                {
                    string[] idTokens = annotationToken.Split('=');
                    return idTokens[1].TrimEnd('\n').TrimEnd('\r').Trim('\"');
                }
            }

            return string.Empty;
        }

        private class CombinerOptionInfo
        {
            public string OptionId { get; set; }
            public string OptionMacroName { get; set; }
            public List<string> PresetFilePathList { get; } = new List<string>();
            public bool IsDefinedInOptionBlock { get; set; }
        }

        private class InsertTextInfo
        {
            public string Id { get; set; }
            public string Text { get; set; }
        }

        private static IEnumerable<CombinerOptionInfo> ExtractCombinerProcessAnnotation(string sourceCode)
        {
            List<CombinerOptionInfo> combinerOptionInfos = new List<CombinerOptionInfo>();
            string[] lines = sourceCode.Split('\n');
            foreach (string line in lines)
            {
                int commentStartIndex = line.IndexOf("//");
                if ((commentStartIndex != -1) && line.Contains("@@") && line.Contains(CombinerUtility.CombinerProcessAnnotation))
                {
                    string annotationComment = line.Substring(commentStartIndex);
                    string id = CombinerUtility.FindAnnotationValue(CombinerProcessAnnotation, annotationComment);
                    combinerOptionInfos.Add(new CombinerOptionInfo()
                    {
                        OptionId = id,
                    });
                }
            }

            return combinerOptionInfos;
        }

        private static string FindMacroName(string line)
        {
            string[] spaceSplitedList = line.Split(new char[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
            if (spaceSplitedList.Length < 3)
            {
                return string.Empty;
            }

            if (spaceSplitedList[0] == "#")
            {
                if (spaceSplitedList[1] != "define")
                {
                    return string.Empty;
                }
                else if (spaceSplitedList.Length < 4)
                {
                    return spaceSplitedList[2];
                }
                else
                {
                    return string.Empty;
                }
            }
            else if (spaceSplitedList[0] == "#define")
            {
                return spaceSplitedList[1];
            }
            else
            {
                return string.Empty;
            }
        }

        private static bool ContainsOptionId(string sourceCode, string optionId)
        {
            return sourceCode.Contains($"id=\"{optionId}\"");
        }

        private static bool FindChoiceValue(out string choiceValue, string line, string targetOptionId)
        {
            choiceValue = string.Empty;
            string macroName = FindMacroName(line);
            if (string.IsNullOrEmpty(macroName))
            {
                return false;
            }

            int commentStartIndex = line.IndexOf("//");
            if (commentStartIndex == -1)
            {
                return false;
            }

            string comment = line.Substring(commentStartIndex);
            string id = CombinerUtility.FindAnnotationValue("id", comment);
            if (targetOptionId != id)
            {
                return false;
            }

            choiceValue = CombinerUtility.FindAnnotationValue("choice", comment);
            return true;
        }

        private static bool FindDefaultValue(out string defaultValue, string line, string targetOptionId)
        {
            defaultValue = string.Empty;
            string macroName = FindMacroName(line);
            if (string.IsNullOrEmpty(macroName))
            {
                return false;
            }

            int commentStartIndex = line.IndexOf("//");
            if (commentStartIndex == -1)
            {
                return false;
            }

            string comment = line.Substring(commentStartIndex);
            string id = CombinerUtility.FindAnnotationValue("id", comment);
            if (targetOptionId != id)
            {
                return false;
            }

            defaultValue = CombinerUtility.FindAnnotationValue("default", comment);
            return true;
        }

        private static string ReplaceOrAddOptionChoiceForLine(string line, string targetOptionId,
            ShaderOptionChoice newChoice)
        {
            string newLine = line;

            string oldChoiceValue;
            bool isOptionAvailable = FindChoiceValue(out oldChoiceValue, newLine, targetOptionId);
            if (isOptionAvailable)
            {
                string newChoiceAnnotation = $" choice=\"{newChoice.CreateSerializableData()}\"";
                if (string.IsNullOrEmpty(oldChoiceValue))
                {
                    // 新規追加
                    newLine += newChoiceAnnotation;
                }
                else
                {
                    // 既存 choice の書き換え
                    newLine = newLine.Replace($" choice=\"{oldChoiceValue}\"", newChoiceAnnotation);
                }
            }

            return newLine;
        }

        private static string ReplaceOrAddOptionDefaultForLine(string line, string targetOptionId,
            string newDefaultChoiceValue)
        {
            string newLine = line;

            string oldDefaultValue;
            bool isOptionAvailable = FindDefaultValue(out oldDefaultValue, newLine, targetOptionId);
            if (isOptionAvailable)
            {
                string newDefaultAnnotation = $" default=\"{newDefaultChoiceValue}\"";
                if (string.IsNullOrEmpty(oldDefaultValue))
                {
                    // 新規追加
                    newLine += newDefaultAnnotation;
                }
                else
                {
                    // 既存 default の書き換え
                    newLine = newLine.Replace($" default=\"{oldDefaultValue}\"", newDefaultAnnotation);
                }
            }

            return newLine;
        }

        private static string ReplaceOrAddOptionChoiceAndDefault(
            string optionDefGlslSourceCode,
            string targetOptionId,
            ShaderOptionChoice newChoice,
            string newDefaultChoiceValue)
        {
            // オプションの choice を書き換える
            if (!ContainsOptionId(optionDefGlslSourceCode, targetOptionId))
            {
                return optionDefGlslSourceCode;
            }

            string returnCode = GlslUtility.EstimateReturnCode(optionDefGlslSourceCode);

            List<string> newLines = new List<string>();
            string[] lines = optionDefGlslSourceCode.Split(new string[] { returnCode }, StringSplitOptions.None);
            foreach (string line in lines)
            {
                string newLine = ReplaceOrAddOptionChoiceForLine(line, targetOptionId, newChoice);
                newLine = ReplaceOrAddOptionDefaultForLine(newLine, targetOptionId, newDefaultChoiceValue);
                newLines.Add(newLine);
            }

            // 書き換え前後の状態をできるだけ保つために改行コードは EstimateReturnCode() の推測値を使う。
            StringBuilder newSourceCode = new StringBuilder(string.Join(returnCode, newLines));

            // 書き換え前後の状態をできるだけ保つために、末尾の改行コードの追加は
            // 書き換え前後で行数が異り、かつ最終行が空でない場合に行う。
            if ((lines.Length != newLines.Count) && !string.IsNullOrEmpty(newLines.LastOrDefault()))
            {
                newSourceCode.Append(returnCode);
            }

            return newSourceCode.ToString();
        }

        /// <summary>
        /// ソースコードの先頭にオプション変数用の #define プリプロセッサーを埋め込みます。
        /// </summary>
        /// <param name="sourceCode"></param>
        /// <param name="optionInfos"></param>
        /// <param name="presetFilePaths"></param>
        private static string EmbedOptionValueDefinitions(
            string sourceCode,
            IEnumerable<CombinerOptionInfo> combinerOptionInfos,
            IEnumerable<string> presetFilePaths)
        {
            // TODO: このメソッドではすべてのオプション定義を埋め込んでしまうので、ソースコードに定義されているものだけを埋め込むように修正する
            // (今の実装だと複数のcombiner_processが複数ファイルに分かれて記述されている場合に冗長な定義が挿入されてしまう)

            StringBuilder optionDefinitionSourceCodeBuilder = new StringBuilder();
            int combinerVariationValue = CombinerActiveVariationStartValue;

            string returnCode = GlslUtility.EstimateReturnCode(sourceCode);
            string beginMark = $"// {BeginMark} {Assembly.GetExecutingAssembly().GetName().Name}{returnCode}";
            string endMark = $"// {EndMark} {Assembly.GetExecutingAssembly().GetName().Name}{returnCode}";

            // 以前に挿入されたコードは一旦取り除く
            int endIndex = sourceCode.IndexOf(endMark);
            if (endIndex != -1)
            {
                sourceCode = sourceCode.Substring(endIndex + endMark.Length);
            }

            // 新規コードの挿入
            optionDefinitionSourceCodeBuilder.Append(beginMark);
            foreach (string presetFilePath in presetFilePaths)
            {
                string presetFileName = System.IO.Path.GetFileNameWithoutExtension(presetFilePath);
                optionDefinitionSourceCodeBuilder.AppendLine($"#define {presetFileName} ({combinerVariationValue})");
                ++combinerVariationValue;
            }

            foreach (var info in combinerOptionInfos)
            {
                // マクロが定義されていないものは追加する
                if (string.IsNullOrEmpty(info.OptionMacroName))
                {
                    info.OptionMacroName = info.OptionId.ToUpper();
                    optionDefinitionSourceCodeBuilder.AppendLine($"#define {info.OptionMacroName} (0) // @@ id=\"{info.OptionId}\"");
                }
            }
            optionDefinitionSourceCodeBuilder.Append(endMark);

            return optionDefinitionSourceCodeBuilder.ToString() + sourceCode;
        }

        private static string ReplaceChoiceMacroValue(string sourceCode, string choiceMacroName, string choiceValue)
        {
            (string autoGenSourceCode, string originalSourceCode) = ExtractAutoGenCodeAndOriginalCode(sourceCode);
            string returnCode = GlslUtility.EstimateReturnCode(sourceCode);
            StringBuilder newAutogenCode = new StringBuilder();
            foreach (var line in autoGenSourceCode.Split(new string[] { returnCode }, StringSplitOptions.None))
            {
                if (line.Contains("#define") && line.Contains(choiceMacroName))
                {
                    newAutogenCode.Append($"#define {choiceMacroName} ({choiceValue}){returnCode}");
                }
                else
                {
                    newAutogenCode.Append($"{line}{returnCode}");
                }
            }

            string newAutogenCodeText = newAutogenCode.ToString();
            return newAutogenCodeText.Substring(0, newAutogenCodeText.Length - returnCode.Length) + originalSourceCode;
        }

        private static (string autoGenCode, string originalCode) ExtractAutoGenCodeAndOriginalCode(string sourceCode)
        {
            string returnCode = GlslUtility.EstimateReturnCode(sourceCode);
            string endMark = $"// END generated code by {Assembly.GetExecutingAssembly().GetName().Name}{returnCode}";

            string originalSourceCode = sourceCode;
            string autoGenSourceCode = string.Empty;
            int endIndex = sourceCode.IndexOf(endMark);
            if (endIndex == -1)
            {
                return (string.Empty, sourceCode);
            }

            autoGenSourceCode = sourceCode.Substring(0, endIndex + endMark.Length);
            originalSourceCode = sourceCode.Substring(endIndex + endMark.Length);
            return (autoGenSourceCode, originalSourceCode);
        }

        private static string ReplaceAutoGeneratedCode(
            string sourceCode,
            string oldText,
            string newText)
        {
            (string autoGenSourceCode, string originalSourceCode) = ExtractAutoGenCodeAndOriginalCode(sourceCode);
            return autoGenSourceCode.Replace(oldText, newText) + originalSourceCode;
        }

        private static string EmbedOptionChoice(
            string sourceCode, IEnumerable<CombinerOptionInfo> targetOptionInfos, ShaderOptionChoice choice, string defaultChoiceValue)
        {
            foreach (var optionInfo in targetOptionInfos)
            {
                sourceCode = ReplaceOrAddOptionChoiceAndDefault(sourceCode, optionInfo.OptionId, choice, defaultChoiceValue);
            }

            return sourceCode;
        }

        private static IEnumerable<ShaderSrc> EnumerateSourcesWithIncludes(ShaderSrc shaderSource, ShaderDefinition shaderDef)
        {
            yield return shaderSource;
            string sourceCode = shaderSource.Stream.Value;
            foreach (var includeFilePath in GlslUtility.ExtractIncludeFilePaths(sourceCode))
            {
                ShaderSrc includeSource = shaderDef.ShaderSrcs.First(x => x.Path == includeFilePath);
                yield return includeSource;
            }
        }
    }
}
