﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// RRenderState RDepthTest RAlphaTest RColorBlend RLogicalBlend
// RSampler ROriginalTexsrt
// RStencilTest
// RMaterial ROriginalColor

//=============================================================================
// include
//=============================================================================
#include "DccMaterial.h"
#include "DccImage.h"
#include "DccAnimation.h"
#include "DccOutput.h"

using namespace std;

//=============================================================================
// dcc ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {

//-----------------------------------------------------------------------------
//! @brief デプステストを出力します。
//-----------------------------------------------------------------------------
void RDepthTest::Out(std::ostream& os, const int tabCount) const
{
    static const char* const funcStrs[] =
    {
        "never", "less", "equal", "lequal", "greater", "nequal", "gequal", "always"
    };

    os << RTab(tabCount) << "<depth_test"
       << " enable=\"" << RBoolStr(m_Enable)
       << "\" write=\"" << RBoolStr(m_Write)
       << "\" func=\"" << funcStrs[m_Func]
       << "\" />" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief アルファテストを出力します。
//-----------------------------------------------------------------------------
void RAlphaTest::Out(std::ostream& os, const int tabCount) const
{
    static const char* const funcStrs[] =
    {
        "never", "less", "equal", "lequal", "greater", "nequal", "gequal", "always"
    };

    os << RTab(tabCount) << "<alpha_test"
       << " enable=\"" << RBoolStr(m_Enable)
       << "\" func=\"" << funcStrs[m_Func]
       << "\" value=\"" << m_Value
       << "\" />" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief カラーブレンドを出力します。
//-----------------------------------------------------------------------------
void RColorBlend::Out(std::ostream& os, const int tabCount) const
{
    static const char* const funcStrs[] =
    {
        "zero", "one",
        "src_color", "one_minus_src_color",
        "src_alpha", "one_minus_src_alpha",
        "dst_color", "one_minus_dst_color",
        "dst_alpha", "one_minus_dst_alpha",
        "const_color", "one_minus_const_color",
        "const_alpha", "one_minus_const_alpha",
        "src1_color", "one_minus_src1_color",
        "src1_alpha", "one_minus_src1_alpha",
        "src_alpha_saturate",
    };
    static const char* const opStrs[] =
    {
        "add", "src_minus_dst", "min", "max", "dst_minus_src"
    };

    const int& tc = tabCount;
    os << RTab(tc) << "<color_blend" << R_ENDL;
    os << RTab(tc + 1) << "rgb_src_func=\"" << funcStrs[m_RgbSrcFunc] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "rgb_dst_func=\"" << funcStrs[m_RgbDstFunc] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "rgb_op=\"" << opStrs[m_RgbOp] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "alpha_src_func=\"" << funcStrs[m_AlphaSrcFunc] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "alpha_dst_func=\"" << funcStrs[m_AlphaDstFunc] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "alpha_op=\"" << opStrs[m_AlphaOp] << "\"" << R_ENDL;
    ROutVecAttr(os, tc + 1, "const_color", m_ConstColor);
    os << RTab(tc) << "/>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 論理ブレンドを出力します。
//-----------------------------------------------------------------------------
void RLogicalBlend::Out(std::ostream& os, const int tabCount) const
{
    static const char* const opStrs[] =
    {
        "clear", "set", "copy", "inv_copy", "no_op", "inv",
        "and", "nand", "or", "nor", "xor", "equiv",
        "rev_and", "inv_and", "rev_or", "inv_or"
    };

    os << RTab(tabCount) << "<logical_blend op=\"" << opStrs[m_Op] << "\" />" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief レンダーステートを出力します。
//-----------------------------------------------------------------------------
void RRenderState::Out(std::ostream& os, const int tabCount) const
{
    static const char* const modeStrs[] = { "custom", "opaque", "mask", "translucent" };
    static const char* const displayFaceStrs[] =
    {
        "both", "front", "back", "none"
    };
    static const char* const blendModeStrs[] = { "none", "color", "logic" };

    const int& tc = tabCount;
    os << RTab(tc) << "<render_state mode=\"" << modeStrs[m_Mode]
       << "\" display_face=\"" << displayFaceStrs[m_DisplayFace]
       << "\" blend_mode=\"" << blendModeStrs[m_BlendMode]
       << "\">" << R_ENDL;
    m_DepthTest.Out(os, tc + 1);
    m_AlphaTest.Out(os, tc + 1);
    m_ColorBlend.Out(os, tc + 1);
    m_LogicalBlend.Out(os, tc + 1);
    os << RTab(tc) << "</render_state>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief レンダーステートのモードを設定します。
//!
//! @param[in] mode レンダーステートのモードです。
//-----------------------------------------------------------------------------
void RRenderState::SetMode(const Mode mode)
{
    m_Mode = mode;
    if (m_Mode == CUSTOM)
    {
        return;
    }

    m_BlendMode    = BLEND_NONE;
    m_DepthTest    = RDepthTest();    // デフォルトはデプステスト有効です。
    m_AlphaTest    = RAlphaTest();    // デフォルトはアルファテスト無効です。
    m_ColorBlend   = RColorBlend();   // デフォルトのカラーブレンドを設定します。
    m_LogicalBlend = RLogicalBlend(); // デフォルトの論理ブレンドを設定します。

    switch (m_Mode)
    {
    default:
    case OPA:
        break;

    case MASK:
        m_AlphaTest.m_Enable = true;
        m_AlphaTest.m_Func = RAlphaTest::GEQUAL;
        m_AlphaTest.m_Value = 0.5f;
        break;

    case XLU:
        m_BlendMode = COLOR;
        m_DepthTest.m_Write = false;
        break;
    }
}

//-----------------------------------------------------------------------------
//! @brief ステンシルテストを出力します。
//-----------------------------------------------------------------------------
//void RStencilTest::Out(std::ostream& os, const int tabCount) const
//{
//  os << RTab(tabCount) << "<stencil_test"
//     << " enable=\"" << RBoolStr(m_Enable)
//     << "\" TestFunction=\"Never"
//     << "\" TestReference=\"0"
//     << "\" TestMask=\"255"
//     << "\" FailOperation=\"Keep"
//     << "\" ZFailOperation=\"Keep"
//     << "\" PassOperation=\"Keep"
//     << "\" />" << R_ENDL;
//}

//-----------------------------------------------------------------------------
//! @brief オリジナルテクスチャ SRT の計算方式の文字列を返します。
//-----------------------------------------------------------------------------
std::string ROriginalTexsrt::GetModeString() const
{
    static const char* const modeStrs[] = { "maya", "3dsmax", "softimage" };
    return modeStrs[m_Mode];
}

//-----------------------------------------------------------------------------
//! @brief オリジナルテクスチャ SRT を出力します。
//-----------------------------------------------------------------------------
void ROriginalTexsrt::Out(
    std::ostream& os,
    const int tabCount,
    const int index,
    const std::string& hintStr
) const
{

    const int& tc = tabCount;
    os << RTab(tc) << "<original_texsrt"
       << " index=\"" << index
       << "\" hint=\"" << hintStr
       << "\" uv_hint=\"" << "uv" << m_UvHintIdx
       << "\"" << R_ENDL;
    os << RTab(tc + 1) << "mode=\"" << GetModeString() << "\"" << R_ENDL;
    ROutVecAttr(os, tc + 1, "scale", m_Scale);
    os << RTab(tc + 1) << "rotate=\"" << m_Rotate << "\"" << R_ENDL;
    ROutVecAttr(os, tc + 1, "translate", m_Translate);
    os << RTab(tc) << "/>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャ SRT アニメーションするパラメータの名前を返します。
//-----------------------------------------------------------------------------
const char* ROriginalTexsrt::GetParamName(const int paramIdx)
{
    static const char* const paramNames[] =
    {
        "scale_x",
        "scale_y",
        "rotate",
        "translate_x",
        "translate_y"
    };
    return (0 <= paramIdx && paramIdx < PARAM_COUNT) ?
        paramNames[paramIdx] : "?";
}

//-----------------------------------------------------------------------------
// サンプラの name と hint の値のデータです。
//-----------------------------------------------------------------------------
static const char* const s_SamplerNameHintStrs[RSampler::HINT_COUNT][2] =
{
    { "_a", "albedo"     },
    { "_o", "opacity"    },
    { "_e", "emission"   },
    { "_n", "normal"     },
    { "_t", "tangent"    },
    { "_s", "specular"   },
    { "_r", "reflection" },
    { "_x", "extra"      }, // 実際は使用されません。
};

//-----------------------------------------------------------------------------
//! @brief サンプラの名前を返します。
//-----------------------------------------------------------------------------
std::string RSampler::GetName() const
{
    if (!m_NameOverride.empty())
    {
        return m_NameOverride;
    }
    else
    {
        return (m_Hint < HINT_COUNT) ?
            s_SamplerNameHintStrs[m_Hint][0] + RGetNumberString(m_HintIndex) : "invalid";
    }
}

//-----------------------------------------------------------------------------
//! @brief サンプラのヒント情報の文字列を返します。
//-----------------------------------------------------------------------------
std::string RSampler::GetHintString() const
{
    if (!m_HintOverride.empty())
    {
        return m_HintOverride;
    }
    else if (m_Hint == EXTRA)
    {
        return GetName();
    }
    else
    {
        return  (m_Hint < HINT_COUNT) ?
            s_SamplerNameHintStrs[m_Hint][1] + RGetNumberString(m_HintIndex) : "invalid";
    }
}

//-----------------------------------------------------------------------------
//! @brief サンプラのテクスチャイメージ用のヒント情報の文字列を返します。
//-----------------------------------------------------------------------------
std::string RSampler::GetImageHintString() const
{
    if (!m_HintOverride.empty() || m_Hint == EXTRA)
    {
        // ヒント情報オーバーライドの末尾の数字部分をカットして返します。
        // すべて数字ならカットせずにそのまま返します。
        const std::string samplerHint = GetHintString();
        std::string digit;
        const std::string imgHint = RSplitStringEndDigit(digit, samplerHint);
        return (!imgHint.empty()) ? imgHint : samplerHint;
    }
    else
    {
        switch (m_Hint)
        {
        case ALBEDO    : return RImage::HINT_ALBEDO    ;
        case OPACITY   : return RImage::HINT_OPACITY   ;
        case EMISSION  : return RImage::HINT_EMISSION  ;
        case NORMAL    : return RImage::HINT_NORMAL    ;
        case TANGENT   : return RImage::HINT_TANGENT   ;
        case SPECULAR  : return RImage::HINT_SPECULAR  ;
        case REFLECTION: return RImage::HINT_REFLECTION;
        default: break;
        }
        return "";
    }
}

//-----------------------------------------------------------------------------
//! @brief サンプラを出力します。
//-----------------------------------------------------------------------------
void RSampler::Out(std::ostream& os, const int tabCount, const int samplerIdx) const
{
    static const char* const wrapStrs[] =
    {
        "repeat",
        "mirror",
        "clamp",
        "mirror_once",
        //"clamp_half_border",
        //"mirror_once_half_border",
        //"clamp_border",
        //"mirror_once_border"
    };
    static const char* const filterStrs[] = { "none", "point", "linear" };
    static const char* const maxAnisoStrs[] =
    {
        "aniso_1", "aniso_2", "aniso_4", "aniso_8", "aniso_16"
    };

    const int& tc = tabCount;
    os << RTab(tc) << "<sampler"
       << " sampler_index=\"" << samplerIdx
       << "\" name=\"" << GetName()
       << "\" hint=\"" << GetHintString()
       << "\" tex_name=\"" << m_TexName
       << "\">" << R_ENDL;

    os << RTab(tc + 1) << "<wrap"
       <<   " u=\"" << wrapStrs[m_WrapU]
       << "\" v=\"" << wrapStrs[m_WrapV]
       << "\" w=\"" << wrapStrs[m_WrapW]
       << "\" />" << R_ENDL;
    os << RTab(tc + 1) << "<filter"
       <<   " mag=\"" << filterStrs[m_FilterMag]
       << "\" min=\"" << filterStrs[m_FilterMin]
       << "\" mip=\"" << filterStrs[m_FilterMip]
       << "\" max_aniso=\"" << maxAnisoStrs[m_MaxAniso]
       << "\" />" << R_ENDL;
    os << RTab(tc + 1) << "<lod"
       <<   " min=\"" << m_LodMin
       << "\" max=\"" << m_LodMax
       << "\" bias=\"" << m_LodBias
       << "\" />" << R_ENDL;
    //os << RTab(tc + 1) << "<tex_matrix"
    //   <<   " scale=\"" << m_Scale
    //   << "\" rotate=\"" << m_Rotate
    //   << "\" translate=\"" << m_Translate
    //   << "\" />" << R_ENDL;

    os << RTab(tc) << "</sampler>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief マテリアルを出力します。
//-----------------------------------------------------------------------------
void RMaterial::Out(
    std::ostream& os,
    const int tabCount,
    const int matIdx
) const
{
    const int& tc = tabCount;

    //-----------------------------------------------------------------------------
    // begin material
    os << RTab(tc) << "<material index=\"" << matIdx
       << "\" name=\"" << RGetUtf8FromShiftJis(m_Name)
       << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // material info
    os << RTab(tc + 1) << "<material_info" << R_ENDL;
    os << RTab(tc + 2) << "visibility=\"" << RBoolStr(m_Visibility) << "\"" << R_ENDL;
    os << RTab(tc + 2) << "mesh_adjacency=\"" << RBoolStr(false) << "\"" << R_ENDL;
    os << RTab(tc + 2) << "compress_enable=\"" << RBoolStr(m_CompressEnable) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "/>" << R_ENDL;

    //-----------------------------------------------------------------------------
    // render state
    m_RenderState.Out(os, tc + 1);

    //-----------------------------------------------------------------------------
    // samplers
    ROutArrayElement(os, tc + 1, m_Samplers, "sampler_array");

    //-----------------------------------------------------------------------------
    // マテリアルのユーザーデータ配列を出力します。
    ROutArrayElement(os, tc + 1, m_UserDatas, "user_data_array");

    //-----------------------------------------------------------------------------
    // end material
    os << RTab(tc) << "</material>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief オリジナルカラーを出力します。
//-----------------------------------------------------------------------------
void ROriginalColor::Out(std::ostream& os, const int tabCount, const int index) const
{
    static const char* const hintStrs[] =
    {
        "diffuse" ,
        "opacity" ,
        "ambient" ,
        "emission",
        "specular"
    };

    const int& tc = tabCount;
    os << RTab(tc) << "<original_color"
       << " index=\"" << index
       << "\" hint=\"" << hintStrs[m_Hint]
       << "\"" << R_ENDL;
    ROutVecAttr(os, tc + 1, "color", m_Color);
    os << RTab(tc) << "/>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief オリジナルマテリアルを出力します。
//-----------------------------------------------------------------------------
void RMaterial::OutOriginal(
    std::ostream& os,
    const int tabCount,
    const int matIdx
) const
{
    const int& tc = tabCount;

    //-----------------------------------------------------------------------------
    // begin original material
    os << RTab(tc) << "<original_material index=\"" << matIdx
       << "\" mat_name=\"" << RGetUtf8FromShiftJis(m_Name)
       << "\">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // original color array
    ROriginalColorArray orgColors;
    orgColors.push_back(ROriginalColor(ROriginalColor::DIFFUSE , m_Diffuse ));
    orgColors.push_back(ROriginalColor(ROriginalColor::OPACITY , m_Opacity ));
    orgColors.push_back(ROriginalColor(ROriginalColor::AMBIENT , m_Ambient ));
    orgColors.push_back(ROriginalColor(ROriginalColor::EMISSION, m_Emission));
    if (m_IsSpecularEnable)
    {
        orgColors.push_back(ROriginalColor(ROriginalColor::SPECULAR, m_Specular));
    }
    ROutArrayElement(os, tc + 1, orgColors, "original_color_array");

    //-----------------------------------------------------------------------------
    // original texsrt array
    const int samplerCount = static_cast<int>(m_Samplers.size());
    int orgTexSrtCount = 0;
    for (int iSampler = 0; iSampler < samplerCount; ++iSampler)
    {
        if (!m_Samplers[iSampler].IsReflection())
        {
            ++orgTexSrtCount;
        }
    }

    if (orgTexSrtCount > 0)
    {
        int iOrgTexSrt = 0;
        os << RTab(tc + 1) << "<original_texsrt_array length=\"" << orgTexSrtCount << "\">" << R_ENDL;
        for (int iSampler = 0; iSampler < samplerCount; ++iSampler)
        {
            const RSampler& sampler = m_Samplers[iSampler];
            if (!sampler.IsReflection())
            {
                sampler.m_OriginalTexsrt.Out(os, tc + 2, iOrgTexSrt, sampler.GetHintString());
                ++iOrgTexSrt;
            }
        }
        os << RTab(tc + 1) << "</original_texsrt_array>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // end original material
    os << RTab(tc) << "</original_material>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 他のマテリアルと要素の値が同じなら true を返します。
//-----------------------------------------------------------------------------
bool RMaterial::IsSameElement(const RMaterial& other) const
{
    if (m_RenderState != other.m_RenderState ||

        m_Diffuse     != other.m_Diffuse     ||
        m_Opacity     != other.m_Opacity     ||
        m_Ambient     != other.m_Ambient     ||
        m_Emission    != other.m_Emission    ||
        m_Specular    != other.m_Specular)
    {
        return false;
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief カラーアニメーションするパラメータの名前を返します。
//-----------------------------------------------------------------------------
const char* RMaterial::GetColorParamName(const int paramIdx)
{
    static const char* const paramNames[] =
    {
        "diffuse_r" , "diffuse_g" , "diffuse_b" ,
        "opacity_r" , "opacity_g" , "opacity_b" ,
        "ambient_r" , "ambient_g" , "ambient_b" ,
        "emission_r", "emission_g", "emission_b",
        "specular_r", "specular_g", "specular_b"
    };
    return (0 <= paramIdx && paramIdx < COLOR_PARAM_COUNT) ?
        paramNames[paramIdx] : "?";
}

//=============================================================================
// dcc ネームスペースを終了します。
//=============================================================================
} // namespace dcc
} // namespace tool
} // namespace gfx
} // namespace nn

