﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinPerMaterialAnim.h>
#include <BinMaterial.h>
#include <util/UtilMath.h>
#include <algorithm>


namespace nn {
namespace g3dTool {
namespace
{
NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    shader_param, type, ctype,
    BinAnimCurve::BOOL, BinAnimCurve::BOOL, BinAnimCurve::BOOL, BinAnimCurve::BOOL,
    BinAnimCurve::INT, BinAnimCurve::INT, BinAnimCurve::INT, BinAnimCurve::INT,
    BinAnimCurve::INT, BinAnimCurve::INT, BinAnimCurve::INT, BinAnimCurve::INT,
    BinAnimCurve::FLOAT, BinAnimCurve::FLOAT, BinAnimCurve::FLOAT, BinAnimCurve::FLOAT,
    BinAnimCurve::FLOAT, BinAnimCurve::FLOAT, BinAnimCurve::FLOAT,
    BinAnimCurve::FLOAT, BinAnimCurve::FLOAT, BinAnimCurve::FLOAT,
    BinAnimCurve::FLOAT, BinAnimCurve::FLOAT, BinAnimCurve::FLOAT,
    BinAnimCurve::FLOAT, BinAnimCurve::FLOAT,
    BinAnimCurve::FLOAT, BinAnimCurve::FLOAT
    );
}

void BinPerMaterialAnim::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_nn_per_material_anim& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    std::vector<const nw::g3d::tool::g3dif::elem_anim_curve*> curveArray;
    if( elem.shader_param_anim_array )
    {
        for (auto iter = elem.shader_param_anim_array.Get().cbegin();
            iter != elem.shader_param_anim_array.Get().cend(); ++iter)
        {
            for (auto iterTarget = iter->param_anim_target_array.cbegin();
                iterTarget != iter->param_anim_target_array.cend(); ++iterTarget)
            {
                if (iterTarget->curve)
                {
                    curveArray.push_back(&iterTarget->curve.Get());
                    ++m_ShaderParamCurveCount;
                }
                else
                {
                    ++m_ShaderParamConstantCount;
                }
            }
        }
    }
    if( elem.tex_pattern_anim_array )
    {
        for (auto iter = elem.tex_pattern_anim_array.Get().cbegin(); iter != elem.tex_pattern_anim_array.Get().cend(); ++iter)
        {
            if (iter->curve)
            {
                curveArray.push_back(&iter->curve.Get());
                ++m_TexPatternCurveCount;
            }
            else
            {
                ++m_TexPatternConstantCount;
            }
        }
    }
    if( elem.material_visibility_anim )
    {
        if( elem.material_visibility_anim.Get().curve )
        {
            curveArray.push_back(&elem.material_visibility_anim.Get().curve.Get());
            ++m_MatVisAnimCurveCount;
        }
        else
        {
            ++m_MatVisConstantCount;
        }
    }
    m_ConstantCount = m_ShaderParamConstantCount + m_TexPatternConstantCount + m_MatVisConstantCount;

    m_CurveArray.resize(curveArray.size());
    SetParentBlockArray(m_CurveArray, this);
    BuildArray(pCtx, m_CurveArray, curveArray);

    int curveIndex = 0;
    if( elem.shader_param_anim_array )
    {
        for (auto iter = elem.shader_param_anim_array.Get().cbegin();
            iter != elem.shader_param_anim_array.Get().cend(); ++iter)
        {
            for (auto iterTarget = iter->param_anim_target_array.cbegin();
                iterTarget != iter->param_anim_target_array.cend();
                ++iterTarget)
            {
                if (iterTarget->curve)
                {
                    BinAnimCurve& curve = m_CurveArray[curveIndex];
                    curve.SetTargetOffset(sizeof(nn::Bit32) * iterTarget->component_index.value);
                    curve.SetType(static_cast<BinAnimCurve::Type>(ToEnum_ctype(iter->type.value)));
                    if (iter->type.value == nw::g3d::tool::g3dif::elem_shader_param::type_texsrt
                        && iterTarget->component_index.value == 0)
                    {
                        curve.SetType(BinAnimCurve::INT);
                    }

                    // rotate の場合は無条件で Degree であると仮定する。
                    // Quaternion の場合は SetRotateMode で false に設定する。
                    if ((iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_2D_COMPONENT_INDEX) &&
                        (iter->type.value == nw::g3d::tool::g3dif::elem_shader_param::type_srt2d))
                    {
                        curve.SetDegreeValue(true);
                    }
                    else if (
                        ((iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_3D_COMPONENT_INDEX_X) ||
                        (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_3D_COMPONENT_INDEX_Y) ||
                        (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_3D_COMPONENT_INDEX_Z)) &&
                        iter->type.value == nw::g3d::tool::g3dif::elem_shader_param::type_srt3d)
                    {
                        curve.SetDegreeValue(true);
                    }
                    else if ((iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_2D_COMPONENT_INDEX_SRT) &&
                             (iter->type.value >= nw::g3d::tool::g3dif::elem_shader_param::type_texsrt))
                    {
                        // texsrt と texsrt_ex
                        curve.SetDegreeValue(true);
                    }
                    ++curveIndex;
                }
            }
        }
    }
    if( elem.tex_pattern_anim_array )
    {
        const int texPatternCurveIdxEnd = curveIndex + m_TexPatternCurveCount;
        for( ; curveIndex < texPatternCurveIdxEnd; ++curveIndex )
        {
            m_CurveArray[curveIndex].SetTargetOffset(0);
            m_CurveArray[curveIndex].SetType(BinAnimCurve::INT);
        }
    }
    if( elem.material_visibility_anim )
    {
        if ( elem.material_visibility_anim.Get().curve )
        {
            BinAnimCurve& curve = m_CurveArray[curveIndex];
            curve.SetTargetOffset(0);			//!< TODO: TargetOffsegt とは？とりあえず 0 を入れておく。
            curve.SetType(BinAnimCurve::BOOL);
            ++curveIndex;
        }
    }

    // 文字列の登録。
    pCtx->SetStr( elem.mat_name.value.c_str() );
    if( elem.shader_param_anim_array )
    {
        for (auto iter = elem.shader_param_anim_array.Get().cbegin(); iter != elem.shader_param_anim_array.Get().cend(); ++iter)
        {
            pCtx->SetStr( iter->id.value.c_str() );
        }
    }
    if( elem.tex_pattern_anim_array )
    {
        for (auto iter = elem.tex_pattern_anim_array.Get().cbegin();
            iter != elem.tex_pattern_anim_array.Get().cend(); ++iter)
        {
            pCtx->SetStr(iter->sampler_name.value.c_str());
        }
    }
    if( elem.material_visibility_anim )
    {
        // do nothing.
    }
}

void BinPerMaterialAnim::CalculateSize()
{
    if( m_pElem->shader_param_anim_array )
    {
        m_Chunk[ChunkType_ParamInfo].size = sizeof( nn::g3d::ResShaderParamAnimInfo ) * m_pElem->shader_param_anim_array.Get().size();
    }
    m_Chunk[ChunkType_TexInfo].size = sizeof(nn::g3d::ResTexturePatternAnimInfo) * m_pElem->tex_pattern_anim_array.Get().size();
    m_Chunk[ChunkType_Constant].size = sizeof(nn::g3d::ResAnimConstantData) * m_ConstantCount;
    m_Chunk[ChunkType_Curve].size = sizeof(nn::g3d::ResAnimCurveData) * m_CurveArray.size();
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinPerMaterialAnim::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Curve].offset;
    for (auto iter = m_CurveArray.begin(); iter != m_CurveArray.end(); ++iter)
    {
        iter->SetStructOffset(offset);
        offset += sizeof(nn::g3d::ResAnimCurveData);
    }
}

void BinPerMaterialAnim::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResPerMaterialAnimData& matAnim = *GetPtr<nn::g3d::ResPerMaterialAnimData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ));
    matAnim.shaderParamAnimCount = static_cast<uint16_t>(m_pElem->shader_param_anim_array.Get().size());
    matAnim.curveCount = static_cast<uint16_t>(m_CurveArray.size());
    matAnim.constantCount = static_cast<uint16_t>(m_ConstantCount);
    m_pElem->shader_param_anim_array ?
        matAnim.beginShaderParamCurveIndex = static_cast<uint16_t>( m_BeginCurve ) : matAnim.beginShaderParamCurveIndex = nn::g3d::AnimFlag_NotBound;
    m_pElem->tex_pattern_anim_array ?
        matAnim.beginTexturePatternCurveIndex = m_BeginCurve + m_ShaderParamCurveCount : matAnim.beginTexturePatternCurveIndex = nn::g3d::AnimFlag_NotBound;
    m_pElem->material_visibility_anim ?
        matAnim.beginVisibilityCurveIndex = m_BeginCurve + m_ShaderParamCurveCount + m_TexPatternCurveCount : matAnim.beginVisibilityCurveIndex = nn::g3d::AnimFlag_NotBound;

    if( m_pElem->shader_param_anim_array )
    {
        nn::g3d::ResShaderParamAnimInfo* pParamAnimInfo = GetPtr<nn::g3d::ResShaderParamAnimInfo>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_ParamInfo].offset);
        int paramInfoIndex = 0;
        int curveIndex = 0;
        int constantIndex = 0;
        for (auto iter = m_pElem->shader_param_anim_array.Get().cbegin(); iter != m_pElem->shader_param_anim_array.Get().cend(); ++iter, ++paramInfoIndex)
        {
            nn::g3d::ResShaderParamAnimInfo& paramAnimInfo = pParamAnimInfo[paramInfoIndex];
            pCtx->LinkStr( &paramAnimInfo.pName, iter->id.value.c_str() );
            paramAnimInfo.beginConstant = static_cast<uint16_t>(constantIndex);
            paramAnimInfo.beginCurve =static_cast<uint16_t>(curveIndex);
            paramAnimInfo.subbindIndex = nn::g3d::AnimFlag_NotBound;

            int numFloatCurve = 0;
            int numIntCurve = 0;
            int numConstant = 0;
            for (auto iterTarget = iter->param_anim_target_array.cbegin(); iterTarget != iter->param_anim_target_array.cend(); ++ iterTarget)
            {
                if (iterTarget->curve)
                {
                    BinAnimCurve& curve = m_CurveArray[curveIndex];
                    if (curve.GetType() == BinAnimCurve::FLOAT)
                    {
                        ++numFloatCurve;
                    }
                    else
                    {
                        ++numIntCurve;
                    }

                    ++curveIndex;
                }
                else
                {
                    ++numConstant;
                    ++constantIndex;
                }
            }

            paramAnimInfo.floatCurveCount = static_cast<uint16_t>( numFloatCurve );
            paramAnimInfo.intCurveCount = static_cast<uint16_t>( numIntCurve );
            paramAnimInfo.constantCount = static_cast<uint16_t>( numConstant );
        }

        pCtx->LinkPtr( &matAnim.pShaderParamAnimInfoArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_ParamInfo].offset) );

        nn::g3d::ResAnimConstantData* pConstant = GetPtr<nn::g3d::ResAnimConstantData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Constant].offset);
        int targetIndex = 0;
        for (auto iter = m_pElem->shader_param_anim_array.Get().cbegin(); iter != m_pElem->shader_param_anim_array.Get().cend(); ++iter)
        {
            for (auto iterTarget = iter->param_anim_target_array.cbegin();
                iterTarget != iter->param_anim_target_array.cend();
                ++iterTarget)
            {
                if (!iterTarget->curve)
                {
                    // 型を判定して ChunkType_Constant に値を入れる。
                    if ((iter->type.value == nw::g3d::tool::g3dif::elem_shader_param::type_srt2d) &&
                        (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_2D_COMPONENT_INDEX))
                    {
                        // テクスチャ srt の Rotate の場合は Degree を Radian にして格納する。
                        pConstant[targetIndex].fValue = nn::util::DegreeToRadian(iterTarget->base_value.value);
                    }
                    else if ((iter->type.value == nw::g3d::tool::g3dif::elem_shader_param::type_srt3d) &&
                        ((iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_3D_COMPONENT_INDEX_X) ||
                        (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_3D_COMPONENT_INDEX_Y) ||
                        (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_3D_COMPONENT_INDEX_Z)))
                    {
                        pConstant[targetIndex].fValue = nn::util::DegreeToRadian(iterTarget->base_value.value);
                    }
                    else if (iter->type.value >= nw::g3d::tool::g3dif::elem_shader_param::type_texsrt)
                    {
                        if (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::ROTATE_2D_COMPONENT_INDEX_SRT)
                        {
                            // テクスチャ srt の Rotate の場合は Degree を Radian にして格納する。
                            pConstant[targetIndex].fValue = nn::util::DegreeToRadian(iterTarget->base_value.value);
                        }
                        else if (iterTarget->component_index.value == nw::g3d::tool::g3dif::elem_shader_param::MODE_COMPONENT_INDEX)
                        {
                            // テクスチャ srt の Mode の場合は uint32_t に変換して格納する。
                            pConstant[targetIndex].iValue = static_cast<uint32_t>(iterTarget->base_value.value);
                        }
                        else
                        {
                            pConstant[targetIndex].fValue = iterTarget->base_value.value;
                        }
                    }
                    else if (iter->type.value >= nw::g3d::tool::g3dif::elem_shader_param::type_float)
                    {
                        pConstant[targetIndex].fValue = iterTarget->base_value.value;
                    }
                    else
                    {
                        pConstant[targetIndex].iValue = static_cast<int>(iterTarget->base_value.value);
                    }

                    pConstant[targetIndex].targetOffset = iterTarget->component_index.value * sizeof(Bit32);
                    ++targetIndex;
                }
            }
        }
    }

    if( m_pElem->tex_pattern_anim_array )
    {
        matAnim.texturePatternAnimCount = static_cast<uint16_t>(m_pElem->tex_pattern_anim_array.Get().size());
        nn::g3d::ResTexturePatternAnimInfo* pPatAnimInfo = GetPtr<nn::g3d::ResTexturePatternAnimInfo>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_TexInfo].offset);
        pCtx->LinkPtr( &matAnim.pTexturePatternAnimInfoArray, pPatAnimInfo );

        nn::g3d::ResAnimConstantData* pResAnimConstant = GetPtr<nn::g3d::ResAnimConstantData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Constant].offset);

        uint16_t idxTarget = 0;
        uint16_t idxConstAnim = m_ShaderParamConstantCount;
        uint16_t idxCurve = m_ShaderParamCurveCount;
        for (auto iter = m_pElem->tex_pattern_anim_array.Get().cbegin();
            iter != m_pElem->tex_pattern_anim_array.Get().cend(); ++iter, ++idxTarget)
        {
            nn::g3d::ResTexturePatternAnimInfo& patAnimInfo = pPatAnimInfo[idxTarget];
            pCtx->LinkStr( &patAnimInfo.pName, iter->sampler_name.value.c_str() );
            patAnimInfo.subbindIndex	= -1;

            if (iter->curve)
            {
                patAnimInfo.constantIndex	= nn::g3d::AnimFlag_NotBound;
                patAnimInfo.curveIndex = idxCurve;
                ++idxCurve;
            }
            else
            {
                patAnimInfo.curveIndex		= nn::g3d::AnimFlag_NotBound;
                patAnimInfo.constantIndex	= idxConstAnim;
                pResAnimConstant[idxConstAnim].iValue = static_cast<int32_t>( iter->base_value.value );
                ++idxConstAnim;
            }
        }
    }

    matAnim.visibilityConstantIndex	= nn::g3d::AnimFlag_NotBound;
    matAnim.visibilityCurveIndex	= nn::g3d::AnimFlag_NotBound;
    if( m_pElem->material_visibility_anim )
    {
        if( m_pElem->material_visibility_anim.Get().curve )
        {
            matAnim.visibilityCurveIndex = m_ShaderParamCurveCount + m_TexPatternCurveCount;
        }
        else
        {
            matAnim.visibilityConstantIndex	= m_ShaderParamConstantCount + m_TexPatternConstantCount;
            nn::g3d::ResAnimConstantData* pResAnimConstant = GetPtr<nn::g3d::ResAnimConstantData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Constant].offset);
            pResAnimConstant[matAnim.visibilityConstantIndex].iValue = static_cast<int32_t>( m_pElem->material_visibility_anim->base_value.value );
        }
    }

    pCtx->LinkStr( &matAnim.pName, m_pElem->mat_name.value.c_str() );
    if( m_Chunk[ChunkType_Curve].size )
    {
        pCtx->LinkPtr( &matAnim.pCurveArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Curve].offset));
    }
    if( m_Chunk[ChunkType_Constant].size )
    {
        pCtx->LinkPtr( &matAnim.pConstantArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Constant].offset));
    }
}

}
}
