﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinMaterial.h>
#include <gl/GL.h>
#include <algorithm>
#include <util/UtilMath.h>

#include <GfxHelper.h>

#include <util/UtilError.h>

namespace nn {
namespace g3dTool {

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    shader_param, type, binary,
    nn::g3d::ResShaderParam::Type_Bool, nn::g3d::ResShaderParam::Type_Bool2, nn::g3d::ResShaderParam::Type_Bool3, nn::g3d::ResShaderParam::Type_Bool4,
    nn::g3d::ResShaderParam::Type_Int, nn::g3d::ResShaderParam::Type_Int2, nn::g3d::ResShaderParam::Type_Int3, nn::g3d::ResShaderParam::Type_Int4,
    nn::g3d::ResShaderParam::Type_Uint, nn::g3d::ResShaderParam::Type_Uint2, nn::g3d::ResShaderParam::Type_Uint3, nn::g3d::ResShaderParam::Type_Uint4,
    nn::g3d::ResShaderParam::Type_Float, nn::g3d::ResShaderParam::Type_Float2, nn::g3d::ResShaderParam::Type_Float3, nn::g3d::ResShaderParam::Type_Float4,
    nn::g3d::ResShaderParam::Type_Float2x2, nn::g3d::ResShaderParam::Type_Float2x3, nn::g3d::ResShaderParam::Type_Float2x4,
    nn::g3d::ResShaderParam::Type_Float3x2, nn::g3d::ResShaderParam::Type_Float3x3, nn::g3d::ResShaderParam::Type_Float3x4,
    nn::g3d::ResShaderParam::Type_Float4x2, nn::g3d::ResShaderParam::Type_Float4x3, nn::g3d::ResShaderParam::Type_Float4x4,
    nn::g3d::ResShaderParam::Type_Srt2d, nn::g3d::ResShaderParam::Type_Srt3d,
    nn::g3d::ResShaderParam::Type_Texsrt, nn::g3d::ResShaderParam::Type_Texsrt
    );

struct ShaderParamLess
{
public:
    bool operator()(const nw::g3d::tool::g3dif::elem_shader_param* lhs, const nw::g3d::tool::g3dif::elem_shader_param* rhs) const {
        return lhs->type.value < rhs->type.value;
    }
};

//--------------------------------------------------------------------------------------------------
void BinShaderAssign::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_shader_assign& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    m_DicAttribAssign.Build(pCtx, elem.attrib_assign_array.size());
    m_DicSamplerAssign.Build(pCtx, elem.sampler_assign_array.size());
    m_DicShaderOption.Build(pCtx, elem.shader_option_array.size());

    // attrib_assign
    int attribAssignIndex = 0;
    for (auto iter = elem.attrib_assign_array.cbegin();
        iter != elem.attrib_assign_array.cend();
        ++iter,
        ++attribAssignIndex)
    {
        m_DicAttribAssign.SetName(attribAssignIndex, iter->id.value);
    }

    // sampler_assign
    int samplerAssignIndex = 0;
    for (auto iter = elem.sampler_assign_array.cbegin();
        iter != elem.sampler_assign_array.cend();
        ++iter,
        ++samplerAssignIndex)
    {
        m_DicSamplerAssign.SetName(samplerAssignIndex, iter->id.value);		//!< shader 側 sampler 名
    }

    // shader_option
    int shaderOptionIndex = 0;
    for (auto iter = elem.shader_option_array.cbegin();
        iter != elem.shader_option_array.cend();
        ++iter,
        ++shaderOptionIndex)
    {
        m_DicShaderOption.SetName(shaderOptionIndex, iter->id.value);
    }

    pCtx->SetStr(elem.shading_model.value.c_str());
    pCtx->SetStr(elem.shader_archive.value.c_str());

    for (auto iter = elem.attrib_assign_array.cbegin();
        iter != elem.attrib_assign_array.cend();
        ++iter)
    {
        pCtx->SetStr(iter->id.value.c_str());
        pCtx->SetStr(iter->attrib_name.value.c_str());
    }

    for (auto iter = elem.sampler_assign_array.cbegin();
        iter != elem.sampler_assign_array.cend();
        ++iter)
    {
        pCtx->SetStr(iter->id.value.c_str());
        pCtx->SetStr(iter->sampler_name.value.c_str());
    }

    for (auto iter = elem.shader_option_array.cbegin();
        iter != elem.shader_option_array.cend();
        ++iter)
    {
        pCtx->SetStr(iter->id.value.c_str());
        pCtx->SetStr(iter->value.value.c_str());
    }
}

void BinShaderAssign::CalculateSize()
{
    m_Chunk[ ChunkType_ShaderAssign ].size = sizeof(nn::g3d::ResShaderAssignData);
    m_Chunk[ ChunkType_AttributeAssignStrPtrArray ].size = sizeof( nn::util::BinPtrToString ) * m_pElem->attrib_assign_array.size();
    m_Chunk[ ChunkType_SamplerAssignStrPtrArray ].size = sizeof( nn::util::BinPtrToString ) * m_pElem->sampler_assign_array.size();
    m_Chunk[ ChunkType_ShaderOptionStrPtrArray ].size = sizeof( nn::util::BinPtrToString ) * m_pElem->shader_option_array.size();

    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinShaderAssign::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResShaderAssignData& assign = *GetPtr<nn::g3d::ResShaderAssignData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    pCtx->LinkStr( &assign.pShadingModelName, nn::util::string_view( m_pElem->shading_model.value.c_str() ) );
    pCtx->LinkStr( &assign.pShaderArchiveName, nn::util::string_view( m_pElem->shader_archive.value.c_str() ) );
    assign.revision = static_cast<uint32_t>(m_pElem->revision.value);
    assign.attrAssignCount = static_cast<uint8_t>(m_pElem->attrib_assign_array.size());
    assign.samplerAssignCount = static_cast<uint8_t>(m_pElem->sampler_assign_array.size());
    assign.shaderOptionCount = static_cast<uint16_t>(m_pElem->shader_option_array.size());

    // AttribAssign 周辺の構築
    {
        m_DicAttribAssign.ConvertData(pCtx, assign.pAttrAssignDic, m_pElem->attrib_assign_array );

        nn::util::BinPtrToString* pAttribAssignStrPtrArray =
            GetPtr< nn::util::BinPtrToString >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_AttributeAssignStrPtrArray ].offset );
        pCtx->LinkPtr( &assign.pAttrAssignArray, pAttribAssignStrPtrArray );

        // attrib assign array の構築
        for( auto attribAssignIter = m_pElem->attrib_assign_array.begin(); attribAssignIter != m_pElem->attrib_assign_array.end(); ++attribAssignIter, ++pAttribAssignStrPtrArray )
        {
            pCtx->LinkStr( pAttribAssignStrPtrArray, nn::util::string_view( attribAssignIter->attrib_name.value.c_str() ) );
        }
    }

    // SamplerAssign 周辺の構築
    {
        m_DicSamplerAssign.ConvertData(pCtx, assign.pSamplerAssignDic, m_pElem->sampler_assign_array );

        nn::util::BinPtrToString* pSamplerAssignStrPtrArray =
            GetPtr< nn::util::BinPtrToString >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_SamplerAssignStrPtrArray ].offset );
        pCtx->LinkPtr( &assign.pSamplerAssignArray, pSamplerAssignStrPtrArray );

        for( auto samplerAssignIter = m_pElem->sampler_assign_array.begin(); samplerAssignIter != m_pElem->sampler_assign_array.end(); ++samplerAssignIter, ++pSamplerAssignStrPtrArray )
        {
            pCtx->LinkStr( pSamplerAssignStrPtrArray, nn::util::string_view( samplerAssignIter->sampler_name.value.c_str() ) );
        }
    }

    // shader option の構築
    {
        nn::util::ResDicData* pOptionDic =
            m_DicShaderOption.GetPtr<nn::util::ResDicData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
        pCtx->LinkPtr( &assign.pShaderOptionDic, pOptionDic );

        nn::util::BinPtrToString* pShaderOptionStrPtrArray =
            GetPtr< nn::util::BinPtrToString >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_ShaderOptionStrPtrArray ].offset );
        pCtx->LinkPtr( &assign.pShaderOptionArray, pShaderOptionStrPtrArray );

        for( auto shaderOptionIter = m_pElem->shader_option_array.begin(); shaderOptionIter != m_pElem->shader_option_array.end(); ++shaderOptionIter, ++pShaderOptionStrPtrArray )
        {
            pCtx->LinkStr( pShaderOptionStrPtrArray, nn::util::string_view( shaderOptionIter->value.value.c_str() ) );
        }
    }
}

//--------------------------------------------------------------------------------------------------

void BinMaterial::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_material& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    size_t shaderParamSize = 0;
    // 可変長要素の辞書をコンバート対象に追加する。
    if (elem.shader_assign)
    {
        m_ShaderAssign.SetParentBlock(this);
        m_ShaderAssign.Build(pCtx, elem.shader_assign.Get());

        for (auto iter = elem.shader_assign->shader_param_array.cbegin();
            iter != elem.shader_assign->shader_param_array.cend();
            ++iter)
        {
            m_ShaderParamArray.push_back(&(*iter));
        }

        // shader_param の順序を入れ替えます。
        std::sort(m_ShaderParamArray.begin(), m_ShaderParamArray.end(), ShaderParamLess());

        shaderParamSize = m_ShaderParamArray.size();

        m_DicRenderInfo.Build(pCtx, elem.shader_assign->render_info_array.size());
    }

    m_DicSampler.Build(pCtx, elem.sampler_array.size());
    m_DicShaderParam.Build(pCtx, shaderParamSize);
    m_DicUserData.Build(pCtx, elem.user_data_array.size());

    m_UserDataArray.resize(elem.user_data_array.size());
    SetParentBlockArray(m_UserDataArray, this);
    BuildArray(pCtx, m_UserDataArray, elem.user_data_array);

    // 文字列の登録。
    pCtx->SetStr(elem.name.value.c_str());

    // render_info
    if (elem.shader_assign)
    {
        int renderInfoIndex = 0;
        for (auto iter = elem.shader_assign->render_info_array.cbegin();
            iter != elem.shader_assign->render_info_array.end(); ++iter, ++renderInfoIndex)
        {
            m_DicRenderInfo.SetName(renderInfoIndex, iter->name.value);
            pCtx->SetStr(iter->name.value.c_str());

            for (auto iterRenderInfo = iter->stringArray.cbegin();
                iterRenderInfo != iter->stringArray.cend(); ++iterRenderInfo)
            {
                pCtx->SetStr(iterRenderInfo->c_str());
            }
        }
    }

    // texture_ref
    int textureRefIndex = 0;
    for (auto iter = elem.sampler_array.cbegin();
        iter != elem.sampler_array.cend(); ++iter, ++textureRefIndex)
    {
        pCtx->SetStr(iter->tex_name.value.c_str());
    }

    // sampler
    int samplerIndex = 0;
    for (auto iter = elem.sampler_array.cbegin();
        iter != elem.sampler_array.cend(); ++iter, ++samplerIndex)
    {
        m_DicSampler.SetName(samplerIndex, iter->name.value);
        pCtx->SetStr(iter->name.value.c_str());
    }

    if (m_pElem->shader_assign)
    {
        // shader_param
        int shaderParamIndex = 0;
        for (auto iter = m_ShaderParamArray.cbegin();
            iter != m_ShaderParamArray.cend(); ++iter, ++shaderParamIndex)
        {
            pCtx->SetStr((*iter)->id.value.c_str());
            m_DicShaderParam.SetName(shaderParamIndex, (*iter)->id.value);
        }
    }

    // user_data
    int useDataIndex = 0;
    for (auto iter = elem.user_data_array.cbegin();
        iter != elem.user_data_array.cend(); ++iter, ++useDataIndex)
    {
        m_DicUserData.SetName(useDataIndex, iter->name.value);
    }
}

void BinMaterial::CalculateSamplerArraySize()
{
    // gfx::Sampler の最大サイズの計算、ApiVariation の中から最大の物を選択する。
    m_Chunk[ ChunkType_Sampler ].size = GetGfxMaxImplDataSize<nn::gfx::SamplerImplData>() * m_pElem->sampler_array.size() +
                            sizeof( nn::gfx::SamplerInfoData ) * m_pElem->sampler_array.size();
}

void BinMaterial::CalculateSize()
{
    // Heap for render info block.
    size_t renderInfoSize = 0;
    if( m_pElem->shader_assign )
    {
        for ( int i = 0; i < static_cast<int>( m_pElem->shader_assign->render_info_array.size() ); ++i )
        {
            renderInfoSize += sizeof( nn::g3d::ResRenderInfoData );

            const nw::g3d::tool::g3dif::elem_render_info& renderInfo = m_pElem->shader_assign->render_info_array[ i ];
            if ( renderInfo.count.value > 0 )
            {
                if ( renderInfo.type.value == nw::g3d::tool::g3dif::elem_render_info::type_int ||
                        renderInfo.type.value == nw::g3d::tool::g3dif::elem_render_info::type_float )
                {
                    // int or float を格納する配列のサイズ
                    renderInfoSize += renderInfo.dataSize;
                }
                else
                {
                    // BinString のポインタ配列のサイズ
                    renderInfoSize += sizeof( nn::util::BinPtrToString ) * renderInfo.count.value;
                }
            }
        }
        renderInfoSize = nn::util::align_up( renderInfoSize, 8 );
    }

    m_Chunk[ChunkType_RenderInfo].size = renderInfoSize;

    m_Chunk[ ChunkType_GfxTextureViewPtrArray ].size = sizeof( nn::util::BinTPtr<const nn::gfx::TextureView> ) * m_pElem->sampler_array.size();

    CalculateSamplerArraySize();


    size_t shaderParamSize = 0;

    // MatBuffer のデータ。
    m_SrcParamSize = 0;

    // パラメータ依存関係の解決。
    const int numShaderParam = static_cast<int>(m_ShaderParamArray.size());
    m_ShaderParamDepends.reserve(numShaderParam);
    for (int idx = 0; idx < numShaderParam; ++idx)
    {
        ShaderParamDepend init = { -1, -1 };
        m_ShaderParamDepends.push_back(init);
    }
    for (int idxShaderParam = 0; idxShaderParam < numShaderParam; ++idxShaderParam)
    {
        if (!m_ShaderParamArray[idxShaderParam]->depend.value.empty())
        {
            for (int idxDepend = 0; idxDepend < numShaderParam; ++idxDepend)
            {
                if (m_ShaderParamArray[idxShaderParam]->depend.value ==
                    m_ShaderParamArray[idxDepend]->id.value)
                {
                    m_ShaderParamDepends[idxShaderParam].depend = idxDepend;
                    if (idxShaderParam != idxDepend)
                    {
                        if (m_ShaderParamDepends[idxDepend].depended > 0)
                        {
                            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_UNSUPPORTED, "Identifier_OverlapDependency", m_ShaderParamArray[idxDepend]->id.value.c_str());
                        }
                        m_ShaderParamDepends[idxDepend].depended = idxShaderParam;
                    }
                }
            }
        }
    }

    if ( m_pElem->shader_assign )
    {
        shaderParamSize = sizeof(nn::g3d::ResShaderParamData) * m_ShaderParamArray.size();
        m_MaterialBuffer.reserve(m_ShaderParamArray.size());
        int index = 0;
        for (auto iter = m_ShaderParamArray.cbegin();
            iter != m_ShaderParamArray.cend();
            ++iter, ++index)
        {
            ShaderParameterSize size;
            size.srcOffset = m_SrcParamSize;

            size.srcSize = nw::g3d::tool::g3dif::elem_shader_param::GetSrcParamSize((*iter)->type.value);
            if (m_ShaderParamDepends[index].depend >= 0)
            {
                size.srcSize += sizeof(nn::Bit64);
            }

            // バッファのサイズとバッファ先頭からのオフセットを格納する。
            m_MaterialBuffer.push_back(size);
            m_SrcParamSize += size.srcSize;
        }
    }

    m_Chunk[ ChunkType_MatShaderParam ].size = shaderParamSize;
    m_Chunk[ ChunkType_MatBuff ].size = m_SrcParamSize;

    m_Chunk[ ChunkType_VolatileFlag ].size = nw::g3d::tool::util::Align( m_ShaderParamArray.size(), 32 ) >> 3;
    m_Chunk[ ChunkType_UserDataArray ].size = sizeof( nn::gfx::ResUserDataData ) * m_UserDataArray.size();

    m_Chunk[ ChunkType_TextureDescriptorSlotArray ].size = sizeof( nn::gfx::DescriptorSlotData ) * m_pElem->sampler_array.size();
    m_Chunk[ ChunkType_SamplerDescriptorSlotArray ].size = sizeof( nn::gfx::DescriptorSlotData ) * m_pElem->sampler_array.size();

    m_Chunk[ ChunkType_TextureNamePtrArray ].size = sizeof( nn::util::BinPtrToString ) * m_pElem->sampler_array.size();

    // アドレスを知っていればここでパディングの追加が可能。
    SetBlockSize( Context::MemBlockType_Main, CalcChunk( m_Chunk, ChunkType_Count ));
}

void BinMaterial::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    // UserData Array のオフセット計算
    ptrdiff_t ofsUserDataArray = GetBlockOffset( Context::MemBlockType_Main ) + m_Chunk[ ChunkType_UserDataArray ].offset;
    for ( auto iter = m_UserDataArray.begin(); iter != m_UserDataArray.end(); ++iter )
    {
        iter->SetStructOffset( ofsUserDataArray );
        ofsUserDataArray += sizeof(nn::gfx::ResUserDataData);
    }
}

void BinMaterial::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResMaterialData& material = *GetPtr<nn::g3d::ResMaterialData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    material.blockHeader.signature.SetPacked( nn::g3d::ResMaterial::Signature );
    pCtx->AddBinBlockHeader( &material.blockHeader );
    pCtx->LinkStr( &material.pName, nn::util::string_view( m_pElem->name.value.c_str() ) );

    uint32_t flag = 0;
    flag |= m_pElem->material_info.visibility.value ? nn::g3d::ResMaterial::Flag_Visibility : 0;
    material.flag = flag;

    material.index = static_cast<uint16_t>(m_Index);
    material.textureCount = static_cast<uint8_t>(m_pElem->sampler_array.size());

    if( material.textureCount != 0 )
    {
        pCtx->LinkPtr( &material.pTextureArray,
            GetPtr( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_GfxTextureViewPtrArray ].offset ) );

        nn::util::BinPtrToString* pTextureNameArray =
            GetPtr<nn::util::BinPtrToString>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_TextureNamePtrArray ].offset );
        pCtx->LinkPtr( &material.pTextureNameArray, pTextureNameArray );
        for( auto iter = m_pElem->sampler_array.cbegin(); iter != m_pElem->sampler_array.cend(); ++iter, ++pTextureNameArray )
        {
            pCtx->LinkStr( pTextureNameArray, nn::util::string_view( iter->tex_name.value.data() ) );
        }
    }

    ConvertRenderInfo( material, pCtx );
    ConvertGfxSampler( material, pCtx );
    ConvertShaderParam( material, pCtx );

    material.pUserPtr.Set( nullptr );		//!< NULL を入れているので Link はしない。

    if (m_pElem->shader_assign)
    {
        pCtx->LinkPtr( &material.pShaderAssign, m_ShaderAssign.GetPtr( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) ) );
    }
    else
    {
        material.pShaderAssign.Set(nullptr);
    }

    // UserData
    material.userDataCount = static_cast<uint16_t>(m_UserDataArray.size());
    void* pUserDataArray = GetPtr< void >( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_UserDataArray ].offset );
    pCtx->LinkPtr( &material.pUserDataArray, pUserDataArray );
    m_DicUserData.ConvertData(pCtx, material.pUserDataDic, m_UserDataArray);

    if( m_pElem->sampler_array.size() != 0 )
    {
        nn::util::BytePtr offsetToSamplerDescriptorSlotArray( GetPtr<void>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_SamplerDescriptorSlotArray ].offset ) );
        if( !nn::util::is_aligned( reinterpret_cast<size_t>( offsetToSamplerDescriptorSlotArray.Get() ), 8 ) )
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_INTERNAL, "Identifier_InvalidAddressAlignment", "ResMaterialData pSamplerSlotArray", offsetToSamplerDescriptorSlotArray.Get(), 8 );
        }
        nn::util::BytePtr offsetToTextureDescriptorSlotArray( GetPtr<void>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_TextureDescriptorSlotArray ].offset ) );
        if( !nn::util::is_aligned( reinterpret_cast<size_t>( offsetToTextureDescriptorSlotArray.Get() ), 8 ) )
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_INTERNAL, "Identifier_InvalidAddressAlignment", " ResMaterialData pTextureSlotArray", offsetToTextureDescriptorSlotArray.Get(), 8 );
        }
        // DescriptorSlotを初期化しておく。無効値で初期化される。
        nn::gfx::DescriptorSlotData* pSamplerDescriptorSlotDataArray = offsetToSamplerDescriptorSlotArray.Get<nn::gfx::DescriptorSlotData>();
        nn::gfx::DescriptorSlotData* pTextureDescriptorSlotDataArray = offsetToTextureDescriptorSlotArray.Get<nn::gfx::DescriptorSlotData>();
        for (int idxTex = 0; idxTex < material.textureCount; ++idxTex)
        {
            new(&pSamplerDescriptorSlotDataArray[idxTex]) nn::gfx::DescriptorSlot();
            new(&pTextureDescriptorSlotDataArray[idxTex]) nn::gfx::DescriptorSlot();
        }
        pCtx->LinkPtr( &material.pSamplerSlotArray,
            GetPtr<int32_t>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_SamplerDescriptorSlotArray ].offset ) );
        pCtx->LinkPtr( &material.pTextureSlotArray,
            GetPtr<int32_t>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_TextureDescriptorSlotArray ].offset ) );
    }
}

void BinMaterial::ConvertRenderInfo(nn::g3d::ResMaterialData &material, std::shared_ptr<Context> pCtx )
{
    // RenderInfo に対するヒープ構造
    //		RenderInfoDataArray		<- base = GetPtr<nn::g3d::ResRenderInfoData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_RenderInfo].offset);
    //		BinPtrToSTring Heap		<- heap offset

    if (m_pElem->shader_assign)
    {
        int numRenderInfo = static_cast<int>( m_pElem->shader_assign->render_info_array.size() );
        material.renderInfoCount = static_cast<uint16_t>(numRenderInfo);

        nn::g3d::ResRenderInfoData* pRenderInfo = GetPtr<nn::g3d::ResRenderInfoData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_RenderInfo].offset);

        // RenderInfoData 配列へのポインタをリンクします。
        pCtx->LinkPtr( &material.pRenderInfoArray, pRenderInfo );

        // 辞書のポインタをリンクします。
        nn::util::ResDicData* pDic = m_DicRenderInfo.GetPtr<nn::util::ResDicData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
        pCtx->LinkPtr( &material.pRenderInfoDic, pDic );


        // 文字列ポインタの配列、int/float 配列のデータを埋めていきます。
        // dataPool は下記のメモリ構造になります。
        // BinPtr array		8byte align needed
        // int32_t array	4byte align needed(将来的に int64_t を使うかもしれないので BinPtr の次に配置する）
        // float array		4byte align needed
        size_t	binPtrPoolSize = 0;
        size_t	intPoolSize = 0;
        size_t	floatPoolSize = 0;
        for (auto iter = m_pElem->shader_assign->render_info_array.cbegin();
            iter != m_pElem->shader_assign->render_info_array.cend();
            ++iter )
        {
            if ( iter->count.value > 0 )
            {
                if ( iter->type.value == nw::g3d::tool::g3dif::elem_render_info::type_int )
                {
                    intPoolSize += iter->dataSize;
                }
                else if ( iter->type.value == nw::g3d::tool::g3dif::elem_render_info::type_float )
                {
                    floatPoolSize += iter->dataSize;
                }
                else
                {
                    binPtrPoolSize += iter->count.value * sizeof( nn::util::BinPtrToString );
                }
            }
        }
        if( !nn::util::is_aligned( reinterpret_cast<size_t>( pRenderInfo ), 8 ) )
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_INTERNAL, "Identifier_InvalidAddressAlignment", "ResRenderInfoData", pRenderInfo, 8 );
        }
        nn::util::BytePtr offsetToBinPtrPool( pRenderInfo, static_cast<ptrdiff_t>( sizeof( nn::g3d::ResRenderInfoData ) * numRenderInfo ) );
        if( !nn::util::is_aligned( reinterpret_cast<size_t>( offsetToBinPtrPool.Get() ), 8 ) )
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_INTERNAL, "Identifier_InvalidAddressAlignment", "ResRenderInfoData BinPtrPool", offsetToBinPtrPool.Get(), 8 );
        }
        nn::util::BytePtr offsetToIntDataPool( offsetToBinPtrPool.Get(), static_cast<ptrdiff_t>( binPtrPoolSize ) );
        if( !nn::util::is_aligned( reinterpret_cast<size_t>( offsetToIntDataPool.Get() ), 4 ) )
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_INTERNAL, "Identifier_InvalidAddressAlignment", "ResRenderInfoData of IntDataPool", offsetToIntDataPool.Get(), 4 );
        }
        nn::util::BytePtr offsetToFloatDataPool( offsetToIntDataPool.Get(), static_cast<ptrdiff_t>( intPoolSize ) );
        if( !nn::util::is_aligned( reinterpret_cast<size_t>( offsetToFloatDataPool.Get() ), 4 ) )
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_INTERNAL, "Identifier_InvalidAddressAlignment", "ResRenderInfoData of FloatDataPool", offsetToFloatDataPool.Get(), 4 );
        }
        int index = 0;
        for (auto iter = m_pElem->shader_assign->render_info_array.cbegin(); iter != m_pElem->shader_assign->render_info_array.cend();
            ++iter, ++index, pRenderInfo++ )
        {
            pRenderInfo->arrayLength	= static_cast<uint16_t>( iter->count.value );
            pRenderInfo->type			= static_cast<uint8_t>( iter->type.value );
            pCtx->LinkStr( &pRenderInfo->pName, nn::util::string_view( iter->name.value.c_str() ) );

            if ( pRenderInfo->arrayLength > 0 )
            {
                if ( iter->type.value == nw::g3d::tool::g3dif::elem_render_info::type_int )
                {
                    pCtx->LinkPtr( &pRenderInfo->intValueArray, offsetToIntDataPool.Get< void >() );
                    memcpy( offsetToIntDataPool.Get< void >(), iter->rawdata.get(), iter->dataSize );
                    offsetToIntDataPool += static_cast< ptrdiff_t >( iter->dataSize );
                }
                else if ( iter->type.value == nw::g3d::tool::g3dif::elem_render_info::type_float )
                {
                    pCtx->LinkPtr( &pRenderInfo->floatValueArray, offsetToFloatDataPool.Get< void >() );
                    memcpy( offsetToFloatDataPool.Get< void >(), iter->rawdata.get(), iter->dataSize );
                    offsetToFloatDataPool += static_cast< ptrdiff_t >( iter->dataSize );
                }
                else
                {
                    pCtx->LinkPtr( &pRenderInfo->stringArray, offsetToBinPtrPool.Get() );

                    // 文字列ポインタ配列に参照先を書き込む
                    nn::util::BinPtrToString*	pStr = offsetToBinPtrPool.Get< nn::util::BinPtrToString >();
                    for (int i = 0; i < iter->count.value; ++i, ++pStr )
                    {
                        pCtx->LinkStr( pStr, nn::util::string_view( iter->stringArray[ i ].c_str() ) );
                    }
                    offsetToBinPtrPool += static_cast< ptrdiff_t >( iter->count.value * sizeof( nn::util::BinTPtr<nn::util::BinPtrToString> ) );
                }
            }
            else
            {
                pRenderInfo->stringArray.Set( nullptr );
            }
        }
    }
    else
    {
        material.renderInfoCount = 0;
        material.pRenderInfoDic.Set(nullptr);
    }
}

void BinMaterial::ConvertGfxSampler( nn::g3d::ResMaterialData &material, std::shared_ptr<Context> pCtx )
{
    // Sampler に対するヒープ構造：
    //		SamplerInfoData
    //		Sampler heap
    int gfxSamplerCount = static_cast< int >( m_pElem->sampler_array.size() );
    material.samplerCount = static_cast< uint8_t >( gfxSamplerCount );

    nn::gfx::SamplerInfoData* pSamplerInfoData =
        GetPtr< nn::gfx::SamplerInfoData > (pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_Sampler ].offset );

    // GfxSamplerInfoDataArray のリンク
    pCtx->LinkPtr( &material.pSamplerInfoArray, pSamplerInfoData );

    // GfxSamplerHeap のリンク
    nn::util::BytePtr gfxSamplerHeap( pSamplerInfoData, static_cast< size_t >( gfxSamplerCount * sizeof( nn::gfx::SamplerInfoData ) ) );
    pCtx->LinkPtr( &material.pSamplerArray, gfxSamplerHeap.Get() );

    // 辞書のポインタをリンクします。
    nn::util::ResDicData* pDic = m_DicSampler.GetPtr< nn::util::ResDicData >( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );
    pCtx->LinkPtr( &material.pSamplerInfoDic, pDic );

    for( auto iter = m_pElem->sampler_array.begin(); iter != m_pElem->sampler_array.end(); ++iter, ++pSamplerInfoData )
    {
        nn::gfx::SamplerInfo* gfxSampler = nn::gfx::DataToAccessor< nn::gfx::SamplerInfoData >( *pSamplerInfoData );
        gfxSampler->SetDefault();
        gfxSampler->SetAddressU( GetGfxTextureAddressMode( iter->wrap.u.value ) );
        gfxSampler->SetAddressV( GetGfxTextureAddressMode( iter->wrap.v.value ) );
        gfxSampler->SetAddressW( GetGfxTextureAddressMode( iter->wrap.w.value ) );
        // pSamplerInfoData->borderColorType =		// TODO: 現状の中間ファイルフォーマットでは指定できないのでデフォルト値にしておく。
        // pSamplerInfoData->comparisonFunction =	// TODO: レンダーステートから引いてくる？
        gfxSampler->SetFilterMode( static_cast<nn::gfx::FilterMode>( GetGfxFilterMode( iter->filter ) ) );
        gfxSampler->SetLodBias( GetGfxLodBias( iter->lod.bias ) );
        gfxSampler->SetMaxAnisotropy( GetGfxMaxAnisotropy( iter->filter.max_aniso.value ) );
        gfxSampler->SetMaxLod( GetGfxLodValue( iter->lod.max ) );
        gfxSampler->SetMinLod( GetGfxLodValue( iter->lod.min ) );
#if defined( GFX_VAL_FORCE_DEFAULT )
        gfxSampler->SetDefault();
#endif
    }
}

void BinMaterial::ConvertShaderParam( nn::g3d::ResMaterialData &material, std::shared_ptr<Context> pCtx )
{
    material.shaderParamCount = 0;
    material.pShaderParamArray.Set(nullptr);
    material.srcParamSize = 0;
    material.rawParamSize = 0;
    material.pSrcParam.Set(nullptr);
    material.pShaderParamDic.Set(nullptr);
    material.shaderParamVolatileCount = 0;

    if (m_pElem->shader_assign)
    {
        material.shaderParamCount = static_cast<uint16_t>(m_ShaderParamArray.size());
        nn::g3d::ResShaderParamData* pShaderParam =
            GetPtr<nn::g3d::ResShaderParamData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_MatShaderParam].offset);
        pCtx->LinkPtr( &material.pShaderParamArray, pShaderParam );
        uint8_t* pMaterialBuffer =
            GetPtr<uint8_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_MatBuff].offset);
        material.srcParamSize = static_cast<uint16_t>(m_SrcParamSize);
        pCtx->LinkPtr( &material.pSrcParam, pMaterialBuffer );
        m_DicShaderParam.ConvertArrayData( pCtx, material.pShaderParamDic, pShaderParam, material.shaderParamCount );

        int index = 0;
        for (auto iter = m_ShaderParamArray.cbegin();
            iter != m_ShaderParamArray.cend();
            ++iter, ++ pShaderParam, ++index)
        {
            if ((*iter)->type.value == nw::g3d::tool::g3dif::elem_shader_param::type_texsrt_ex)
            {
                THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_UNSUPPORTED, "Identifier_AbolishTexsrtEx", (*iter)->id.value.c_str());
            }
            pShaderParam->type = static_cast<uint8_t>(ToEnum_binary((*iter)->type.value));
            pShaderParam->offset = -1;
            pShaderParam->srcOffset = static_cast<uint16_t>(m_MaterialBuffer[index].srcOffset);
            pCtx->LinkStr( &pShaderParam->pId, nn::util::string_view( (*iter)->id.value.c_str() ) );
            //pShaderParam->pCallback.Set(nullptr);
            pShaderParam->dependedIndex = static_cast<uint16_t>(m_ShaderParamDepends[index].depended < 0
                ? index : m_ShaderParamDepends[index].depended);
            pShaderParam->dependIndex = static_cast<uint16_t>(m_ShaderParamDepends[index].depend < 0
                ? index : m_ShaderParamDepends[index].depend);
            pShaderParam->srcSize = static_cast<uint8_t>(m_MaterialBuffer[index].srcSize);

            const void* pSrc = (*iter)->rawdata.get();
            void* pDst = pMaterialBuffer + m_MaterialBuffer[index].srcOffset;
            size_t formatSize = m_MaterialBuffer[index].srcSize;
            size_t dataSize = (*iter)->dataSize;
            size_t padSize = formatSize - dataSize;
            std::memcpy(pDst, pSrc, dataSize);

            nw::g3d::tool::g3dif::elem_shader_param::ValueToSrcParam(pDst, (*iter)->type.value);

            // 依存先ポインタの分を調整します。
            if (m_ShaderParamDepends[index].depend >= 0)
            {
                padSize -= sizeof(Bit64);
            }

            // padSize 必ずは 0 になるように調整します。
            if (padSize != 0)
            {
                DumpLogRecursively();
                THROW_TRANSLATED_ERROR_INTERNAL(ERRCODE_INTERNAL, "Identifier_InternalError");
            }
        }
    }

    // VolatileFlag
    Bit32 *pVolatileFlags = GetPtr<Bit32>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_VolatileFlag].offset);
    pCtx->LinkPtr( &material.pVolatileFlag, pVolatileFlags );
    memset( pVolatileFlags, 0, nw::g3d::tool::util::Align( m_ShaderParamArray.size(), 32) >> 3 );
}

}
}
