﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Arithmetic.h>
#include <BinAnimCurve.h>
#include <util/UtilMath.h>
#include <util/UtilError.h>

namespace nn {
namespace g3dTool {

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    anim_curve, frame_type, binary,
    nn::g3d::ResAnimCurve::FrameType_Quant32,
    nn::g3d::ResAnimCurve::FrameType_Quant32,
    nn::g3d::ResAnimCurve::FrameType_Quant16,
    nn::g3d::ResAnimCurve::FrameType_Quant8
);

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    anim_curve, key_type, binary,
    nn::g3d::ResAnimCurve::KeyType_Quant32,
    nn::g3d::ResAnimCurve::KeyType_Quant32,
    nn::g3d::ResAnimCurve::KeyType_Quant16,
    nn::g3d::ResAnimCurve::KeyType_Quant8,
    nn::g3d::ResAnimCurve::KeyType_Quant32
);

NN_G3D_TOOL_BIN_DEFINE_ENUM_TABLE(
    anim_curve, wrap, binary,
    nn::g3d::ResAnimCurve::WrapMode_Clamp,
    nn::g3d::ResAnimCurve::WrapMode_Repeat,
    nn::g3d::ResAnimCurve::WrapMode_Mirror,
    nn::g3d::ResAnimCurve::WrapMode_RelativeRepeat
    );

void BinAnimCurve::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_anim_curve& elem)
{
    pCtx->blocks.push_back(this);
    m_pElem = &elem;

    // 同一フレームに複数キーがある場合を除去しながら展開する。
    static const int tblStride[] = { 4, 2, 2 };
    static const int tblComp[] = { 7, 4, 2 };	// hermit, linear, step に対応
    int stride = tblStride[elem.curve_type];
    int comp = tblComp[elem.curve_type];
    int size = elem.count.value;
    // TODO: size == 1 はエラー。
    m_FVSPairArray.reserve(size);

    float* pKey = static_cast<float*>(elem.stream.rawdata.get());
    for (int index = 0; index < size - 1; ++index, pKey += stride)
    {
        float frame = *pKey;
        float nextFrame = *(pKey + stride);
        if (frame == nextFrame)
        {
            continue;
        }

        FVSPair pair;
        memcpy(&pair, pKey, sizeof(float) * comp); // comp の先は不定値。
        m_FVSPairArray.push_back(pair);
    }
    // 最終フレームは必ず使用。
    FVSPair pair;
    memcpy(&pair, pKey, sizeof(float) * stride);
    memcpy(&pair.f[stride], &pair.f[0], sizeof(float) * (comp - stride)); // 次フレーム分はコピーで埋める。
    m_FVSPairArray.push_back(pair);
}

void BinAnimCurve::CalculateSize()
{
    // TODO: ベイク対応
    static const int typeSize[] = { 4, 4, 2, 1 };	// f32, f32, s16, u8	に対応している
    static const int compSize[] = { 4, 2, 1 };		// hermit, linear, step に対応している
    auto size =  m_FVSPairArray.size();
    m_Chunk[ChunkType_Frame].size = nw::g3d::tool::util::Align(typeSize[m_pElem->frame_type.value] * size);
    if (m_pElem->key_type.value == nw::g3d::tool::g3dif::elem_anim_curve::key1)
    {
        // TODO: bool であることのチェック。
        m_Chunk[ChunkType_Key].size = nw::g3d::tool::util::Align(size, ALIGNMENT_FLAG) >> 3;
    }
    else if (m_pElem->curve_type == nw::g3d::tool::g3dif::elem_anim_curve::step && m_Type == FLOAT)
    {
        // リニアに変換される。
        m_Chunk[ChunkType_Key].size = nw::g3d::tool::util::Align(typeSize[m_pElem->key_type.value] *
            compSize[nw::g3d::tool::g3dif::elem_anim_curve::linear] * size);
    }
    else
    {
        m_Chunk[ChunkType_Key].size = nw::g3d::tool::util::Align(typeSize[m_pElem->key_type.value] * compSize[m_pElem->curve_type] * size);
    }
    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count));
}

void BinAnimCurve::Convert( std::shared_ptr<Context> pCtx )
{
    nn::g3d::ResAnimCurveData& curve = *GetPtr<nn::g3d::ResAnimCurveData>(pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    uint16_t flag = 0;
    flag |= ToEnum_binary(m_pElem->frame_type.value);
    flag |= ToEnum_binary(m_pElem->key_type.value);
    flag |= (ToEnum_binary(m_pElem->pre_wrap.value) << nn::g3d::ResAnimCurve::Shift_PreWrapMode);
    flag |= (ToEnum_binary(m_pElem->post_wrap.value) << nn::g3d::ResAnimCurve::Shift_PostWrapMode);
    curve.flag = flag;

    curve.keyCount = static_cast<uint16_t>(m_FVSPairArray.size());
    curve.targetOffset = static_cast<uint32_t>(m_TargetOffset);

    QuantizeFrame(pCtx, curve);

    switch (m_pElem->curve_type)
    {
    case nw::g3d::tool::g3dif::elem_anim_curve::hermite:
        QuantizeHermite(pCtx, curve);
        break;
    case nw::g3d::tool::g3dif::elem_anim_curve::linear:
        QuantizeLinear(pCtx, curve);
        break;
    case nw::g3d::tool::g3dif::elem_anim_curve::step:
        switch (m_Type)
        {
        case FLOAT:
            QuantizeStepFloat(pCtx, curve);
            break;
        case INT:
            QuantizeStepInt(pCtx, curve);
            break;
        case BOOL:
            QuantizeStepBool(pCtx, curve);
            break;
        default:
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_UNSUPPORTED, "Identifier_UnsupportedQuantizeType");
        }
        break;
    default:
        THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_UNSUPPORTED, "Identifier_UnsupportedQuantizeType");
    }

    void* pKey = GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Key].offset);
    pCtx->LinkPtr( &curve.pKeyArray, pKey );
}

void BinAnimCurve::QuantizeFrame(std::shared_ptr<Context> pCtx, nn::g3d::ResAnimCurveData& curve)
{
    size_t frameSize = 0;

    if (m_pElem->frame_type.value <= nw::g3d::tool::g3dif::elem_anim_curve::frame32)
    {
        int dstLastIndex = 0;
        float* pDst = GetPtr< float >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Frame].offset);
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst, ++dstLastIndex)
        {
            *pDst = iter->f[0];
        }

        pDst = GetPtr< float >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Frame].offset);
        curve.startFrame = pDst[0];
        curve.endFrame = pDst[dstLastIndex-1];
    }
    else if (m_pElem->frame_type.value == nw::g3d::tool::g3dif::elem_anim_curve::frame16)
    {
        int dstLastIndex = 0;
        int16_t* pDst = GetPtr< int16_t >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Frame].offset);
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst, ++dstLastIndex)
        {
            float frame = iter->f[0];
            frame = nn::g3d::Math::Clamp(frame, -1024.5f, 1024.4f);

            // F32 浮動小数点数を S10.5 固定小数点数に変換
            *pDst = nn::g3d::Math::Round<int16_t>(iter->f[0] * 32.0f);
            frameSize += sizeof(int16_t);
        }

        size_t padding = m_Chunk[ChunkType_Frame].size - frameSize;
        memset(pDst, 0, padding);

        pDst = GetPtr< int16_t >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Frame].offset);
        curve.startFrame = static_cast<float>(pDst[0]) * (1.0f / 32.0f);
        curve.endFrame = static_cast<float>(pDst[dstLastIndex-1]) * (1.0f / 32.0f);
    }
    else if (m_pElem->frame_type.value == nw::g3d::tool::g3dif::elem_anim_curve::frame8)
    {
        int dstLastIndex = 0;
        uint8_t* pDst = GetPtr< uint8_t >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Frame].offset);
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst, ++dstLastIndex)
        {
            float frame = iter->f[0];
            frame = nn::g3d::Math::Clamp(frame, -0.5f, 255.4f);

            *pDst = nn::g3d::Math::Round<uint8_t>(frame);
            frameSize += sizeof(uint8_t);
        }

        size_t padding = m_Chunk[ChunkType_Frame].size - frameSize;
        memset(pDst, 0, padding);

        pDst = GetPtr< uint8_t >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Frame].offset);
        curve.startFrame = pDst[0];
        curve.endFrame = pDst[dstLastIndex-1];
    }
    else
    {
        THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_UNSUPPORTED, "Identifier_UnsupportedQuantizeType");
    }

    pCtx->LinkPtr( &curve.pFrameArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Frame].offset) );
}

void BinAnimCurve::QuantizeHermite(std::shared_ptr<Context> pCtx, nn::g3d::ResAnimCurveData& curve)
{
    curve.flag |= nn::g3d::ResAnimCurve::CurveType_Cubic;

    float coef[4];
    curve.fScale = m_pElem->scale.value;
    curve.fOffset = m_pElem->offset.value;
    curve.fDelta = m_FVSPairArray.back().f[1] - m_FVSPairArray.front().f[1];
    if (m_IsDegreeValue)
    {
        curve.fDelta = nn::util::DegreeToRadian(curve.fDelta);
    }

    struct Coef32 { float coef[4]; };
    struct Coef16 { int16_t coef[4]; };
    struct Coef8 { int8_t coef[4]; };

    void * pDst = GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Key].offset);
    for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter)
    {
        float f = iter->hermite.frame1 - iter->hermite.frame0;
        float v0 = iter->hermite.value0;
        float v1 = iter->hermite.value1;
        float so = iter->hermite.out_slope0;
        float si = iter->hermite.in_slope1;
        if (m_IsDegreeValue)
        {
            v0 = nn::util::DegreeToRadian(v0);
            v1 = nn::util::DegreeToRadian(v1);
            so = nn::util::DegreeToRadian(so);
            si = nn::util::DegreeToRadian(si);
        }
        float v = v1 - v0;
        coef[3] = -2 * v + (so + si) * f;
        coef[2] = 3 * v - (2 * so + si) * f;
        coef[1] = so * f;
        coef[0] = v0;

        if (m_pElem->key_type.value <= nw::g3d::tool::g3dif::elem_anim_curve::key32)
        {
            Coef32* coefDst = static_cast<Coef32*>(pDst);
            coefDst->coef[3] = coef[3];
            coefDst->coef[2] = coef[2];
            coefDst->coef[1] = coef[1];
            coefDst->coef[0] = coef[0];
            pDst = nn::g3d::AddOffset(pDst, sizeof(Coef32));
        }
        else if (m_pElem->key_type.value == nw::g3d::tool::g3dif::elem_anim_curve::key16)
        {
            const float maxValue = static_cast<float>(std::numeric_limits<int16_t>::max());
            const float minValue = static_cast<float>(std::numeric_limits<int16_t>::lowest());

            Coef16* coefDst = static_cast<Coef16*>(pDst);

            // fScale に丸め誤差が発生して結果が int16_t の範囲外に出てしまう可能性があるために clamp 処理をします。
            coefDst->coef[3] = nn::g3d::Math::Round<int16_t>(nn::g3d::Math::Clamp(coef[3] / curve.fScale, minValue, maxValue));
            coefDst->coef[2] = nn::g3d::Math::Round<int16_t>(nn::g3d::Math::Clamp(coef[2] / curve.fScale, minValue, maxValue));
            coefDst->coef[1] = nn::g3d::Math::Round<int16_t>(nn::g3d::Math::Clamp(coef[1] / curve.fScale, minValue, maxValue));
            coefDst->coef[0] = nn::g3d::Math::Round<int16_t>(nn::g3d::Math::Clamp((coef[0] - curve.fOffset) / curve.fScale, minValue, maxValue));

            pDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(Coef16));
        }
        else if (m_pElem->key_type.value == nw::g3d::tool::g3dif::elem_anim_curve::key8)
        {
            const float maxValue = static_cast<float>(std::numeric_limits<int8_t>::max());
            const float minValue = static_cast<float>(std::numeric_limits<int8_t>::lowest());

            Coef8* coefDst = static_cast<Coef8*>(pDst);

            // fScale に丸め誤差が発生して結果が int16_t の範囲外に出てしまう可能性があるために clamp 処理をします。
            coefDst->coef[3] = nn::g3d::Math::Round<int8_t>(nn::g3d::Math::Clamp(coef[3] / curve.fScale, minValue, maxValue));
            coefDst->coef[2] = nn::g3d::Math::Round<int8_t>(nn::g3d::Math::Clamp(coef[2] / curve.fScale, minValue, maxValue));
            coefDst->coef[1] = nn::g3d::Math::Round<int8_t>(nn::g3d::Math::Clamp(coef[1] / curve.fScale, minValue, maxValue));
            coefDst->coef[0] = nn::g3d::Math::Round<int8_t>(nn::g3d::Math::Clamp((coef[0] - curve.fOffset) / curve.fScale, minValue, maxValue));
            pDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(Coef8));
        }
        else
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_UNSUPPORTED, "Identifier_UnsupportedQuantizeType");
        }
    }
}

void BinAnimCurve::QuantizeLinear(std::shared_ptr<Context> pCtx, nn::g3d::ResAnimCurveData& curve)
{
    curve.flag |= nn::g3d::ResAnimCurve::CurveType_Linear;

    float coef[2];
    curve.fScale = m_pElem->scale.value;
    curve.fOffset = m_pElem->offset.value;
    curve.fDelta = m_FVSPairArray.back().f[1] - m_FVSPairArray.front().f[1];
    if (m_IsDegreeValue)
    {
        curve.fDelta = nn::util::DegreeToRadian(curve.fDelta);
    }

    struct Coef32 { float coef[2]; };
    struct Coef16 { int16_t coef[2]; };
    struct Coef8 { int8_t coef[2]; };

    void * pDst = GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Key].offset);
    for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter)
    {
        float v0 = iter->linear.value0;
        float v1 = iter->linear.value1;
        if (m_IsDegreeValue)
        {
            v0 = nn::util::DegreeToRadian(v0);
            v1 = nn::util::DegreeToRadian(v1);
        }
        float v = v1 - v0;
        coef[1] = v;
        coef[0] = v0;


        if (m_pElem->key_type.value <= nw::g3d::tool::g3dif::elem_anim_curve::key32)
        {
            Coef32* coefDst = static_cast<Coef32*>(pDst);
            coefDst->coef[1] = coef[1];
            coefDst->coef[0] = coef[0];
            pDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(Coef32));
        }
        else if (m_pElem->key_type.value == nw::g3d::tool::g3dif::elem_anim_curve::key16)
        {
            Coef16* coefDst = static_cast<Coef16*>(pDst);
            coefDst->coef[1] = nn::g3d::Math::Round<int16_t>(coef[1] / curve.fScale);
            coefDst->coef[0] = nn::g3d::Math::Round<int16_t>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(Coef16));
        }
        else if (m_pElem->key_type.value == nw::g3d::tool::g3dif::elem_anim_curve::key8)
        {
            Coef8* coefDst = static_cast<Coef8*>(pDst);
            coefDst->coef[1] = nn::g3d::Math::Round<int8_t>(coef[1] / curve.fScale);
            coefDst->coef[0] = nn::g3d::Math::Round<int8_t>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(Coef8));
        }
        else
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_UNSUPPORTED, "Identifier_UnsupportedQuantizeType");
        }
    }
}

void BinAnimCurve::QuantizeStepFloat(std::shared_ptr<Context> pCtx, nn::g3d::ResAnimCurveData& curve)
{
    // リニア扱い。
    curve.flag |= nn::g3d::ResAnimCurve::CurveType_Linear;

    float coef[2];
    curve.fScale = m_pElem->scale.value;
    curve.fOffset = m_pElem->offset.value;
    curve.fDelta = m_FVSPairArray.back().f[1] - m_FVSPairArray.front().f[1];
    if (m_IsDegreeValue)
    {
        curve.fDelta = nn::util::DegreeToRadian(curve.fDelta);
    }

    struct Coef32 { float coef[2]; };
    struct Coef16 { int16_t coef[2]; };
    struct Coef8 { int8_t coef[2]; };

    void * pDst = GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Key].offset);
    for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter)
    {
        coef[1] = 0.0f;
        coef[0] = iter->step.value0;
        if (m_IsDegreeValue)
        {
            coef[0] = nn::util::DegreeToRadian(coef[0]);
        }

        if (m_pElem->key_type.value <= nw::g3d::tool::g3dif::elem_anim_curve::key32)
        {
            Coef32* coefDst = static_cast<Coef32*>(pDst);
            coefDst->coef[1] = coef[1];
            coefDst->coef[0] = coef[0];
            pDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(Coef32));
        }
        else if (m_pElem->key_type.value == nw::g3d::tool::g3dif::elem_anim_curve::key16)
        {
            Coef16* coefDst = static_cast<Coef16*>(pDst);
            coefDst->coef[1] = nn::g3d::Math::Round<int16_t>(coef[1] / curve.fScale);
            coefDst->coef[0] = nn::g3d::Math::Round<int16_t>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(Coef16));
        }
        else if (m_pElem->key_type.value == nw::g3d::tool::g3dif::elem_anim_curve::key8)
        {
            Coef8* coefDst = static_cast<Coef8*>(pDst);
            coefDst->coef[1] = nn::g3d::Math::Round<int8_t>(coef[1] / curve.fScale);
            coefDst->coef[0] = nn::g3d::Math::Round<int8_t>((coef[0] - curve.fOffset) / curve.fScale);
            pDst = nw::g3d::tool::util::AddOffset(pDst, sizeof(Coef8));
        }
        else
        {
            THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_UNSUPPORTED, "Identifier_UnsupportedQuantizeType");
        }
    }
}

void BinAnimCurve::QuantizeStepInt(std::shared_ptr<Context> pCtx, nn::g3d::ResAnimCurveData& curve)
{
    curve.flag |= nn::g3d::ResAnimCurve::CurveType_StepInt;
    curve.iScale = 0;
    curve.iOffset = static_cast<int>(m_pElem->offset.value);
    curve.iDelta = static_cast<int>(m_FVSPairArray.back().f[1] - m_FVSPairArray.front().f[1]);

    if (m_pElem->key_type.value <= nw::g3d::tool::g3dif::elem_anim_curve::key32)
    {
        int32_t* pDst = GetPtr< int32_t >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Key].offset);
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst)
        {
            *pDst = nn::g3d::Math::Round<int32_t>(iter->step.value0);
        }
    }
    else if (m_pElem->key_type.value == nw::g3d::tool::g3dif::elem_anim_curve::key16)
    {
        int16_t* pDst = GetPtr< int16_t >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Key].offset);
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst)
        {
            *pDst = nn::g3d::Math::Round<int16_t>(iter->step.value0 - curve.iOffset);
        }
    }
    else if (m_pElem->key_type.value == nw::g3d::tool::g3dif::elem_anim_curve::key8)
    {
        int8_t* pDst = GetPtr< int8_t >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Key].offset);
        for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++pDst)
        {
            *pDst = nn::g3d::Math::Round<int8_t>(iter->step.value0 - curve.iOffset);
        }
    }
    else
    {
        THROW_TRANSLATED_BINARY_BLOCK_ERROR(ERRCODE_UNSUPPORTED, "Identifier_UnsupportedQuantizeType");
    }
}

void BinAnimCurve::QuantizeStepBool(std::shared_ptr<Context> pCtx, nn::g3d::ResAnimCurveData& curve)
{
    curve.flag |= nn::g3d::ResAnimCurve::CurveType_StepBool;

    uint32_t* pDst = GetPtr< uint32_t >(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Key].offset);
    size_t byteSize = nw::g3d::tool::util::Align(m_FVSPairArray.size(), ALIGNMENT_FLAG) >> 3;
    memset(pDst, 0, byteSize);
    int index = 0;
    for (auto iter = m_FVSPairArray.cbegin(); iter != m_FVSPairArray.cend(); ++iter, ++index)
    {
        if (iter->step.value0 != 0.0f)
        {
            pDst[index >> 5] |= 0x1 << (index & 0x1F);
        }
    }
    curve.iScale = 1;
    curve.iOffset = 0;
    curve.iDelta = static_cast<int>(m_FVSPairArray.back().f[1] - m_FVSPairArray.front().f[1]);
}

}
}
