﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/g3d_ResSkeleton.h>

#include <g3dif/Skeleton.h>
#include <bindefs.h>
#include <BinaryBlock.h>
#include <BinDictionary.h>
#include <BinUserData.h>

namespace nn {
namespace g3dTool {


//! @brief ボーンデータのバイナリ化を行うクラスです。
class BinBone : public BinaryBlock
{
public:
    BinBone()
        : BinaryBlock()
        , m_pElem(nullptr)
        , m_DicUserData()
        , m_UserDataArray()
        , m_Index(-1)
        , m_RotateMode(nw::g3d::tool::g3dif::elem_skeleton_info::euler_xyz)
        , m_MirroringState(nn::g3d::ResBone::MirroringState_CenterPreRotate)
    {
    }

    void Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_bone& elem);
    virtual void CalculateSize();
    virtual void CalculateOffset( std::shared_ptr<Context> pCtx );
    virtual void Convert( std::shared_ptr<Context> pCtx );

    virtual void DumpLog() const
    {
        PRINT_SYSTEM_LOG("Bone : %hs", m_pElem->name.value.c_str());
    }

    void SetIndex(int index) { m_Index = index; }
    void SetRotateMode(nw::g3d::tool::g3dif::elem_skeleton_info::enum_rotate_mode rotateMode) { m_RotateMode = rotateMode; }
    void SetMirroringState(nn::g3d::ResBone::MirroringState mirroringState) { m_MirroringState = mirroringState; }
    nn::g3d::ResBone::MirroringState GetMirroringState() { return m_MirroringState; }

private:
    Bit32 ConvertSRT(nn::g3d::ResBoneData &bone);

private:
    const nw::g3d::tool::g3dif::elem_bone* m_pElem;

    BinDictionary m_DicUserData;
    std::vector<BinUserData> m_UserDataArray;

    int m_Index;
    nw::g3d::tool::g3dif::elem_skeleton_info::enum_rotate_mode m_RotateMode;
    nn::g3d::ResBone::MirroringState m_MirroringState;

    enum
    {
        ChunkType_UserDataData,
        ChunkType_Count
    };

    Chunk m_Chunk[ChunkType_Count];
};

//--------------------------------------------------------------------------------------------------

//! @brief スケルトンデータのバイナリ化を行うクラスです。
class BinSkeleton : public BinaryBlock
{
public:
    BinSkeleton()
        : BinaryBlock()
        , m_pElem(nullptr)
        , m_DicBone()
        , m_BoneArray()
        , m_NumSmoothBone(0)
        , m_NumRigidBone(0)
        , m_IsMirroringEnabled(false)
        , m_MirroringMode(nn::g3d::ResSkeleton::MirroringMode_X)
        , m_MirroringBoneTable()
    {
    }

    void Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_skeleton& elem);
    virtual void CalculateSize();
    virtual void CalculateOffset( std::shared_ptr<Context> pCtx );
    virtual void Convert( std::shared_ptr<Context> pCtx );

    virtual void DumpLog() const
    {
        PRINT_SYSTEM_LOG("Skeleton");
    }

    const std::vector<nw::g3d::tool::g3dif::elem_skeleton::MatrixPalette>& GetMatrixPalette()
    {
        return m_pElem->matrixPalette;
    }

    const nw::g3d::tool::util::Mtx34_t& GetInvMatrix(int boneIndex)
    {
        return m_pElem->bone_array[boneIndex].inv_model_matrix.Get();
    }

private:
    void CalculateInvMatrix(nn::util::Matrix4x3f* pOut, const nw::g3d::tool::g3dif::elem_bone& elemBone);
    void BuildMirroringInformation();
    void ConvertBone(nn::g3d::ResSkeletonData &skeleton, std::shared_ptr<Context> pCtx );

    const nw::g3d::tool::g3dif::elem_skeleton* m_pElem;

    BinDictionary m_DicBone;
    std::vector<BinBone> m_BoneArray;
    int m_NumSmoothBone;
    int m_NumRigidBone;

    bool m_IsMirroringEnabled;
    nn::g3d::ResSkeleton::MirroringMode m_MirroringMode;
    std::vector<int16_t> m_MirroringBoneTable;

    enum ChunkType
    {
        ChunkType_Skelton,
        ChunkType_Bone,
        ChunkType_MtxBoneTbl,
        ChunkType_InvModelMtx,
        ChunkType_MirroringBoneTbl,
        ChunkType_Count
    };

    Chunk m_Chunk[ChunkType_Count];
};

}
}
