﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d/g3d_ResMaterialAnim.h>
#include <g3dif/ShaderParamAnim.h>
#include <bindefs.h>
#include <BinaryBlock.h>
#include <BinDictionary.h>
#include <BinAnimCurve.h>
#include <BinUserData.h>


namespace nn {
namespace g3dTool {

//! @brief マテリアルごとのアニメーションデータのバイナリ化を行うクラスです。
class BinShaderParamMatAnim : public BinaryBlock
{
public:
    BinShaderParamMatAnim()
        : BinaryBlock()
        , m_pElem(nullptr)
        , m_CurveArray()
        , m_BeginCurve(0)
        , m_BeginParam(0)
        , m_NumConstant(0)
    {
        for( int chunkIndex = 0; chunkIndex < ChunkType_Count; chunkIndex++ )
        {
            m_Chunk[ chunkIndex ].size		= 0;
            m_Chunk[ chunkIndex ].alignment	= 0;
            m_Chunk[ chunkIndex ].offset	= -1;
        }
    }

    void Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_shader_param_mat_anim& elem);
    virtual void CalculateSize();
    virtual void CalculateOffset( std::shared_ptr<Context> pCtx );
    virtual void Convert( std::shared_ptr<Context> pCtx );

    virtual void DumpLog() const
    {
        PRINT_SYSTEM_LOG("ShaderParamMatAnim : %hs", m_pElem->mat_name.value.c_str());
    }

    int GetCurveCount() const { return static_cast<int>(m_CurveArray.size()); }
    int GetParamCount() const
    {
        int paramCount = 0;
        for (auto iter = m_pElem->param_anim_array.cbegin(); iter != m_pElem->param_anim_array.cend(); ++iter)
        {
            paramCount += static_cast<int>(iter->param_anim_target_array.size());
        }
        return paramCount;
    }

    void SetBeginCurve(int beginCurve) { this->m_BeginCurve = beginCurve; }
    void SetBeginParam(int beginTexture) { this->m_BeginParam = beginTexture; }

private:
    const nw::g3d::tool::g3dif::elem_shader_param_mat_anim* m_pElem;

    std::vector<BinAnimCurve> m_CurveArray;
    int m_BeginCurve;
    int m_BeginParam;
    int m_NumConstant;

    enum
    {
        PARAM_INFO,
        CONSTANT,
        ChunkType_Curve,
        ChunkType_Count
    };

    Chunk m_Chunk[ChunkType_Count];
};

//! @brief シェーダパラメータアニメーションデータのバイナリ化を行うクラスです。
class BinShaderParamAnim : public BinaryBlock
{
public:
    BinShaderParamAnim()
        : BinaryBlock()
        , m_pElem(nullptr)
        , m_ShaderParamMatAnimArray()
        , m_ParamCount(0)
        , m_CurveCount(0)
        , m_DicUserData()
        , m_UserDataArray()
    {
    }

    void Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_shader_param_anim& elem);
    virtual void CalculateSize();
    virtual void CalculateOffset( std::shared_ptr<Context> pCtx );
    virtual void Convert( std::shared_ptr<Context> pCtx );
    virtual void Adjust( std::shared_ptr<Context> pCtx );

    virtual void DumpLog() const
    {
        PRINT_SYSTEM_LOG("ShaderParamAnim : %hs", m_pElem->name.c_str());
    }

private:
    const nw::g3d::tool::g3dif::elem_shader_param_anim* m_pElem;

    std::vector<BinShaderParamMatAnim> m_ShaderParamMatAnimArray;
    int m_ParamCount;	// 配下の ResPerMaterialAnimData の総数です。
    int m_CurveCount;

    BinDictionary m_DicUserData;
    std::vector<BinUserData> m_UserDataArray;

    enum ChunkType
    {
        //SHADER_PARAM_ANIM,	// 親から貰います。
        ChunkType_BindIndex,
        ChunkType_ShaderParamMaterialAnim,
        ChunkType_UserDataData,
        ChunkType_Count,
    };

    Chunk m_Chunk[ChunkType_Count];
};

}
}
