﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <set>
#include <bindefs.h>

#if defined( near ) && defined( far )	// windows.h で両方 define される
#undef near
#undef far
#include <nn/g3d/g3d_ResFile.h>	// nn:util:math で near, far を使用している
#define near
#define far
#else
#include <nn/g3d/g3d_ResFile.h>
#endif

#include <BinaryBlock.h>
#include <BinDictionary.h>
#include <BinModel.h>
#include <BinSkeletalAnim.h>
#include <BinTexPatternAnim.h>
#include <BinShaderParamAnim.h>
#include <BinMaterialVisibilityAnim.h>
#include <BinVisibilityAnim.h>
#include <BinSceneAnim.h>
#include <BinShapeAnim.h>
#include <BinMaterialAnim.h>

namespace nn { namespace g3dTool {


class BinExternalFile : public BinaryBlock
{
public:
    struct ExternalFileElem
    {
        std::shared_ptr<void> data;
        std::string name;
        size_t size;
        size_t alignment;
    };

    BinExternalFile()
        : BinaryBlock()
        , m_pElem(nullptr)
    {
    }

    void Build( std::shared_ptr<Context> pCtx, const ExternalFileElem& elem );
    virtual void CalculateSize();
    virtual void Convert( std::shared_ptr<Context> pCtx );

    virtual void DumpLog() const
    {
        PRINT_SYSTEM_LOG("ExternalFile : %hs", m_pElem->name.c_str());
    }

private:
    const ExternalFileElem* m_pElem;

    enum
    {
        EXTERNAL_FILE,
        ChunkType_Count
    };

    Chunk m_Chunk[ChunkType_Count];
};

class BinFile : public BinaryBlock
{
public:
    BinFile()
        : BinaryBlock()
        , m_fmdNameArray()
        , m_ftxNameArray()
        , m_fskNameArray()
        , m_ftpNameArray()
        , m_fspNameArray()
        , m_fclNameArray()
        , m_ftsNameArray()
        , m_fvbNameArray()
        , m_fvmNameArray()
        , m_fsnNameArray()
        , m_fshNameArray()
        , m_dataNameArray()
        , m_fmaNameArray()
        , m_fmdArray()
        , m_fskArray()
        , m_ftpArray()
        , m_fspArray()
        , m_fclArray()
        , m_ftsArray()
        , m_fvmArray()
        , m_fvbArray()
        , m_fsnArray()
        , m_fshArray()
        , m_fmaArray()
        , m_externalFileArray()
        , m_DicModel()
        , m_DicSkeletalAnim()
        , m_DicMaterialAnim()
        , m_DicBoneVisibilityAnim()
        , m_DicSceneAnim()
        , m_DicShapeAnim()
        , m_DicExternalFile()
        , m_ModelArray()
        , m_SkeletalAnimArray()
        , m_TexPatternAnimArray()
        , m_ShaderParamAnimArray()
        , m_ColorAnimArray()
        , m_TexSrtAnimArray()
        , m_BoneVisibilityAnimArray()
        , m_SceneAnimArray()
        , m_ShapeAnimArray()
        , m_ExternalFileArray()
        , m_MaterialAnimArray()
        , m_Name()
        , m_Alignment(0)
        , m_TextureAlignment(0)
        , m_ExternalFileAlignment(0)
    {
        for( int i = 0; i < ChunkType_Count; i++ )
        {
            m_Chunk[ i ].size		= 0;
            m_Chunk[ i ].alignment	= 0;
            m_Chunk[ i ].offset		= -1;
        }
    }

    void Add(std::shared_ptr<nw::g3d::tool::g3dif::elem_model> fmd)
    {
        if (m_fmdNameArray.find(fmd->name) != m_fmdNameArray.cend())
        {
            THROW_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "It is impossible to convert same name models: %hs.fmd(a|b)", fmd->name.c_str());
        }
        m_fmdNameArray.insert(fmd->name);
        m_fmdArray.push_back(fmd);
    }
    void Add(std::shared_ptr<const nw::g3d::tool::g3dif::elem_skeletal_anim> fsk)
    {
        if (m_fskNameArray.find(fsk->name) != m_fskNameArray.cend())
        {
            THROW_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "It is impossible to convert same name skeletal_anims: %hs.fsk(a|b)", fsk->name.c_str());
        }
        m_fskNameArray.insert(fsk->name);
        m_fskArray.push_back(fsk);
    }
    void Add(std::shared_ptr<const nw::g3d::tool::g3dif::elem_tex_pattern_anim> ftp)
    {
        if (m_ftpNameArray.find(ftp->name) != m_ftpNameArray.cend())
        {
            THROW_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "It is impossible to convert same name tex_pattern_anims: %hs.ftp(a|b)", ftp->name.c_str());
        }
        m_ftpNameArray.insert(ftp->name);
        m_ftpArray.push_back(ftp);
    }
    void AddFsp(std::shared_ptr<const nw::g3d::tool::g3dif::elem_shader_param_anim> fsp)
    {
        if (m_fspNameArray.find(fsp->name) != m_fspNameArray.cend())
        {
            THROW_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "It is impossible to convert same name shader_param_anims: %hs.fsp(a|b)", fsp->name.c_str());
        }
        m_fspNameArray.insert(fsp->name);
        m_fspArray.push_back(fsp);
    }
    void AddFcl(std::shared_ptr<const nw::g3d::tool::g3dif::elem_shader_param_anim> fcl)
    {
        if (m_fclNameArray.find(fcl->name) != m_fclNameArray.cend())
        {
            THROW_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "It is impossible to convert same name color_anims: %hs.fcl(a|b)", fcl->name.c_str());
        }
        m_fclNameArray.insert(fcl->name);
        m_fclArray.push_back(fcl);
    }
    void AddFts(std::shared_ptr<const nw::g3d::tool::g3dif::elem_shader_param_anim> fts)
    {
        if (m_ftsNameArray.find(fts->name) != m_ftsNameArray.cend())
        {
            THROW_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "It is impossible to convert same name texture_srt_anims: %hs.fts(a|b)", fts->name.c_str());
        }
        m_ftsNameArray.insert(fts->name);
        m_ftsArray.push_back(fts);
    }
    void Add(std::shared_ptr<const nw::g3d::tool::g3dif::elem_mat_visibility_anim> fvm)
    {
        if (m_fvmNameArray.find(fvm->name) != m_fvmNameArray.cend())
        {
            THROW_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "It is impossible to convert same name mat_visibility_anims: %hs.fvm(a|b)", fvm->name.c_str());
        }
        m_fvmNameArray.insert(fvm->name);
        m_fvmArray.push_back(fvm);
    }
    void Add(std::shared_ptr<const nw::g3d::tool::g3dif::elem_bone_visibility_anim> fvb)
    {
        if (m_fvbNameArray.find(fvb->name) != m_fvbNameArray.cend())
        {
            THROW_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "It is impossible to convert same name bone_visibility_anims: %hs.fvb(a|b)", fvb->name.c_str());
        }
        m_fvbNameArray.insert(fvb->name);
        m_fvbArray.push_back(fvb);
    }
    void Add(std::shared_ptr<const nw::g3d::tool::g3dif::elem_scene_anim> fsn)
    {
        if (m_fsnNameArray.find(fsn->name) != m_fsnNameArray.cend())
        {
            THROW_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "It is impossible to convert same name scene_anims: %hs.fsn(a|b)", fsn->name.c_str());
        }
        m_fsnNameArray.insert(fsn->name);
        m_fsnArray.push_back(fsn);
    }
    void Add(std::shared_ptr<const nw::g3d::tool::g3dif::elem_shape_anim> fsh)
    {
        if (m_fshNameArray.find(fsh->name) != m_fshNameArray.cend())
        {
            THROW_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "It is impossible to convert same name shape_anims: %hs.fsh(a|b)", fsh->name.c_str());
        }
        m_fshNameArray.insert(fsh->name);
        m_fshArray.push_back(fsh);
    }
    void Add(std::shared_ptr<const nw::g3d::tool::g3dif::elem_nn_material_anim> fma)
    {
        if (m_fmaNameArray.find(fma->name) != m_fmaNameArray.cend())
        {
            THROW_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "It is impossible to convert same name material_anims: %hs.fsh(a|b)", fma->name.c_str());
        }
        m_fmaNameArray.insert(fma->name);
        m_fmaArray.push_back(fma);
    }

    void Add(BinExternalFile::ExternalFileElem& elem)
    {
        if (m_dataNameArray.find(elem.name) != m_dataNameArray.cend())
        {
            THROW_ERROR(ERRCODE_FILE_NAME_OVERLAPPED, "It is impossible to convert same name external_file: %hs", elem.name.c_str());
        }
        m_dataNameArray.insert(elem.name);
        m_externalFileArray.push_back(elem);
    }

    void Build( std::shared_ptr<Context> pCtx );
    virtual void CalculateSize();
    virtual void CalculateOffset( std::shared_ptr<Context> pCtx );
    virtual void Convert( std::shared_ptr<Context> pCtx );
    virtual void Adjust( std::shared_ptr<Context> pCtx );

    virtual void DumpLog() const
    {
        PRINT_SYSTEM_LOG("File : %hs", m_Name.c_str());
    }

    void SetName(const std::string& name)
    {
        m_Name = name;
    }

    void SetMaxAlignment(int alignment)
    {
        m_Alignment = alignment;
    }

    size_t GetMaxAlignment() const
    {
        return m_Alignment;
    }

    int GetTextureAlignment() const
    {
        return m_TextureAlignment;
    }

    int GetExternalFileAlignment() const
    {
        return m_ExternalFileAlignment;
    }

    const std::string& GetName() const
    {
        return m_Name;
    }

private:
    std::set<std::string> m_fmdNameArray;
    std::set<std::string> m_ftxNameArray;
    std::set<std::string> m_fskNameArray;
    std::set<std::string> m_ftpNameArray;
    std::set<std::string> m_fspNameArray;
    std::set<std::string> m_fclNameArray;
    std::set<std::string> m_ftsNameArray;
    std::set<std::string> m_fvbNameArray;
    std::set<std::string> m_fvmNameArray;
    std::set<std::string> m_fsnNameArray;
    std::set<std::string> m_fshNameArray;
    std::set<std::string> m_dataNameArray;
    std::set<std::string> m_fmaNameArray;

    std::vector<std::shared_ptr<nw::g3d::tool::g3dif::elem_model>> m_fmdArray;
    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_skeletal_anim>> m_fskArray;
    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_tex_pattern_anim>> m_ftpArray;
    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_shader_param_anim>> m_fspArray;
    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_shader_param_anim>> m_fclArray;
    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_shader_param_anim>> m_ftsArray;
    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_mat_visibility_anim>> m_fvmArray;
    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_bone_visibility_anim>> m_fvbArray;
    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_scene_anim>> m_fsnArray;
    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_shape_anim>> m_fshArray;
    std::vector<std::shared_ptr<const nw::g3d::tool::g3dif::elem_nn_material_anim>> m_fmaArray;

    std::vector<BinExternalFile::ExternalFileElem> m_externalFileArray;

    BinDictionary m_DicModel;
    BinDictionary m_DicSkeletalAnim;
    BinDictionary m_DicMaterialAnim;		// 従来の TexPat, ShaderParam, ColorAnim, TexSrt, MatVis をまとめたもの。
    BinDictionary m_DicBoneVisibilityAnim;
    BinDictionary m_DicSceneAnim;
    BinDictionary m_DicShapeAnim;
    BinDictionary m_DicExternalFile;

    std::vector<BinModel> m_ModelArray;

    std::vector<BinSkeletalAnim> m_SkeletalAnimArray;
    std::vector<BinTexPatternAnim> m_TexPatternAnimArray;
    std::vector<BinShaderParamAnim> m_ShaderParamAnimArray;
    std::vector<BinShaderParamAnim> m_ColorAnimArray;
    std::vector<BinShaderParamAnim> m_TexSrtAnimArray;
    std::vector<BinMatVisibilityAnim> m_MatVisibilityAnimArray;
    std::vector<BinBoneVisibilityAnim> m_BoneVisibilityAnimArray;
    std::vector<BinSceneAnim> m_SceneAnimArray;
    std::vector<BinShapeAnim> m_ShapeAnimArray;

    std::vector<BinExternalFile> m_ExternalFileArray;
    std::vector<BinMaterialAnim> m_MaterialAnimArray;

    std::string m_Name;
    int m_Alignment;
    int m_TextureAlignment;
    int m_ExternalFileAlignment;

    enum ChunkType
    {
        ChunkType_File,
        ChunkType_ModelArray,		//!< ResModelData 配列のチャンクです。
        ChunkType_SkeltalAnimArray,	//!< ResSkeltalAnimData 配列のチャンクです。
        ChunkType_MaterialAnimArray,	//!< ResMaterialAnimData 配列のチャンクです。
        ChunkType_BoneVisibilityAnimArray,	//!< ResBoneVisibilityAnimData 配列のチャンクです。
        ChunkType_SceneAnimArray,	//!< ResSceneAnimData 配列のチャンクです。
        ChunkType_ShapeAnimArray,	//!< ResShapeAnimData 配列のチャンクです。
        ChunkType_GfxMemoryPoolInfoData,	//!< gfx::MemoryPoolInfoData
        ChunkType_ExternalFile,
        ChunkType_Count
    };

    Chunk m_Chunk[ChunkType_Count];
};



} // namespace tool
} // namespace g3d
