﻿namespace G3dCore.ViewModels
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using G3dCore.Entities;
    using G3dCore.Messaging;
    using G3dCore.Storages;
    using G3dCore.Utilities;
    using G3dCore.Windows.Actions;
    using Opal.App;
    using Opal.Operations;
    using Opal.Services;
    using Opal.ViewModels;

    /// <summary>
    /// 中間ファイルビューモデルクラスです。
    /// </summary>
    public class G3dFileViewModel : ObservableValueViewModel
    {
        private readonly G3dFile file;
        private readonly OperationManager operationManager = new OperationManager();
        private string filePath = string.Empty;
        private IDocument targetDocument = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="file">設定する中間ファイルデータです。</param>
        public G3dFileViewModel(G3dFile file)
        {
            Debug.Assert(file != null);

            this.file = file;
            this.filePath = file.FilePath;
        }

        /// <summary>
        /// 名前を取得します。
        /// </summary>
        public string Name
        {
            get
            {
                return Path.GetFileName(this.filePath);
            }
        }

        /// <summary>
        /// ファイル名を取得設定します。
        /// </summary>
        public string FilePath
        {
            get
            {
                return this.filePath;
            }

            set
            {
                this.SetProperty(ref this.filePath, value, () => this.file.FilePath = this.filePath);
            }
        }

        /// <summary>
        /// G3d 中間ファイルを取得します。
        /// </summary>
        public G3dFile File
        {
            get
            {
                return this.file;
            }
        }

        /// <summary>
        /// ドキュメントを開きます。
        /// </summary>
        public void OpenDocument()
        {
            if (this.targetDocument != null)
            {
                if (!AppManager.HasDocument(this.targetDocument))
                {
                    this.targetDocument = null;
                }
                else
                {
                    AppManager.SetActiveDocument(this.targetDocument);
                    return;
                }
            }

            switch (this.file.Kind)
            {
                case G3dKind.ShaderConfig:
                    this.targetDocument = AppManager.CreateDocument("G3d", "ShaderConfig");
                    break;
                default:
                    // 不正な形式
                    break;
            }

            Debug.Assert(this.targetDocument != null);

            this.targetDocument.Title = this.Name;
            this.targetDocument.Setup(this);

            AppManager.AddDocument(this.targetDocument);
            AppManager.SetActiveDocument(this.targetDocument);
        }

        /// <summary>
        /// ドキュメントを保存します。
        /// </summary>
        /// <param name="fileName">保存するファイル名です。</param>
        public void SaveDocument(string fileName)
        {
            G3dIfUtility.WriteFile(this.File, fileName);
        }

        /// <summary>
        /// ドキュメントを閉じます。
        /// </summary>
        public void CloseDocument()
        {
            if (this.targetDocument != null)
            {
                AppManager.RemoveDocument(this.targetDocument);
                this.targetDocument = null;
            }

            var weakStorage = AppManager.GetStorage<G3dStorage>();
            Debug.Assert(weakStorage != null);

            G3dStorage storage = null;
            if (weakStorage.TryGetTarget(out storage))
            {
                storage.RemoveFileViewModel(this);
            }
        }

        /// <summary>
        /// 未保存の場合は確認をしてからドキュメントを閉じます。
        /// </summary>
        /// <returns>処理がキャンセルされた場合には false を返します。</returns>
        public bool ConfirmAndCloseDocument()
        {
            if (!this.file.IsDirty)
            {
                this.CloseDocument();
                return true;
            }

            var confirmDialogAction = new ConfirmDialogAction();
            var manualTrigger = new ManualTrigger();
            manualTrigger.Actions.Add(confirmDialogAction);

            var input = new ConfirmDialogAction.InputArg()
            {
                FileName = System.IO.Path.GetFileName(this.filePath)
            };

            bool canceled = false;
            var eventArgs = new MessageEventArgs(
                new Message(input),
                m =>
                {
                    var output = (ConfirmDialogAction.OutputArg)m.Response;
                    if (output.Result == true)
                    {
                        this.SaveDocument(this.filePath);
                        this.CloseDocument();
                    }
                    else if (output.Result == false)
                    {
                        this.CloseDocument();
                    }
                    else
                    {
                        canceled = true;
                    }
                });

            manualTrigger.Invoke(eventArgs);
            return !canceled;
        }

#region Operation

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <param name="operation">実行するオペレーションです。</param>
        public void ExecuteOperation(Operation operation)
        {
            Debug.Assert(operation != null);
            this.operationManager.Execute(operation);

#if false
            OperationExecutedArg e = new OperationExecutedArg();
            e.Sender = this;
            e.Operation = operation;

            if (this.executeOperationHookPlugins != null)
            {
                foreach (var plugin in this.executeOperationHookPlugins)
                {
                    plugin.PreExecuteOperation(e);
                }
            }

            this.operationManager.Execute(operation);

            if (this.executeOperationHookPlugins != null)
            {
                foreach (var plugin in this.executeOperationHookPlugins)
                {
                    plugin.PostExecuteOperation(e);
                }
            }
#endif
        }

        /// <summary>
        /// Undoを実行します。
        /// </summary>
        public void Undo()
        {
            this.operationManager.Undo();

#if false
            UndoExecutedArg e = new UndoExecutedArg();
            e.Sender = this;
            e.Operation = this.operationManager.UndoStack.Last();

            if (this.undoHookPlugins != null)
            {
                foreach (var plugin in this.undoHookPlugins)
                {
                    plugin.PreUndo(e);
                }
            }

            this.operationManager.Undo();

            if (this.undoHookPlugins != null)
            {
                foreach (var plugin in this.undoHookPlugins)
                {
                    plugin.PostUndo(e);
                }
            }
#endif
        }

        /// <summary>
        /// Undo可能かどうかを返します。
        /// </summary>
        /// <returns>結果を返します。</returns>
        public bool CanUndo()
        {
            return this.operationManager.CanUndo();
        }

        /// <summary>
        /// Redoを実行します。
        /// </summary>
        public void Redo()
        {
            this.operationManager.Redo();

#if false
            RedoExecutedArg e = new RedoExecutedArg();
            e.Sender = this;
            e.Operation = this.operationManager.RedoStack.Last();

            if (this.undoHookPlugins != null)
            {
                foreach (var plugin in this.redoHookPlugins)
                {
                    plugin.PreRedo(e);
                }
            }

            this.operationManager.Redo();

            if (this.undoHookPlugins != null)
            {
                foreach (var plugin in this.redoHookPlugins)
                {
                    plugin.PostRedo(e);
                }
            }
#endif
        }

        /// <summary>
        /// Redo可能かどうかを返します。
        /// </summary>
        /// <returns>結果を返します。</returns>
        public bool CanRedo()
        {
            return this.operationManager.CanRedo();
        }

#endregion
    }
}
