﻿namespace G3dCore.Utilities
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Runtime.Serialization;
    using System.Text;
    using System.Threading.Tasks;
    using System.Xml;
    using System.Xml.Linq;
    using System.Xml.Serialization;
    using System.Xml.XPath;

    /// <summary>
    /// XML 操作用ユーティリティです。
    /// </summary>
    public class XmlUtility
    {
        /// <summary>
        /// T を XmlElement に変換します。
        /// </summary>
        /// <typeparam name="T">変換するオブジェクトの型です。</typeparam>
        /// <param name="obj">変換するオブジェクトです。</param>
        /// <param name="level">level のタブインデントを XmlElement 内部に設定します。</param>
        /// <returns>変換した XmlElement を返します。</returns>
        public static XmlElement ConvertObjectToXmlElement<T>(T obj, int level)
        {
            // インデントのレベルは1以上
            Debug.Assert(level > 0);

            // タワーに変換
            var root = new TTower<T>();
            var leaf = root;

            for (int i = 1; i < level; i++)
            {
                leaf.Tower = new TTower<T>();
                leaf = leaf.Tower;
            }

            leaf.Data = obj;

            // シリアライズ
            var stream = new MemoryStream();
            XmlWriterSettings settings = new XmlWriterSettings();
            using (XmlWriter writer = XmlWriter.Create(stream, settings))
            {
                var serializer = new XmlSerializer(typeof(TTower<T>));
                serializer.Serialize(writer, root);
            }

            stream.Seek(0, SeekOrigin.Begin);

            // 変換
            var doc = XDocument.Load(stream, LoadOptions.PreserveWhitespace);
            doc.Root.Name = typeof(XmlElementTower).Name;
            var element = doc.XPathSelectElement("//Data");
            element.Name = GetXmlElementName(typeof(T));

            // デシリアライズ
            XmlElementTower node;
            using (XmlReader reader = doc.CreateReader())
            {
                var deserializer = new XmlSerializer(typeof(XmlElementTower));
                node = deserializer.Deserialize(reader) as XmlElementTower;
            }

            // タワーから要素を取り出します。
            while (node.Tower != null)
            {
                node = node.Tower;
            }

            return node.Elements[0];
        }

        /// <summary>
        /// オブジェクト を XmlElement に変換します。
        /// </summary>
        /// <param name="obj">変換するオブジェクトです。</param>
        /// <returns>変換した XmlElement を返します。</returns>
        public static XmlElement ConvertObjectToXmlElement(object obj)
        {
            Type type = obj.GetType();

            // シリアライズ
            var stream = new MemoryStream();
            XmlWriterSettings settings = new XmlWriterSettings();
            using (XmlWriter writer = XmlWriter.Create(stream, settings))
            {
                var xnameSpace = new XmlSerializerNamespaces();
                xnameSpace.Add("", "");
                var serializer = new XmlSerializer(type);
                serializer.Serialize(writer, obj, xnameSpace);
            }

            stream.Seek(0, SeekOrigin.Begin);

            // 変換
            var doc = XDocument.Load(stream, LoadOptions.PreserveWhitespace);
            doc.Root.Name = GetXmlElementName(type);

            // デシリアライズ
            XmlElement element;
            using (XmlReader reader = doc.CreateReader())
            {
                var deserializer = new XmlSerializer(typeof(XmlElement));
                element = deserializer.Deserialize(reader) as XmlElement;
            }

            return element;
        }

        /// <summary>
        /// XmlElement を自動推測された型のオブジェクトに変換します。
        /// </summary>
        /// <param name="element">変換する XmlElement です。</param>
        /// <returns>変換したオブジェクトを返します。</returns>
        public static object ConvertXmlElementToObject(XmlElement element)
        {
            var type = GetTypeFromXmlElement(element);
            Debug.Assert(type != null);

            return ConvertXmlElementToObject(element, type);
        }

        /// <summary>
        /// XmlElement を T に変換します。
        /// </summary>
        /// <typeparam name="T">変換するオブジェクトの型です。</typeparam>
        /// <param name="element">変換する XmlElement です。</param>
        /// <returns>変換したオブジェクトを返します。</returns>
        public static T ContertXmlElementToObject<T>(XmlElement element) where T : class
        {
            return ConvertXmlElementToObject(element, typeof(T)) as T;
        }

        /// <summary>
        /// XmlElement を 特定の型のオブジェクト に変換します。
        /// </summary>
        /// <param name="element">変換する XmlElement です。</param>
        /// <param name="type">変換するオブジェクトの型です。</param>
        /// <param name="rootName">変換するオブジェクトのルート要素名です。</param>
        /// <returns>変換したオブジェクトを返します。</returns>
        public static object ConvertXmlElementToObject(XmlElement element, Type type)
        {
            // シリアライズ
            var doc = new XDocument();
            using (XmlWriter writer = doc.CreateWriter())
            {
                var serializer = new XmlSerializer(typeof(XmlElement));
                serializer.Serialize(writer, element);
            }

            // 変換
            doc.Root.Name = GetXmlElementName(type);

            // デシリアライズ
            object t;
            using (XmlReader reader = doc.CreateReader())
            {
                var deserializer = new XmlSerializer(type);
                t = deserializer.Deserialize(reader);
            }

            return t;
        }

        /// <summary>
        /// 特定の型のオブジェクトをシリアライズされた文字列に変換します。
        /// </summary>
        /// <typeparam name="T">シリアライズするオブジェクトの型を指定します。</typeparam>
        /// <param name="obj">シリアライズするオブジェクトです。</param>
        /// <returns>シリアライズ後の文字列を返します。</returns>
        public static string SerializeToString<T>(T obj)
            where T : new()
        {
            if (!IsSerializable(typeof(T)))
            {
                throw new InvalidOperationException("対象オブジェクトはシリアライズ可能な型である必要があります。");
            }

            XmlSerializer serializer = new XmlSerializer(obj.GetType());

            using (StringWriter writer = new StringWriter())
            {
                serializer.Serialize(writer, obj);

                return writer.ToString();
            }
        }

        /// <summary>
        /// XML文字列を特定の型のオブジェクトにデシリアライズします。
        /// </summary>
        /// <typeparam name="T">デシリアライズ後の型を指定します。</typeparam>
        /// <param name="xml">デシリアライズするXML文字列を渡します。</param>
        /// <returns>デシリアライズされたオブジェクトを返します。失敗した場合はnullを返します。</returns>
        public static T DeserializeFromString<T>(string xml)
            where T : class, new()
        {
            return DeserializeFromString(xml, typeof(T)) as T;
        }

        /// <summary>
        /// XML文字列を特定の型のオブジェクトにデシリアライズします。
        /// </summary>
        /// <param name="xml">デシリアライズするXML文字列を渡します。</param>
        /// <param name="outputType">デシリアライズ後の型を指定します。</param>
        /// <returns>デシリアライズされたオブジェクトを返します。失敗した場合はnullを返します。</returns>
        public static object DeserializeFromString(string xml, Type outputType)
        {
            if (!IsSerializable(outputType))
            {
                return null;
            }

            XmlSerializer serializer = new XmlSerializer(outputType);

            using (StringReader reader = new StringReader(xml))
            {
                try
                {
                    return serializer.Deserialize(reader);
                }
                catch (Exception)
                {
                    return null;
                }
            }
        }

        private static bool IsSerializable(Type type)
        {
            return type.IsSerializable || typeof(ISerializable).IsAssignableFrom(type);
        }

        private static Type GetTypeFromXmlElement(XmlElement element)
        {
            System.Reflection.Assembly[] assemblies = AppDomain.CurrentDomain.GetAssemblies();
            foreach (var assembly in assemblies)
            {
                foreach (Type type in assembly.GetTypes())
                {
                    if (type.IsSerializable && GetXmlElementName(type) == element.Name)
                    {
                        return type;
                    }
                }
            }

            return null;
        }

        private static string GetXmlElementName(Type type)
        {
            Debug.Assert(type.IsSerializable);

            string xmlRootElementName = type.Name;
            var attributes = type.GetCustomAttributesData();
            foreach (var attribute in attributes)
            {
                if (attribute.AttributeType == typeof(XmlRootAttribute) && attribute.ConstructorArguments.Count > 0)
                {
                    var arg = attribute.ConstructorArguments[0];
                    Debug.Assert(arg.ArgumentType == typeof(string));

                    xmlRootElementName = arg.Value as string;
                    break;
                }
            }

            return xmlRootElementName;
        }

        /// <summary>
        /// 変換用ヘルパクラスです。
        /// </summary>
        /// <typeparam name="T">変換する型です。</typeparam>
        public class TTower<T>
        {
            /// <summary>
            /// TTower です。
            /// </summary>
            [XmlElement]
            public TTower<T> Tower { get; set; }

            /// <summary>
            /// データです。
            /// </summary>
            public T Data { get; set; }
        }

        /// <summary>
        /// 変換用ヘルパクラスです。
        /// </summary>
        public class XmlElementTower
        {
            /// <summary>
            /// XmlElementTower です。
            /// </summary>
            [XmlElement]
            public XmlElementTower Tower { get; set; }

            /// <summary>
            /// エレメントです。
            /// </summary>
            [XmlAnyElement]
            public XmlElement[] Elements { get; set; }
        }
    }
}
