﻿namespace G3dCore.IO
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;

    /// <summary>
    /// パスに関するユーティリティクラスです。
    /// </summary>
    public static class PathUtility
    {
        /// <summary>
        /// basePath を基準とした filePath の相対パスを取得します。
        /// </summary>
        /// <param name="basePath">基準パスです。</param>
        /// <param name="filePath">ファイルパスです。</param>
        /// <returns>相対パスを返します。</returns>
        public static string MakeRelativePath(string basePath, string filePath)
        {
            if (!basePath.EndsWith("\\"))
            {
                basePath += "\\";
            }
            if (!filePath.EndsWith("\\"))
            {
                filePath += "\\";
            }

            string relativePath = string.Empty;

            try
            {
                Uri uriBase = new Uri(basePath);
                Uri uriPath = new Uri(filePath);
                relativePath = FormatPathSeparator(uriBase.MakeRelativeUri(uriPath).ToString(), false);
                if (relativePath.CompareTo("") == 0)
                {
                    relativePath = ".";
                }
            }
            catch
            {
            }

            return relativePath;
        }

        /// <summary>
        /// パスのセパレータを整形します。
        /// </summary>
        /// <param name="path">整形するパスです。</param>
        /// <param name="separateBySlash">スラッシュ区切りにする場合に true を指定します。</param>
        /// <returns>整形したパスを返します。</returns>
        public static string FormatPathSeparator(string path, bool separateBySlash)
        {
            string formattedPath;

            if (separateBySlash)
            {
                // "shader/src/test.glsl"（ファイル）、"shader/src"（ディレクトリ）形式にする。
                formattedPath = path.Replace('\\', '/');
                if (formattedPath.EndsWith("/"))
                {
                    formattedPath = formattedPath.Substring(0, formattedPath.Length - 1);
                }
            }
            else
            {
                // "shader\src\test.glsl"（ファイル）、"shader\src"（ディレクトリ）形式にする。
                formattedPath = path.Replace('/', '\\');
                if (formattedPath.EndsWith("\\"))
                {
                    formattedPath = formattedPath.Substring(0, formattedPath.Length - 1);
                }
            }

            return formattedPath;
        }

        /// <summary>
        /// 環境変数を展開した文字列を取得します。
        /// </summary>
        /// <param name="path">環境変数が含まれたパスです。</param>
        /// <returns>環境変数を展開した文字列を返します。</returns>
        public static string GetExpandEnvironmentVariable(string path)
        {
            string expandedPath = System.Environment.ExpandEnvironmentVariables(path);
            if (expandedPath.Contains('%'))
            {
                return path;
            }

            return FormatPathSeparator(expandedPath, false);
        }
    }
}
